/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.loader.reader.hdfs;

import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hugegraph.loader.constant.Constants;
import org.apache.hugegraph.loader.exception.LoadException;
import org.apache.hugegraph.loader.progress.FileItemProgress;
import org.apache.hugegraph.loader.progress.InputItemProgress;
import org.apache.hugegraph.loader.reader.Readable;
import org.apache.hugegraph.loader.reader.file.FileLineFetcher;
import org.apache.hugegraph.loader.reader.file.FileReader;
import org.apache.hugegraph.loader.reader.file.OrcFileLineFetcher;
import org.apache.hugegraph.loader.reader.file.ParquetFileLineFetcher;
import org.apache.hugegraph.loader.source.InputSource;
import org.apache.hugegraph.loader.source.file.Compression;
import org.apache.hugegraph.loader.source.file.DirFilter;
import org.apache.hugegraph.loader.source.file.FileFilter;
import org.apache.hugegraph.loader.source.hdfs.HDFSSource;
import org.apache.hugegraph.loader.source.hdfs.KerberosConfig;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public class HDFSFileReader
extends FileReader {
    private static final Logger LOG = Log.logger(HDFSFileReader.class);
    private final FileSystem hdfs;
    private final Configuration conf = this.loadConfiguration();
    private static boolean hasLogin = false;
    public static final ScheduledExecutorService RELOGIN_EXECUTOR = Executors.newSingleThreadScheduledExecutor();
    private static boolean isCheckKerberos = false;

    public HDFSFileReader(HDFSSource source) {
        super(source);
        try {
            this.enableKerberos(source);
            this.hdfs = this.getFileSystem(this.conf);
        }
        catch (IOException e) {
            throw new LoadException("Failed to create HDFS file system", e);
        }
    }

    public FileSystem getFileSystem(Configuration conf) throws IOException {
        return FileSystem.get((Configuration)conf);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void enableKerberos(HDFSSource source) throws IOException {
        KerberosConfig kerberosConfig = source.kerberosConfig();
        if (kerberosConfig == null || !kerberosConfig.enable()) return;
        System.setProperty("java.security.krb5.conf", kerberosConfig.krb5Conf());
        UserGroupInformation.setConfiguration((Configuration)this.conf);
        Class<HDFSFileReader> clazz = HDFSFileReader.class;
        synchronized (HDFSFileReader.class) {
            if (!hasLogin) {
                UserGroupInformation.loginUserFromKeytab((String)kerberosConfig.principal(), (String)kerberosConfig.keyTab());
                hasLogin = true;
            }
            // ** MonitorExit[var3_3] (shouldn't be in output)
            HDFSFileReader.cronCheckKerberos();
            return;
        }
    }

    private static void cronCheckKerberos() {
        if (!isCheckKerberos) {
            RELOGIN_EXECUTOR.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    try {
                        UserGroupInformation.getCurrentUser().checkTGTAndReloginFromKeytab();
                        LOG.info("Check Kerberos Tgt And Relogin From Keytab Finish.");
                    }
                    catch (IOException e) {
                        LOG.error("Check Kerberos Tgt And Relogin From Keytab Error", (Throwable)e);
                    }
                }
            }, 0L, 10L, TimeUnit.MINUTES);
            LOG.info("Start Check Keytab TGT And Relogin Job Success.");
            isCheckKerberos = true;
        }
    }

    public FileSystem fileSystem() {
        return this.hdfs;
    }

    @Override
    public HDFSSource source() {
        return (HDFSSource)super.source();
    }

    @Override
    public FileReader newFileReader(InputSource source, Readable readable) {
        HDFSFileReader reader = new HDFSFileReader((HDFSSource)source);
        reader.readables((Iterator<Readable>)ImmutableSet.of((Object)readable).iterator());
        return reader;
    }

    @Override
    public void close() {
        super.close();
    }

    public void closeFileSystem(FileSystem fileSystem) {
        try {
            fileSystem.close();
        }
        catch (IOException e) {
            LOG.warn("Failed to close reader for {} with exception {}", new Object[]{this.source(), e.getMessage(), e});
        }
    }

    @Override
    public boolean multiReaders() {
        return true;
    }

    @Override
    protected List<Readable> scanReadables() throws IOException {
        Path path = new Path(this.source().path());
        FileFilter filter = this.source().filter();
        ArrayList<Readable> paths = new ArrayList<Readable>();
        FileStatus status = this.hdfs.getFileStatus(path);
        if (status.isFile()) {
            if (!filter.reserved(path.getName())) {
                throw new LoadException("Please check path name and extensions, ensure that at least one path is available for reading");
            }
            paths.add(new HDFSFile(this.hdfs, path));
        } else {
            Path[] subPaths;
            assert (status.isDirectory());
            FileStatus[] statuses = this.hdfs.listStatus(path);
            for (Path subPath : subPaths = FileUtil.stat2Paths((FileStatus[])statuses)) {
                if (this.hdfs.getFileStatus(subPath).isFile() && this.isReservedFile(subPath)) {
                    paths.add(new HDFSFile(this.hdfs, subPath, this.source().path()));
                }
                if (!status.isDirectory()) continue;
                for (Path dirSubPath : this.listDirWithFilter(subPath)) {
                    if (!this.isReservedFile(dirSubPath)) continue;
                    paths.add(new HDFSFile(this.hdfs, dirSubPath, this.source().path()));
                }
            }
        }
        return paths;
    }

    private boolean isReservedFile(Path path) throws IOException {
        FileStatus status = this.hdfs.getFileStatus(path);
        FileFilter filter = this.source().filter();
        return status.getLen() > 0L && filter.reserved(path.getName());
    }

    private List<Path> listDirWithFilter(Path dir) throws IOException {
        DirFilter dirFilter = this.source().dirFilter();
        ArrayList<Path> files = new ArrayList<Path>();
        FileStatus status = this.hdfs.getFileStatus(dir);
        if (status.isFile()) {
            files.add(dir);
        }
        if (status.isDirectory() && dirFilter.reserved(dir.getName())) {
            FileStatus[] statuses = this.hdfs.listStatus(dir);
            Path[] subPaths = FileUtil.stat2Paths((FileStatus[])statuses);
            if (subPaths == null) {
                throw new LoadException("Error while listing the files of dir path '%s'", dir);
            }
            for (Path subFile : subPaths) {
                if (this.hdfs.getFileStatus(subFile).isFile()) {
                    files.add(subFile);
                }
                if (!this.hdfs.getFileStatus(subFile).isDirectory()) continue;
                files.addAll(this.listDirWithFilter(subFile));
            }
        }
        return files;
    }

    @Override
    protected FileLineFetcher createLineFetcher() {
        if (Compression.ORC == this.source().compression()) {
            return new OrcFileLineFetcher(this.source(), this.conf);
        }
        if (Compression.PARQUET == this.source().compression()) {
            return new ParquetFileLineFetcher(this.source(), this.conf);
        }
        return new FileLineFetcher(this.source());
    }

    private Configuration loadConfiguration() {
        Configuration conf = new Configuration();
        conf.addResource(new Path(this.source().coreSitePath()));
        if (this.source().hdfsSitePath() != null) {
            conf.addResource(new Path(this.source().hdfsSitePath()));
        }
        return conf;
    }

    private void checkExist(Path path) {
        try {
            LOG.debug("to check exist {}", (Object)path.getName());
            if (!this.hdfs.exists(path)) {
                throw new LoadException("Please ensure the file or directory exists: '%s'", path);
            }
            LOG.debug("finished check exist {}", (Object)path.getName());
        }
        catch (IOException e) {
            throw new LoadException("An exception occurred while checking HDFS path: '%s'", (Throwable)e, path);
        }
    }

    private static class HDFSFile
    implements Readable {
        private final FileSystem hdfs;
        private final Path path;
        private final String inputPath;

        private HDFSFile(FileSystem hdfs, Path path) {
            this(hdfs, path, (String)null);
        }

        private HDFSFile(FileSystem hdfs, Path path, String inputpath) {
            this.hdfs = hdfs;
            this.path = path;
            this.inputPath = inputpath;
        }

        public FileSystem hdfs() {
            return this.hdfs;
        }

        @Override
        public String name() {
            return this.relativeName();
        }

        private String relativeName() {
            if (!StringUtils.isEmpty((CharSequence)this.inputPath) && Paths.get(this.inputPath, new String[0]).isAbsolute()) {
                String strPath = this.path.toUri().getPath();
                return Paths.get(this.inputPath, new String[0]).relativize(Paths.get(strPath, new String[0])).toString();
            }
            return this.path.getName();
        }

        @Override
        public Path path() {
            return this.path;
        }

        @Override
        public InputStream open() throws IOException {
            return this.hdfs.open(this.path);
        }

        @Override
        public InputItemProgress inputItemProgress() {
            byte[] bytes;
            long timestamp;
            String name = this.path.getName();
            try {
                timestamp = this.hdfs.getFileStatus(this.path).getModificationTime();
            }
            catch (IOException e) {
                throw new LoadException("Failed to get last modified time for HDFS path '%s'", (Throwable)e, this.path);
            }
            try {
                FileChecksum checksum = this.hdfs.getFileChecksum(this.path);
                if (checksum == null) {
                    throw new LoadException("The checksum of HDFS path '%s' is null", this.path);
                }
                bytes = checksum.getBytes();
            }
            catch (IOException e) {
                throw new LoadException("Failed to calculate checksum for HDFS path '%s'", (Throwable)e, this.path);
            }
            String checkSum = new String(bytes, Constants.CHARSET);
            return new FileItemProgress(name, timestamp, checkSum, 0L);
        }

        public String toString() {
            return "HDFS: " + this.path;
        }
    }
}

