/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import java.util.Spliterators;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.utils.Exit;
import org.apache.kafka.server.util.CommandLineUtils;

public class LogCompactionTester {
    private static final Random RANDOM = new Random();

    public static void main(String[] args) throws Exception {
        OptionParser parser = new OptionParser(false);
        Options options = new Options(parser);
        OptionSet optionSet = parser.parse(args);
        if (args.length == 0) {
            CommandLineUtils.printUsageAndExit((OptionParser)parser, (String)"A tool to test log compaction. Valid options are: ");
        }
        CommandLineUtils.checkRequiredArgs((OptionParser)parser, (OptionSet)optionSet, (OptionSpec[])new OptionSpec[]{options.brokerOpt, options.numMessagesOpt});
        long messages = (Long)optionSet.valueOf(options.numMessagesOpt);
        CompressionType compressionType = CompressionType.forName((String)((String)optionSet.valueOf(options.messageCompressionOpt)));
        Integer compressionLevel = (Integer)optionSet.valueOf(options.compressionLevelOpt);
        int percentDeletes = (Integer)optionSet.valueOf(options.percentDeletesOpt);
        int dups = (Integer)optionSet.valueOf(options.numDupsOpt);
        String brokerUrl = (String)optionSet.valueOf(options.brokerOpt);
        int topicCount = (Integer)optionSet.valueOf(options.topicsOpt);
        int sleepSecs = (Integer)optionSet.valueOf(options.sleepSecsOpt);
        long testId = RANDOM.nextLong();
        Set topics = IntStream.range(0, topicCount).mapToObj(i -> "log-cleaner-test-" + testId + "-" + i).collect(Collectors.toCollection(LinkedHashSet::new));
        LogCompactionTester.createTopics(brokerUrl, topics);
        System.out.println("Producing " + messages + " messages..to topics " + String.join((CharSequence)",", topics));
        Path producedDataFilePath = LogCompactionTester.produceMessages(brokerUrl, topics, messages, compressionType, compressionLevel, dups, percentDeletes);
        System.out.println("Sleeping for " + sleepSecs + "seconds...");
        TimeUnit.MILLISECONDS.sleep((long)sleepSecs * 1000L);
        System.out.println("Consuming messages...");
        Path consumedDataFilePath = LogCompactionTester.consumeMessages(brokerUrl, topics);
        long producedLines = LogCompactionTester.lineCount(producedDataFilePath);
        long consumedLines = LogCompactionTester.lineCount(consumedDataFilePath);
        double reduction = 100.0 * (1.0 - (double)consumedLines / (double)producedLines);
        System.out.printf("%d rows of data produced, %d rows of data consumed (%.1f%% reduction).%n", producedLines, consumedLines, reduction);
        System.out.println("De-duplicating and validating output files...");
        LogCompactionTester.validateOutput(producedDataFilePath.toFile(), consumedDataFilePath.toFile());
        Files.deleteIfExists(producedDataFilePath);
        Files.deleteIfExists(consumedDataFilePath);
        System.out.println("Data verification is completed");
    }

    private static void createTopics(String brokerUrl, Set<String> topics) throws Exception {
        Properties adminConfig = new Properties();
        adminConfig.put("bootstrap.servers", brokerUrl);
        try (Admin adminClient = Admin.create((Properties)adminConfig);){
            Map<String, String> topicConfigs = Map.of("cleanup.policy", "compact");
            List<NewTopic> newTopics = topics.stream().map(name -> new NewTopic(name, 1, 1).configs(topicConfigs)).toList();
            adminClient.createTopics(newTopics).all().get();
            ArrayList pendingTopics = new ArrayList();
            LogCompactionTester.waitUntilTrue(() -> {
                try {
                    Set allTopics = (Set)adminClient.listTopics().names().get();
                    pendingTopics.clear();
                    pendingTopics.addAll(topics.stream().filter(topicName -> !allTopics.contains(topicName)).toList());
                    return pendingTopics.isEmpty();
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new RuntimeException(e);
                }
            }, () -> "timed out waiting for topics: " + String.valueOf(pendingTopics));
        }
    }

    private static void validateOutput(File producedDataFile, File consumedDataFile) {
        try (BufferedReader producedReader = LogCompactionTester.externalSort(producedDataFile);
             BufferedReader consumedReader = LogCompactionTester.externalSort(consumedDataFile);){
            Iterator<TestRecord> produced = TestRecordUtils.valuesIterator(producedReader);
            Iterator<TestRecord> consumed = TestRecordUtils.valuesIterator(consumedReader);
            File producedDedupedFile = new File(producedDataFile.getAbsolutePath() + ".deduped");
            File consumedDedupedFile = new File(consumedDataFile.getAbsolutePath() + ".deduped");
            try (BufferedWriter producedDeduped = Files.newBufferedWriter(producedDedupedFile.toPath(), StandardCharsets.UTF_8, new OpenOption[0]);
                 BufferedWriter consumedDeduped = Files.newBufferedWriter(consumedDedupedFile.toPath(), StandardCharsets.UTF_8, new OpenOption[0]);){
                int total = 0;
                int mismatched = 0;
                while (produced.hasNext() && consumed.hasNext()) {
                    TestRecord p = produced.next();
                    producedDeduped.write(p.toString());
                    producedDeduped.newLine();
                    TestRecord c = consumed.next();
                    consumedDeduped.write(c.toString());
                    consumedDeduped.newLine();
                    if (!p.equals(c)) {
                        ++mismatched;
                    }
                    ++total;
                }
                System.out.printf("Validated %d values, %d mismatches.%n", total, mismatched);
                LogCompactionTester.require(!produced.hasNext(), "Additional values produced not found in consumer log.");
                LogCompactionTester.require(!consumed.hasNext(), "Additional values consumed not found in producer log.");
                LogCompactionTester.require(mismatched == 0, "Non-zero number of row mismatches.");
                Files.deleteIfExists(producedDedupedFile.toPath());
                Files.deleteIfExists(consumedDedupedFile.toPath());
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static BufferedReader externalSort(File file) throws IOException {
        Process process;
        Path tempDir = Files.createTempDirectory("log_compaction_test", new FileAttribute[0]);
        ProcessBuilder builder = new ProcessBuilder("sort", "--key=1,2", "--stable", "--buffer-size=20%", "--temporary-directory=" + tempDir.toString(), file.getAbsolutePath());
        builder.redirectError(ProcessBuilder.Redirect.INHERIT);
        try {
            process = builder.start();
        }
        catch (IOException e) {
            try {
                Files.deleteIfExists(tempDir);
            }
            catch (IOException cleanupException) {
                e.addSuppressed(cleanupException);
            }
            throw new IOException("Failed to start sort process. Ensure 'sort' command is available.", e);
        }
        return new BufferedReader(new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8), 0xA00000);
    }

    private static long lineCount(Path filePath) throws IOException {
        try (Stream<String> lines = Files.lines(filePath);){
            long l = lines.count();
            return l;
        }
    }

    private static void require(boolean requirement, String message) {
        if (!requirement) {
            System.err.println("Data validation failed : " + message);
            Exit.exit((int)1);
        }
    }

    private static Path produceMessages(String brokerUrl, Set<String> topics, long messages, CompressionType compressionType, Integer compressionLevel, int dups, int percentDeletes) throws IOException {
        HashMap<String, Object> producerProps = new HashMap<String, Object>();
        producerProps.put("max.block.ms", String.valueOf(Long.MAX_VALUE));
        producerProps.put("bootstrap.servers", brokerUrl);
        producerProps.put("compression.type", compressionType.name);
        if (compressionLevel != null) {
            switch (compressionType) {
                case GZIP: {
                    producerProps.put("compression.gzip.level", compressionLevel);
                    break;
                }
                case LZ4: {
                    producerProps.put("compression.lz4.level", compressionLevel);
                    break;
                }
                case ZSTD: {
                    producerProps.put("compression.zstd.level", compressionLevel);
                    break;
                }
                default: {
                    System.out.println("Warning: Compression level " + compressionLevel + " is ignored for compression type " + compressionType.name + ". Only gzip, lz4, and zstd support compression levels.");
                }
            }
        }
        try (KafkaProducer producer = new KafkaProducer(producerProps, (Serializer)new ByteArraySerializer(), (Serializer)new ByteArraySerializer());){
            int keyCount = (int)(messages / (long)dups);
            Path producedFilePath = Files.createTempFile("kafka-log-cleaner-produced-", ".txt", new FileAttribute[0]);
            System.out.println("Logging produce requests to " + String.valueOf(producedFilePath));
            try (BufferedWriter producedWriter = Files.newBufferedWriter(producedFilePath, StandardCharsets.UTF_8, new OpenOption[0]);){
                List<String> topicsList = List.copyOf(topics);
                int size = topicsList.size();
                for (long i = 0L; i < messages * (long)size; ++i) {
                    String topic = topicsList.get((int)(i % (long)size));
                    int key = RANDOM.nextInt(keyCount);
                    boolean delete = i % 100L < (long)percentDeletes;
                    ProducerRecord record = delete ? new ProducerRecord(topic, (Object)String.valueOf(key).getBytes(StandardCharsets.UTF_8), null) : new ProducerRecord(topic, (Object)String.valueOf(key).getBytes(StandardCharsets.UTF_8), (Object)String.valueOf(i).getBytes(StandardCharsets.UTF_8));
                    producer.send(record);
                    producedWriter.write(new TestRecord(topic, key, i, delete).toString());
                    producedWriter.newLine();
                }
            }
            Path path = producedFilePath;
            return path;
        }
    }

    private static Path consumeMessages(String brokerUrl, Set<String> topics) throws IOException {
        Path consumedFilePath = Files.createTempFile("kafka-log-cleaner-consumed-", ".txt", new FileAttribute[0]);
        System.out.println("Logging consumed messages to " + String.valueOf(consumedFilePath));
        try (Consumer<String, String> consumer = LogCompactionTester.createConsumer(brokerUrl);
             BufferedWriter consumedWriter = Files.newBufferedWriter(consumedFilePath, StandardCharsets.UTF_8, new OpenOption[0]);){
            consumer.subscribe(topics);
            while (true) {
                ConsumerRecords consumerRecords;
                if ((consumerRecords = consumer.poll(Duration.ofSeconds(20L))).isEmpty()) {
                    Path path = consumedFilePath;
                    return path;
                }
                consumerRecords.forEach(record -> {
                    try {
                        boolean delete = record.value() == null;
                        long value = delete ? -1L : Long.parseLong((String)record.value());
                        TestRecord testRecord = new TestRecord(record.topic(), Integer.parseInt((String)record.key()), value, delete);
                        consumedWriter.write(testRecord.toString());
                        consumedWriter.newLine();
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                });
            }
        }
    }

    private static Consumer<String, String> createConsumer(String brokerUrl) {
        Map<String, String> consumerProps = Map.of("group.id", "log-cleaner-test-" + RANDOM.nextInt(Integer.MAX_VALUE), "bootstrap.servers", brokerUrl, "auto.offset.reset", "earliest");
        return new KafkaConsumer(consumerProps, (Deserializer)new StringDeserializer(), (Deserializer)new StringDeserializer());
    }

    private static void waitUntilTrue(Supplier<Boolean> condition, Supplier<String> timeoutMessage) throws InterruptedException {
        long defaultMaxWaitMs = 15000L;
        long defaultPollIntervalMs = 100L;
        long endTime = System.currentTimeMillis() + 15000L;
        while (System.currentTimeMillis() < endTime) {
            try {
                if (condition.get().booleanValue()) {
                    return;
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            TimeUnit.MILLISECONDS.sleep(100L);
        }
        throw new RuntimeException(timeoutMessage.get());
    }

    public static class Options {
        public final OptionSpec<Long> numMessagesOpt;
        public final OptionSpec<String> messageCompressionOpt;
        public final OptionSpec<Integer> compressionLevelOpt;
        public final OptionSpec<Integer> numDupsOpt;
        public final OptionSpec<String> brokerOpt;
        public final OptionSpec<Integer> topicsOpt;
        public final OptionSpec<Integer> percentDeletesOpt;
        public final OptionSpec<Integer> sleepSecsOpt;
        public final OptionSpec<Void> helpOpt;

        public Options(OptionParser parser) {
            this.numMessagesOpt = parser.accepts("messages", "The number of messages to send or consume.").withRequiredArg().describedAs("count").ofType(Long.class).defaultsTo((Object)Long.MAX_VALUE, (Object[])new Long[0]);
            this.messageCompressionOpt = parser.accepts("compression-type", "message compression type").withOptionalArg().describedAs("compressionType").ofType(String.class).defaultsTo((Object)"none", (Object[])new String[0]);
            this.compressionLevelOpt = parser.accepts("compression-level", "The compression level to use with the specified compression type.").withOptionalArg().describedAs("level").ofType(Integer.class);
            this.numDupsOpt = parser.accepts("duplicates", "The number of duplicates for each key.").withRequiredArg().describedAs("count").ofType(Integer.class).defaultsTo((Object)5, (Object[])new Integer[0]);
            this.brokerOpt = parser.accepts("bootstrap-server", "The server(s) to connect to.").withRequiredArg().describedAs("url").ofType(String.class);
            this.topicsOpt = parser.accepts("topics", "The number of topics to test.").withRequiredArg().describedAs("count").ofType(Integer.class).defaultsTo((Object)1, (Object[])new Integer[0]);
            this.percentDeletesOpt = parser.accepts("percent-deletes", "The percentage of updates that are deletes.").withRequiredArg().describedAs("percent").ofType(Integer.class).defaultsTo((Object)0, (Object[])new Integer[0]);
            this.sleepSecsOpt = parser.accepts("sleep", "Time in milliseconds to sleep between production and consumption.").withRequiredArg().describedAs("ms").ofType(Integer.class).defaultsTo((Object)0, (Object[])new Integer[0]);
            this.helpOpt = parser.acceptsAll(List.of("h", "help"), "Display help information");
        }
    }

    public static class TestRecordUtils {
        private static final int READ_AHEAD_LIMIT = 4906;

        public static TestRecord readNext(BufferedReader reader) throws IOException {
            String line = reader.readLine();
            if (line == null) {
                return null;
            }
            TestRecord curr = TestRecord.parse(line);
            String peekedLine;
            while ((peekedLine = TestRecordUtils.peekLine(reader)) != null) {
                TestRecord next = TestRecord.parse(peekedLine);
                if (!next.getTopicAndKey().equals(curr.getTopicAndKey())) {
                    return curr;
                }
                curr = next;
                reader.readLine();
            }
            return curr;
        }

        public static Iterator<TestRecord> valuesIterator(final BufferedReader reader) {
            return Spliterators.iterator(new Spliterators.AbstractSpliterator<TestRecord>(Long.MAX_VALUE, 16){

                @Override
                public boolean tryAdvance(java.util.function.Consumer<? super TestRecord> action) {
                    try {
                        TestRecord rec;
                        while ((rec = TestRecordUtils.readNext(reader)) != null && rec.delete) {
                        }
                        if (rec == null) {
                            return false;
                        }
                        action.accept(rec);
                        return true;
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                }
            });
        }

        public static String peekLine(BufferedReader reader) throws IOException {
            reader.mark(4906);
            String line = reader.readLine();
            reader.reset();
            return line;
        }
    }

    public record TestRecord(String topic, int key, long value, boolean delete) {
        @Override
        public String toString() {
            return this.topic + "\t" + this.key + "\t" + this.value + "\t" + (this.delete ? "d" : "u");
        }

        public String getTopicAndKey() {
            return this.topic + this.key;
        }

        public static TestRecord parse(String line) {
            String[] components = line.split("\t");
            if (components.length != 4) {
                throw new IllegalArgumentException("Invalid TestRecord format: " + line);
            }
            return new TestRecord(components[0], Integer.parseInt(components[1]), Long.parseLong(components[2]), "d".equals(components[3]));
        }
    }
}

