utils::globalVariables(c(
  "y", "est.prob", "Sensitivity", "Cutpoint", "Specificity", "Specificity1",
  "pair", "theta", "deltachi", "deltabeta", "sum_deltachi", "sum_deltabeta", 'data', 'prediction', 'mod',
  'E_bin', 'draw', 'elpd_loo', 'mean_observed', 'mean_r', 'model', 'model_name',
  'prob', 'sd_observed', 'sd_r', 'se', 'y_rep'
))


#' @title Match Variables to Model Coefficient Names
#'
#' @description
#' Internal helper used by \code{coeff.OR()} and \code{delta.coefficient()} to identify the relevant model coefficients
#' corresponding to user-specified variables. Handles both numeric and categorical variables.
#'
#' @param data A data frame containing the original variables used in the model to match.
#' @param variable A character vector of variable names to match.
#' @param coef_names A character vector of coefficient names from the model to match.
#'
#' @return A character vector of coefficient names that correspond to the specified variables.
#'
#' @noRd
#'
#'

match_variables <- function(data, variable, coef_names) {
  exact_vars <- variable[sapply(variable, function(v) is.numeric(data[[v]]))]  #Numeric
  cat_vars   <- variable[!(variable %in% exact_vars)]  #Categorical

  matched <- c()

  # Exact match for numeric vars
  if (length(exact_vars) > 0) {
    pattern_exact <- paste0("^", paste(exact_vars, collapse = "$|^"), "$")
    matched <- c(matched, coef_names[grepl(pattern_exact, coef_names)])
  }

  # Prefix match for categorical vars
  if (length(cat_vars) > 0) {
    pattern_prefix <- paste0("^", paste(cat_vars, collapse = "|^"))
    matched <- c(matched, coef_names[grepl(pattern_prefix, coef_names)])
  }

  return(unique(matched))
}


#' @title Simulate Posterior Predictive Samples from a Bayesian Logistic Regression Model
#'
#' @description
#' This helper function generates simulated binary outcomes based on posterior draws from a fitted Bayesian logistic regression model using `rstan`.
#'
#' @param mod A fitted model object of class `stanfit` generated by `rstan::stan()`.
#' @param dat A data frame containing the predictor variables used in the model.
#' @param y A character string indicating the name of the binary outcome variable in `dat`.
#' @param sim An integer specifying the number of posterior draws (simulations) to use for prediction.
#' @param inter Optional. A character string specifying the name of the intercept parameter in the model. If `NULL`, no intercept is added.
#' @param var_param A character vector specifying the names of the slope parameters in the posterior (i.e., the names used in `posterior_pred`).
#' @param var_names A character vector with the names of the variables in the dataset that correspond to the model's slopes.
#'
#' @return A matrix of simulated outcomes of dimension `sim` × `N`, where `N` is the number of observations. Each row corresponds to a posterior draw.
#'
#' @noRd
#'
#'

predict_posterior <- function(mod, dat, y, sim, inter = NULL, var_param, var_names){
  N <- dim(dat)[1]
  posterior_pred <- rstan::extract(mod)

  ysim <- matrix(NA, nrow = sim, ncol = N)

  ## Doing the matrix for the calculations
  matrix_model <- do.call(cbind, posterior_pred[var_param])
  data_model <- do.call(cbind, data[var_names])

  for (s in 1:sim) {
    if(is.null(inter)){
      p <- plogis(matrix_model[s] %*% t(data_model))
      ysim[s, ] <- rbinom(N, size = 1, prob = p)
    }else{
      p <- plogis(matrix_model[s, inter] + matrix_model[s, -which(colnames(matrix_model) == inter)] %*% t(data_model))
      ysim[s, ] <- rbinom(N, size = 1, prob = p)
    }
  }

  return(ysim)
}

