% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.correlation.R
\name{sim.correlation}
\alias{sim.correlation}
\title{Generate a Random Correlation Matrix via C-Vine Partial Correlations}
\usage{
sim.correlation(
  I,
  eta = 1,
  skew = 0,
  positive = FALSE,
  permute = TRUE,
  maxiter = 10
)
}
\arguments{
\item{I}{Dimension of the correlation matrix (must be \eqn{I \geq 1}).}

\item{eta}{LKJ concentration parameter (\eqn{\eta > 0}). When \eqn{\eta = 1} and \eqn{\text{skew} = 0}, the
distribution is uniform over correlation matrices. Larger \eqn{\eta} values concentrate
mass near the identity matrix. Critical for positive definiteness: Requires \eqn{\eta > (I-2)/2}
to theoretically guarantee positive definiteness (Theorem 1, Joe & Kurowicka 2026). Default is 1.}

\item{skew}{Skewness parameter (\eqn{-1 < \text{skew} < 1}). Controls asymmetry in the partial correlation distribution:
\itemize{
\item \eqn{\text{skew} > 0}: Biased toward positive partial correlations
\item \eqn{\text{skew} < 0}: Biased toward negative partial correlations
\item \eqn{\text{skew} = 0}: Symmetric distribution (default)
}}

\item{positive}{Logical. If \code{TRUE}, restricts partial correlations to \eqn{(0,1)} and enforces positive definiteness. Default is \code{FALSE}.}

\item{permute}{Logical. If \code{TRUE}, applies a random permutation to rows/columns to ensure exchangeability (invariance to variable ordering). Default is \code{TRUE}.}

\item{maxiter}{Integer. Maximum number of generation attempts before numerical adjustment when \code{positive = TRUE}. Default is 10.}
}
\value{
An \eqn{I \times I} positive definite correlation matrix with unit diagonal.
}
\description{
This function generates a random \eqn{I \times I} correlation matrix using the C-vine partial correlation
parameterization described in Joe & Kurowicka (2026). The method constructs the matrix recursively using
partial correlations organized in a C-vine structure, with distributional properties controlled by LKJ
concentration and skewness parameters.
}
\details{
The algorithm follows four key steps:
\enumerate{
\item Partial correlation sampling:
For tree level \eqn{k = 1, \dots, I-1} and node \eqn{j = k+1, \dots, I}, partial correlations \eqn{\rho_{k,j \mid 1:(k-1)}} are sampled as:
\deqn{
  \alpha_k = \eta + \frac{I - k - 1}{2}, \quad
  a_k = \alpha_k (1 + \text{skew}), \quad
  b_k = \alpha_k (1 - \text{skew})
}
\itemize{
\item If \code{positive = FALSE}:
\deqn{\rho_{k,j} \sim 2 \cdot \mathrm{Beta}(a_k, b_k) - 1}
\item If \code{positive = TRUE}:
\deqn{\rho_{k,j} \sim \mathrm{Beta}(a_k, b_k)}
}
\item Recursive matrix construction (C-vine):
The correlation matrix \eqn{\mathbf{R}} is built without matrix inversion using backward recursion:
\itemize{
\item Tree 1 (raw correlations): \eqn{R_{1j} = \rho_{1,j}} for \eqn{j = 2,\dots,I}
\item Trees \eqn{l \geq 2}: For pairs \eqn{(l,j)} where \eqn{l = 2,\dots,I-1} and \eqn{j = l+1,\dots,I}:
\deqn{
    c \gets \rho_{l,j \mid 1:(l-1)} \\
    \text{for } k = l-1 \text{ down to } 1: \\
    \quad c \gets c \cdot \sqrt{(1 - \rho_{k,l}^2)(1 - \rho_{k,j}^2)} + \rho_{k,l} \cdot \rho_{k,j} \\
    R_{lj} \gets c
  }
}
This implements the dynamic programming approach from Joe & Kurowicka (2026, Section 2.1).
\item Positive definiteness enforcement (when \code{positive = TRUE}):
\itemize{
\item Attempt up to \code{maxiter} generations
\item On failure, project to nearest positive definite correlation matrix using \code{\link[Matrix]{nearPD}} with \code{corr = TRUE}
\item Final matrix has minimum eigenvalue > 1e-8
}
\item Exchangeability (optional):
If \code{permute = TRUE}, rows/columns are randomly permuted before returning the matrix.
}
}
\note{
When \code{positive = TRUE}, the function guarantees positive definiteness either through direct generation
(with retries) or numerical projection. The theoretical guarantee \eqn{\eta > (I-2)/2} is recommended for high dimensions.
}
\examples{
# Default 3x3 correlation matrix
sim.correlation(3)

# 5x5 matrix concentrated near identity (eta=3)
sim.correlation(5, eta = 3)

# Skewed toward positive correlations (no permutation)
sim.correlation(4, skew = 0.7, permute = FALSE)

# Positive partial correlations (enforced positive definiteness)
R <- sim.correlation(6, positive = TRUE)
min(eigen(R, symmetric = TRUE, only.values = TRUE)$values)  # > 1e-8

# High-dimensional case (I=20) with theoretical guarantee
R <- sim.correlation(20, eta = 10)  # eta=10 > (20-2)/2=9
min(eigen(R, symmetric = TRUE, only.values = TRUE)$values)

}
\references{
Joe, H., & Kurowicka, D. (2026). Random correlation matrices generated via partial correlation C-vines. Journal of Multivariate Analysis, 211, 105519. https://doi.org/10.1016/j.jmva.2025.105519
}
