#' Proportion Positive Predictions
#'
#' Internal function used by \code{\link{predict.logforest}} to determine the proportion of logic regression trees
#' within a logic forest that predict a class of one for new observations.  
#' It also returns the predicted class values based on a specified cutoff.
#'
#' @param predictmatrix A matrix of predicted values from each tree (rows = observations, columns = trees).
#' @param cutoff Numeric value specifying the proportion of trees that must predict a class of one for the 
#'               overall prediction to be class one.
#'
#' @return A list with:
#' \item{predmat}{A two-column matrix where the first column is the proportion of trees predicting class one 
#' for each observation, and the second column is the binary predicted class (0 or 1).}
#'
#' @details
#' This function is called internally by \code{\link{predict.logforest}} and is not intended for direct use.
#' It calculates, for each observation, the fraction of trees in the logic forest predicting a positive outcome,
#' and then assigns a predicted class based on whether this fraction meets or exceeds the \code{cutoff}.
#'
#' @author Bethany Wolf \email{wolfb@@musc.edu}
#'
#' @note This is a supplementary function and not intended to be used independently of the other functions in the package.
#'
#' @keywords internal
#' @export
proportion.positive<-function(predictmatrix, cutoff)
{
  q<-nrow(predictmatrix)
  ntrees<-ncol(predictmatrix)
  status<-c()
  predict.pos<-c()
  for (a in 1:q)
  {
    number.diseasepositive<-sum(predictmatrix[a,])
    proportion.predictpositive<-number.diseasepositive/ntrees
    if (proportion.predictpositive >= cutoff)
      disease.status <- 1
    else if (proportion.predictpositive < cutoff)
      disease.status <- 0
    status<-append(status, disease.status)  
    predict.pos<-append(predict.pos, proportion.predictpositive)
  }
  predmat<-cbind(predict.pos, status)
  ans<-list(predmat=predmat)
}