#' @title Package Startup Messages
#'
#' @description
#' Functions called when package is loaded and attached.
#'
#' @return No return value, called for side effects.
#' @name zzz
#' @keywords internal
NULL


.onLoad <- function(libname, pkgname) {
    # Check for optional dependencies and set package options
    options(SVG.has.geometry = requireNamespace("geometry", quietly = TRUE))
    options(SVG.has.RANN = requireNamespace("RANN", quietly = TRUE))
    options(SVG.has.BRISC = requireNamespace("BRISC", quietly = TRUE))
    options(SVG.has.CompQuadForm = requireNamespace("CompQuadForm", quietly = TRUE))
    options(SVG.has.BiocParallel = requireNamespace("BiocParallel", quietly = TRUE))
}


.onAttach <- function(libname, pkgname) {
    version <- utils::packageVersion(pkgname)

    packageStartupMessage(
        "SVG: Spatially Variable Genes Detection Methods\n",
        "Version: ", version, "\n",
        "Author: Zaoqu Liu (liuzaoqu@163.com)\n",
        "GitHub: https://github.com/Zaoqu-Liu/SVG\n",
        "---\n",
        "Available methods: MERINGUE, binSpect, SPARK-X, nnSVG\n",
        "Use CalSVG() for unified interface or CalSVG_*() for specific methods"
    )

    # Check dependencies
    missing_deps <- c()

    if (!getOption("SVG.has.geometry", FALSE)) {
        missing_deps <- c(missing_deps, "geometry (for Delaunay triangulation)")
    }
    if (!getOption("SVG.has.RANN", FALSE)) {
        missing_deps <- c(missing_deps, "RANN (for KNN network)")
    }
    if (!getOption("SVG.has.BRISC", FALSE)) {
        missing_deps <- c(missing_deps, "BRISC (required for nnSVG method)")
    }

    if (length(missing_deps) > 0) {
        packageStartupMessage(
            "\nOptional packages not installed:\n",
            paste(" -", missing_deps, collapse = "\n"),
            "\nInstall with: install.packages(c('geometry', 'RANN', 'BRISC'))"
        )
    }
}


#' Check if C++ Functions are Available
#'
#' @description
#' Checks if the compiled C++ code is available and working.
#'
#' @return Logical indicating whether C++ is available.
#'
#' @keywords internal
.check_cpp <- function() {
    tryCatch({
        # Test with small matrix
        test_mat <- matrix(1:4, 2, 2)
        result <- dist_matrix_cpp(test_mat)
        return(TRUE)
    }, error = function(e) {
        return(FALSE)
    })
}
