% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils_spatial.R
\name{buildSpatialNetwork}
\alias{buildSpatialNetwork}
\title{Build Spatial Neighborhood Network}
\usage{
buildSpatialNetwork(
  coords,
  method = c("delaunay", "knn"),
  k = 10L,
  filter_dist = NA,
  binary = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{coords}{Numeric matrix of spatial coordinates.
Rows are spatial locations, columns are coordinate dimensions (typically x, y).}

\item{method}{Character string specifying the network construction method.
\itemize{
  \item \code{"delaunay"}: Delaunay triangulation (default). Creates a network
    where neighbors are determined by triangulation. Works well for relatively
    uniform spatial distributions.
  \item \code{"knn"}: K-nearest neighbors. Each spot is connected to its k
    nearest neighbors based on Euclidean distance.
}}

\item{k}{Integer. Number of nearest neighbors for KNN method. Default is 10.
Ignored when \code{method = "delaunay"}.}

\item{filter_dist}{Numeric or NA. Maximum distance threshold for neighbors.
Pairs with distance > filter_dist are not considered neighbors.
Default is NA (no filtering).}

\item{binary}{Logical. If TRUE (default), return binary adjacency matrix (0/1).
If FALSE, return distance-weighted adjacency matrix.}

\item{verbose}{Logical. Whether to print progress messages. Default is FALSE.}
}
\value{
A square numeric matrix representing the spatial adjacency/weight matrix.
  Row and column names correspond to the spatial locations (from rownames of coords).
  \itemize{
    \item If \code{binary = TRUE}: Values are 1 (neighbors) or 0 (non-neighbors)
    \item If \code{binary = FALSE}: Values are Euclidean distances (0 for non-neighbors)
  }
}
\description{
Constructs a spatial neighborhood network from spatial coordinates using
either Delaunay triangulation or K-nearest neighbors (KNN) approach.
}
\details{
\strong{Delaunay Triangulation:}
Creates a network based on Delaunay triangulation, which maximizes the minimum
angle of all triangles. This is a natural way to define neighbors in 2D/3D space.
Requires the \code{geometry} package.

\strong{K-Nearest Neighbors:}
Connects each point to its k nearest neighbors based on Euclidean distance.
More robust to irregular spatial distributions but requires choosing k.
Requires the \code{RANN} package.
}
\examples{
# Generate example coordinates
set.seed(42)
coords <- cbind(x = runif(100), y = runif(100))
rownames(coords) <- paste0("spot_", 1:100)

\donttest{
# Build network using Delaunay (requires geometry package)
if (requireNamespace("geometry", quietly = TRUE)) {
    W_delaunay <- buildSpatialNetwork(coords, method = "delaunay")
}

# Build network using KNN (requires RANN package)
if (requireNamespace("RANN", quietly = TRUE)) {
    W_knn <- buildSpatialNetwork(coords, method = "knn", k = 6)
}
}

}
\seealso{
\code{\link{getSpatialNeighbors_Delaunay}}, \code{\link{getSpatialNeighbors_KNN}}
}
