/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ACmodule.h" 
#define H5F_FRIEND      

#include "H5private.h"  
#include "H5ACpkg.h"    
#include "H5Eprivate.h" 
#include "H5Fpkg.h"     

herr_t
H5AC_stats(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    
    (void)H5C_stats(f->shared->cache, H5F_OPEN_NAME(f), false);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

#ifndef NDEBUG

herr_t
H5AC_dump_cache(const H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    if (H5C_dump_cache(f->shared->cache, H5F_OPEN_NAME(f)) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_dump_cache() failed.");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5AC_get_entry_ptr_from_addr(const H5F_t *f, haddr_t addr, void **entry_ptr_ptr)
{
    H5C_t *cache_ptr;           
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    if (H5C_get_entry_ptr_from_addr(cache_ptr, addr, entry_ptr_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_entry_ptr_from_addr() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5AC_flush_dependency_exists(H5F_t *f, haddr_t parent_addr, haddr_t child_addr, bool *fd_exists_ptr)
{
    H5C_t *cache_ptr;        
    herr_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    ret_value = H5C_flush_dependency_exists(cache_ptr, parent_addr, child_addr, fd_exists_ptr);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5AC_verify_entry_type(const H5F_t *f, haddr_t addr, const H5AC_class_t *expected_type, bool *in_cache_ptr,
                       bool *type_ok_ptr)
{
    H5C_t *cache_ptr;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    if (H5C_verify_entry_type(cache_ptr, addr, expected_type, in_cache_ptr, type_ok_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_verify_entry_type() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
bool
H5AC_get_serialization_in_progress(H5F_t *f)
{
    H5C_t *cache_ptr;
    bool   ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    
    ret_value = H5C_get_serialization_in_progress(cache_ptr);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
bool
H5AC_cache_is_clean(const H5F_t *f, H5AC_ring_t inner_ring)
{
    H5C_t *cache_ptr;
    bool   ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    ret_value = H5C_cache_is_clean(cache_ptr, inner_ring);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 
