\name{icm_test}
\alias{icm_test}
\title{Test the Validity of an Independent Component Model}
\description{
Performs a test of the independent component model (ICM) assumption. The function first estimates
independent components using an ICA method and then tests whether the estimated components are
mutually independent. The test statistic is an L2-type functional of empirical characteristic functions,
with optional score transformations and resampling-based p-values.
}
\usage{
icm_test(X, n.rep = 200, score = "Identity", weight = "Gauss",
         ica = "JADE", strategy = "bootstrap", ncores = NULL,
         iseed = NULL, eps = 1e-06, maxiter = 100, g = "tanh",
         method = "sym", inR = FALSE, n.init = 2)
}
\arguments{
  \item{X}{Matrix or data frame of numeric data; rows correspond to observations and columns to variables.}
  \item{n.rep}{Number of bootstrap replications or permutations used to approximate the null distribution. Recommended are to perform several thousands.}
  \item{score}{Score transformation applied marginally prior to testing:
    \code{"Identity"} (raw estimated components),
    \code{"Rank"} (componentwise ranks),
    or \code{"VdW"} (van der Waerden/normal scores).}
  \item{weight}{Weight function used in the L2-type statistic: \code{"Gauss"} (Gaussian) or \code{"Laplace"} (Laplace).}
  \item{ica}{ICA method used to estimate the mixing/unmixing matrix: \code{"JADE"}, \code{"FOBI"}, or \code{"fICA"}.}
  \item{strategy}{Resampling strategy used to obtain p-values: \code{"bootstrap"} or \code{"permutation"}.}
  \item{ncores}{Number of CPU cores to use for resampling computations. If \code{NULL}, computation is done sequentially.}
  \item{iseed}{Optional integer seed for reproducibility of resampling.}
  \item{eps}{Convergence tolerance used by the ICA routines (where applicable).}
  \item{maxiter}{Maximum number of iterations used by the ICA routines (where applicable).}
  \item{g}{Nonlinearity used by \code{"fICA"}; see \code{\link[fICA]{fICA}}. Only used when \code{ica = "fICA"}.}
  \item{method}{FastICA variant: \code{"sym"}, \code{"sym2"}, or \code{"def"}. Only used when \code{ica = "fICA"}.}
  \item{inR}{Logical; if \code{TRUE}, compute FastICA in R. Only used when \code{ica = "fICA"}.}
  \item{n.init}{Number of random initializations for symmetric FastICA variants. Only used when \code{ica = "fICA"}.}
}
\details{
The independent component model assumes that the observed random vector \eqn{X} can be written as
\eqn{X = A S}, where \eqn{A} is an invertible mixing matrix and the latent components \eqn{S}
have mutually independent coordinates. This function assesses that assumption in two steps:

\enumerate{
  \item \strong{ICA estimation.} An unmixing matrix is estimated from \code{X} using the method selected
  via \code{ica}. This yields estimated components \eqn{\hat S}.
  \item \strong{Independence testing.} The null hypothesis is that the coordinates of \eqn{\hat S}
  are mutually independent. The test statistic is an L2-type distance between an empirical
  characteristic function of the joint distribution and the product of the corresponding marginal
  characteristic functions. The distance is computed with a weight function chosen by \code{weight}.
}

\strong{Score transformations.}
To improve robustness and/or obtain distribution-free behavior, the test can be applied not only to
\eqn{\hat S} itself (\code{score = "Identity"}), but also after marginal score transformations.
With \code{score = "Rank"}, each component is replaced by its scaled ranks, which yields a test that is
invariant under strictly increasing marginal transformations and distribution-free under the null.
With \code{score = "VdW"}, van der Waerden (normal) scores are used; this option is currently available for Gaussian weights.

\strong{P-values via resampling.}
The null distribution of the statistic is approximated by resampling:
\itemize{
  \item \code{strategy = "permutation"} breaks cross-component dependence by independently permuting
  the rows within each component (leaving marginal distributions unchanged).
  \item \code{strategy = "bootstrap"} generates a reference distribution using resampling with replacement
  under an independence structure for the components.
}
The p-value is computed as the proportion of resampled statistics at least as large as the observed one.
}
\value{
A list of class \code{"ticm"} and \code{"htest"} with components including:
\item{Tboot}{Numeric vector of bootstrap replicates of the test statistic (present when \code{strategy = "bootstrap"}).}
\item{Tperm}{Numeric vector of permutation replicates of the test statistic (present when \code{strategy = "permutation"}).}
\item{ICA}{The ICA result applied to \code{X} (typically an object of class \code{"bss"}).}
\item{statistic}{The observed value of the test statistic.}
\item{p.value}{Approximate p-value computed from \code{Tboot} or \code{Tperm}.}
\item{parameter}{The number of resampling replicates \code{n.rep}.}
\item{method}{A character string describing the selected ICA method, score type, weight, and resampling strategy.}
\item{data.name}{Name of the input data \code{X}.}
\item{alternative}{A character string describing the alternative (lack of mutual independence of the estimated components).}
}
\references{
Hallin, M., Meintanis, S.G., and Nordhausen, K. (2024),
\emph{Consistent distribution free affine invariant tests for the validity of independent component models}.
arXiv:2404.07632.
}
\seealso{
\code{\link{ind_test}},
\code{\link[fICA]{fICA}},
\code{\link[JADE]{JADE}},
\code{\link[JADE]{FOBI}}
}
\examples{
n <- 300
A <- matrix(rnorm(9), 3, 3)
s1 <- rt(n, 6)
s2 <- rexp(n, 1)
s3 <- runif(n)
S <- cbind(s1, s2, s3)
X <- S \%*\% t(A)

# in practice, the number of resamples should be much larger
res <- icm_test(X)
res
plot(res)
}
\keyword{multivariate}










