#Compute the acf and drop lag-0
my_tsa_acf <-function(x, lag.max=NULL, plot=FALSE){
  n <- length(x)-1
  z <- stats::acf(x,type="correlation", lag.max, plot)
  return(z[1:n])
}
#my_tsa_acf( c(1,2,1,2))

#We numerically approximate the acf for a given
#periodic function p and a given trawl function g
#'@title kernel_generic_trawl_acf
#'@details We numerically approximate the acf for a given
#' periodic function p and a given trawl function g
#'@param p kernel function to be used in the weighted trawl process
#'@param g trawl function
#'@param x argument for which the acf will be computed (this can be a vector)
#'@return acf or vector of acfs evaluated in x
#'@example man/examples/kernel_generic_trawl_acf.R
#'@export
kernel_generic_trawl_acf <- function(p, g, x){

  l <- length(x)
  acf_vec <- numeric(l)

  for(i in 1:l){

    f1<-function(y){p(y)*p(y+x[i])*g(y+x[i])}

    cov <- stats::integrate(f1,0,Inf)$value

    f2<-function(y){p(y)*p(y)*g(y)}

    var <- stats::integrate(f2,0,Inf)$value

    acf_vec[i]<-cov/var
  }

  return(acf_vec)


}

#We numerically approximate the acf for a given
#periodic function p and a given trawl function g
#'@title kernel_generic_trawl_acf_incrtol
#'@details We numerically approximate the acf for a given
#' periodic function p and a given trawl function g
#' We allow for an increased tolerance level in the
#' integral computations.
#'@param p kernel function to be used in the weighted trawl process
#'@param g trawl function
#'@param x argument for which the acf will be computed (this can be a vector)
#'@return acf or vector of acfs evaluated in x
#'@export
kernel_generic_trawl_acf_incrtol <- function(p, g, x){

  l <- length(x)
  acf_vec <- numeric(l)

  for(i in 1:l){

    f1<-function(y){p(y)*p(y+x[i])*g(y+x[i])}

    cov <- stats::integrate(f1,0,Inf,rel.tol=.Machine$double.eps^.05)$value

    f2<-function(y){p(y)*p(y)*g(y)}

    var <- stats::integrate(f2,0,Inf,rel.tol=.Machine$double.eps^.05)$value

    acf_vec[i]<-cov/var
  }

  return(acf_vec)


}



#We numerically approximate the acf for a given
#periodic function p and a given trawl function
#'@title kernel_trawl_acf
#'@param p kernel function to be used in the weighted trawl process
#'@param trawlfct the trawl function used (Exp, supIG or LM)
#'@param trawlfct_par parameter vector of trawl function
#'@param x argument for which the acf will be computed (this can be a vector)
#'@details We numerically approximate the acf for a given
#' periodic function p and a given trawl function g
#'@return acf or vector of acfs evaluated in x
#'@export
kernel_trawl_acf <- function(p, trawlfct, trawlfct_par, x){

  if(trawlfct=="Exp"){    g <- function(x) {trawl_Exp(-x,trawlfct_par[1])}}
  if(trawlfct=="supIG"){    g <- function(x) {trawl_supIG(-x,trawlfct_par[1],trawlfct_par[2])}}
  if(trawlfct=="LM"){    g <- function(x) {trawl_LM(-x,trawlfct_par[1],trawlfct_par[2])}}


  l <- length(x)
  acf_vec <- numeric(l)

  for(i in 1:l){

    f1<-function(y){p(y)*p(y+x[i])*g(y+x[i])}

    cov <- stats::integrate(f1,0,Inf)$value

    f2<-function(y){p(y)*p(y)*g(y)}

    var <- stats::integrate(f2,0,Inf)$value

    acf_vec[i]<-cov/var
  }

  return(acf_vec)


}


#'@title kernel_trawl_acf_exp_sin
#'@param x argument for which the acf will be computed (this can be a vector)
#'@param lambda parameter in the exponential trawl function
#'@param tau period in the sine function
#'@details We compute the acf of a weighted trawl process with exponential trawl
#'function and periodic function given by the sine function
#'@return acf or vector of acfs evaluated in x
#'@export

kernel_trawl_acf_exp_sin<-function(x, lambda, tau){
  l <- length(x)
  acf_vec <- numeric(l)
  for(i in 1:l){
    acf_vec[i]<- (4*pi)^(-1)*exp(-lambda *x[i])*(sin(2*pi*x[i]/tau)*tau*lambda+4*pi*cos(2*pi*x[i]/tau))
  }
  return(acf_vec)
}

#'Fits an exponential trawl function to equidistant time series data
#'@name fit_exp_sin_trawl_periodfixed
#'@param x vector of equidistant time series data
#'@param Delta interval length of the time grid used in the time series, the
#'  default is 1
#'@param tau period in the sine function, the default is 1
#'@param GMMlag lag length used in the GMM estimation, the default is tau
#'@param plotacf binary variable specifying whether or not the empirical and
#'  fitted autocorrelation function should be plotted
#'@param lags number of lags to be used in the plot of the autocorrelation
#'  function
#'@return lambda: the memory parameter \eqn{\lambda} in the exponential trawl
#'@details The trawl function is parametrised by the parameter \eqn{\lambda > 0}
#'  as follows: \deqn{g(x) = e^{\lambda x},  \mbox{ for }  x \le 0.} The
#'  Lebesgue measure of the corresponding trawl set is given by \eqn{1/\lambda}.
#'@export
fit_exp_sin_trawl_periodfixed <- function(x, tau=1, GMMlag, Delta=1, plotacf=FALSE,lags=100){

  if(missing(GMMlag)){GMMlag <-tau}

  my_acf <- my_tsa_acf(x)


  fit_exp_sin_trawl_foroptim <- function(y){

    lambda <- y[1]

    lag <- GMMlag
    lss <- 0

    for(i in 1:lag)
    {
      Cor <- kernel_trawl_acf_exp_sin(Delta*i,lambda,tau)
      lss <- lss+(my_acf$acf[i] - Cor)^2
    }
    lss
  }

  o <- DEoptim::DEoptim(fit_exp_sin_trawl_foroptim,0,100,control=DEoptim::DEoptim.control(itermax = 1000, trace = FALSE))

  if(plotacf){

    tt <- (1:lags)
    my_tsa_acf(x, lag.max=lags, plot=TRUE)
    #TSA::acf(x,lag.max=lags,main = "", ylab="ACF", xlab="Lags",drop.lag.0 = TRUE)
    graphics::lines(tt, kernel_trawl_acf_exp_sin(tt*Delta,o$optim$bestmem[1],tau), lty =1,col=2, lwd=2)
  }

  return(list("lambda"=as.numeric(o$optim$bestmem[1])))

}


#'Fits an exponential trawl function to equidistant time series data
#'@name fit_exp_sin_trawl
#'@param x vector of equidistant time series data
#'@param Delta interval length of the time grid used in the time series, the
#'  default is 1
#'@param GMMlag lag length used in the GMM estimation, the default is 10
#'@param plotacf binary variable specifying whether or not the empirical and
#'  fitted autocorrelation function should be plotted
#'@param lags number of lags to be used in the plot of the autocorrelation
#'  function
#'@return lambda: the memory parameter \eqn{\lambda} in the exponential trawl
#'@details The trawl function is parametrised by the parameter \eqn{\lambda > 0}
#'  as follows: \deqn{g(x) = e^{\lambda x},  \mbox{ for }  x \le 0.} The
#'  Lebesgue measure of the corresponding trawl set is given by \eqn{1/\lambda}.
#'  Both parameters \eqn{\lambda} and \eqn{\tau} are estimated by GMM.
#'@export
fit_exp_sin_trawl <- function(x, GMMlag=10, Delta=1, plotacf=FALSE,lags=100){



  my_acf <- my_tsa_acf(x)

  fit_exp_sin_trawl_foroptim <- function(y){

    lambda <- y[1]
    tau <- y[2]

    lag <- GMMlag
    lss <- 0

    for(i in 1:lag)
    {
      Cor <- kernel_trawl_acf_exp_sin(Delta*i,lambda,tau)
      lss <- lss+(my_acf$acf[i] - Cor)^2
    }
    lss
  }

  o <- DEoptim::DEoptim(fit_exp_sin_trawl_foroptim,c(0,0),c(100,100),control=DEoptim::DEoptim.control(itermax = 1000, trace = FALSE))

  if(plotacf){

    tt <- (1:lags)
    #TSA::acf(x,lag.max=lags,main = "", ylab="ACF", xlab="Lags",drop.lag.0 = TRUE)
    my_tsa_acf(x,lag.max=lags, plot=TRUE)
    graphics::lines(tt, kernel_trawl_acf_exp_sin(tt*Delta,o$optim$bestmem[1],o$optim$bestmem[2]), lty =1,col=2, lwd=2)
  }

  return(list("lambda"=as.numeric(o$optim$bestmem[1]), "tau"=o$optim$bestmem[2]))

}


#'Fits an exponential trawl function to equidistant time series data
#'@name fit_exp_sin_trawl_periodogram
#'@param x vector of equidistant time series data
#'@param Delta interval length of the time grid used in the time series, the
#'  default is 1
#'@param GMMlag lag length used in the GMM estimation, the default is 10
#'@param plotacf binary variable specifying whether or not the empirical and
#'  fitted autocorrelation function should be plotted
#'@param lags number of lags to be used in the plot of the autocorrelation
#'  function
#'@return lambda: the memory parameter \eqn{\lambda} in the exponential trawl
#'@details The trawl function is parametrised by the parameter \eqn{\lambda > 0}
#'  as follows: \deqn{g(x) = e^{\lambda x},  \mbox{ for }  x \le 0.} The
#'  Lebesgue measure of the corresponding trawl set is given by \eqn{1/\lambda}.
#'  The period parameter is estimated using the smoothed periodogram and the
#'  trawl paramter is estimated using GMM.
#'@export
fit_exp_sin_trawl_periodogram <- function(x, GMMlag=10, Delta=1, plotacf=FALSE,lags=100){

  #estimate the period tau using the smoothed periodogram
  p_smooth <- LSTS::smooth.periodogram(x, plot=TRUE)
  p_smooth_max_loc <- nnet::which.is.max(p_smooth$smooth.periodogram)

  tau <-2*pi/p_smooth$lambda[p_smooth_max_loc]*Delta

  my_acf <- my_tsa_acf(x)

  fit_exp_sin_trawl_foroptim <- function(y){

    lambda <- y[1]

    lag <- GMMlag
    lss <- 0

    for(i in 1:lag)
    {
      Cor <- kernel_trawl_acf_exp_sin(Delta*i,lambda,tau)
      lss <- lss+(my_acf$acf[i] - Cor)^2
    }
    lss
  }

  o <- DEoptim::DEoptim(fit_exp_sin_trawl_foroptim,0,100,control=DEoptim::DEoptim.control(itermax = 1000, trace = FALSE))

  if(plotacf){

    tt <- (1:lags)
    #TSA::acf(x,lag.max=lags,main = "", ylab="ACF", xlab="Lags",drop.lag.0 = TRUE)
    my_tsa_acf(x, lag.max=lags, plot=TRUE)
    graphics::lines(tt, kernel_trawl_acf_exp_sin(tt*Delta,o$optim$bestmem[1],tau), lty =1,col=2, lwd=2)
  }

  return(list("lambda"=as.numeric(o$optim$bestmem[1]), "tau"=tau))

}



#'Fits a LM periodic trawl function to equidistant time series data
#'@name fit_LM_sin_trawl
#'@param x vector of equidistant time series data
#'@param Delta interval length of the time grid used in the time series, the
#'  default is 1
#'@param GMMlag lag length used in the GMM estimation, the default is 10
#'@param plotacf binary variable specifying whether or not the empirical and
#'  fitted autocorrelation function should be plotted
#'@param lags number of lags to be used in the plot of the autocorrelation
#'  function
#'@return lambda: the memory parameter \eqn{\lambda} in the exponential trawl
#'@details The trawl function is parametrised by the parameter \eqn{\lambda > 0}
#'  as follows: \deqn{g(x) = e^{\lambda x},  \mbox{ for }  x \le 0.} The
#'  Lebesgue measure of the corresponding trawl set is given by \eqn{1/\lambda}.
#'  Both parameters \eqn{\lambda} and \eqn{\tau} are estimated by GMM.
#'@export
fit_LM_sin_trawl <- function(x, GMMlag=10, Delta=1, plotacf=FALSE,lags=100){



  my_acf <- my_tsa_acf(x)

  fit_LM_sin_trawl_foroptim <- function(y){

    alpha <- y[1]
    H <- y[2]
    tau <- y[3]

    lag <- GMMlag
    lss <- 0

    g<-function(x){trawl_LM(-x,alpha,H)}
    p<-function(x){sin(2*pi*x/tau)}

    for(i in 1:lag)
    {
      Cor <- kernel_generic_trawl_acf_incrtol(p, g, Delta*i)
      lss <- lss+(my_acf$acf[i] - Cor)^2
    }
    lss
  }

  o <- DEoptim::DEoptim(fit_LM_sin_trawl_foroptim,c(0,1,0),c(100,100,100),control=DEoptim::DEoptim.control(itermax = 1000, trace = FALSE))

  if(plotacf){
    alpha <- o$optim$bestmem[1]
    H <- o$optim$bestmem[2]
    tau <- o$optim$bestmem[3]
    g<-function(x){trawl_LM(-x,alpha,H)}
    p<-function(x){sin(2*pi*x/tau)}
    tt <- (1:lags)
    #TSA::acf(x,lag.max=lags,main = "", ylab="ACF", xlab="Lags",drop.lag.0 = TRUE)
    my_tsa_acf(x, lag.max=lags, plot=TRUE)
    graphics::lines(tt, kernel_generic_trawl_acf(p, g, tt*Delta), lty =1,col=2, lwd=2)
  }

  return(list("alpha"=as.numeric(o$optim$bestmem[1]), "H"=o$optim$bestmem[2], "tau"=o$optim$bestmem[3]))

}


