#' Returns summary statistics
#' @title my_results
#' @param x Numeric vector of data values
#' @param sd Optional parameter giving the standard deviation of the normal
#' distribution used for computing the coverage probabilities. Default is 1.
#' @param digits Optional parameter specifying to how many digits the results
#' should be rounded. Default is 3.
#' @param reduced_percentiles Optional logical parameter specifying whether only
#'  the upper three levels of 90%, 95%, 99% should be included.
#'  Default is FALSE.
#' @details This function returns the sample mean, sample standard deviation
#' and the coverage probabilities at levels 75%, 80%, 85%, 90%, 95%, 99%
#' (or just 90%, 95%, 99% if reduced_percentiles=TRUE) compared to the
#' standard normal quantiles. Coverage probability is calculated as the
#' proportion of |x| values that fall within the specified quantile range.
#' @return A numeric vector containing:
#' \itemize{
#'   \item Sample mean
#'   \item Sample standard deviation
#'   \item Coverage probabilities at the specified levels
#' }
#' When reduced_percentiles=FALSE: returns 8 values(mean, sd, 6 coverage levels)
#' When reduced_percentiles=TRUE: returns 5 values (mean, sd, 3 coverage levels)
#' @export
#' @example man/examples/my_results.R
my_results <- function(x, sd = 1, digits = 3, reduced_percentiles = FALSE) {
  n <- base::length(x)

  basic_stats <- c(
    base::mean(x),
    sqrt(stats::var(x))
  )

  if (reduced_percentiles) {
    percentile_stats <- c(
      # 90%
      sum(abs(x) < stats::qnorm(0.95, mean = 0, sd = sd)) / n,
      # 95%
      sum(abs(x) < stats::qnorm(0.975, mean = 0, sd = sd)) / n,
      # 99%
      sum(abs(x) < stats::qnorm(0.995, mean = 0, sd = sd)) / n
    )
  } else {
    percentile_stats <- c(
      # 75%
      sum(abs(x) < stats::qnorm(0.875, mean = 0, sd = sd)) / n,
      # 80%
      sum(abs(x) < stats::qnorm(0.9, mean = 0, sd = sd)) / n,
      # 85%
      sum(abs(x) < stats::qnorm(0.925, mean = 0, sd = sd)) / n,
      # 90%
      sum(abs(x) < stats::qnorm(0.95, mean = 0, sd = sd)) / n,
      # 95%
      sum(abs(x) < stats::qnorm(0.975, mean = 0, sd = sd)) / n,
      # 99%
      sum(abs(x) < stats::qnorm(0.995, mean = 0, sd = sd)) / n
    )
  }

  value <- c(basic_stats, percentile_stats)
  return(base::round(value, digits))
}


#'Returns the mean squared error between two vectors
#'@title my_mse
#'@param x vector
#'@param y vector
#'@return Mean square error between the two vectors x and y
#'@example man/examples/my_mse.R
#'@export
my_mse <-function(x, y){
  length<-base::length(x)
  return(sum((x-y)^2)/length)
}



#'Returns the mean absolute error between two vectors
#'@title my_mse
#'@param x vector
#'@param y vector
#'@return Mean absolute error between the two vectors x and y
#'@export
#'@example man/examples/my_mae.R
my_mae <-function(x, y){
  length<-base::length(x)
  return(sum(abs(x-y))/length)
}
