#' Summarize Continuous and Categorical Variables with Grouping and P-Values
#'
#' `sum_stat_p` generates a descriptive summary table for both continuous and categorical variables,
#' stratified by a grouping variable. It automatically computes appropriate statistical tests
#' (Chi-square, Fisher's exact, t-test, Wilcoxon, ANOVA, or Kruskal–Wallis) based on variable type,
#' number of groups, and data distribution. Continuous variables can be summarized as mean (SD) or median (IQR),
#' and categorical variables as counts and percentages.
#'
#' The output is formatted as a `flextable` with footnotes indicating which summary statistics
#' were used and which statistical tests were applied.
#'
#' @param data A `data.frame` or `tibble` containing the variables to summarize.
#' @param by A string specifying the grouping variable for stratified summaries.
#' @param statistic Character string indicating how to summarize continuous variables:
#'   `"mean_sd"` (default) or `"med_iqr"`.
#' @param test_type Character string specifying the statistical test for group comparisons:
#'   `"auto"` (default, chooses test based on data), `"t.test"`, `"wilcox"`, `"anova"`,
#'   `"kruskal"`, `"chisq"`, or `"fisher"`.
#' @param paired Logical. If TRUE and only two groups exist, performs a paired t-test for continuous variables. Default is FALSE.
#' @importFrom tibble as_tibble
#' @importFrom stats t.test wilcox.test aov kruskal.test chisq.test fisher.test
#' @importFrom flextable flextable set_header_labels autofit bold add_footer_lines
#' @import tidyr
#'
#' @return A `flextable` object displaying a publication-ready summary table including:
#' - Counts and percentages for categorical variables
#' - Mean (SD) or median (IQR) for continuous variables
#' - P-values for group comparisons
#' - Footnotes describing which summary statistics and tests were used
#'
#' @export
#'
#' @examples
#' # Summary of iris dataset by species
#' sum_stat_p(iris, by = "Species", statistic = "mean_sd", test_type = "auto")
#'
#' # Summary of CO2 dataset by Type with paired t-test
#' sum_stat_p(CO2, by = "Type", statistic = "mean_sd", test_type = "t.test", paired = TRUE)
#'
#' # Summary using median and IQR
#' sum_stat_p(iris, by = "Species", statistic = "med_iqr", test_type = "kruskal")
sum_stat_p <- function(data, by, statistic = "mean_sd", test_type = "auto", paired = FALSE) {
  data <- as_tibble(data)
  is_cat <- function(x) is.factor(x) || is.character(x)

  summary_list <- list()
  used_tests <- c()

  for (colname in names(data)) {
    if (colname == by) next
    var <- data[[colname]]
    pval <- NA
    test_used <- ""

    # ----- P-VALUE SECTION -----
    try({
      if (is_cat(var)) {
        tbl <- table(data[[colname]], data[[by]])
        if (test_type == "fisher" || (test_type == "auto" && any(tbl < 5))) {
          pval <- fisher.test(tbl)$p.value
          test_used <- "Fisher's Exact"
        } else {
          pval <- chisq.test(tbl)$p.value
          test_used <- "Chi-square"
        }
      } else {
        g <- unique(data[[by]])
        if (length(g) == 2) {
          if (paired) {
            pval <- t.test(var ~ data[[by]], paired = TRUE)$p.value
            test_used <- "Paired t-test"
          } else if (test_type == "wilcox") {
            pval <- wilcox.test(var ~ data[[by]])$p.value
            test_used <- "Wilcoxon Rank-Sum"
          } else if (test_type == "t.test" || test_type == "auto") {
            pval <- t.test(var ~ data[[by]])$p.value
            test_used <- "Student's t-test"
          }
        } else {
          if (test_type == "kruskal" || test_type == "auto") {
            pval <- kruskal.test(var ~ data[[by]])$p.value
            test_used <- "Kruskal-Wallis"
          } else if (test_type == "anova") {
            pval <- summary(aov(var ~ data[[by]]))[[1]][["Pr(>F)"]][1]
            test_used <- "ANOVA"
          }
        }
      }
    }, silent = TRUE)

    used_tests <- c(used_tests, test_used)

    # ----- SUMMARY SECTION -----
    if (is_cat(var)) {
      tbl <- data %>%
        group_by(!!sym(colname), !!sym(by)) %>%
        summarise(n = n(), .groups = "drop") %>%
        complete(!!sym(colname), !!sym(by), fill = list(n = 0)) %>%
        group_by(!!sym(by)) %>%
        mutate(pct = round(n / sum(n) * 100, 2)) %>%  # <-- 2 decimals
        ungroup() %>%
        mutate(
          Variable = colname,
          Characteristic = as.character(!!sym(colname)),
          label = paste0(n, " (", pct, "%)")
        ) %>%
        select(Variable, Characteristic, !!sym(by), label) %>%
        pivot_wider(names_from = !!sym(by), values_from = label) %>%
        mutate(`p-value` = if_else(row_number() == 1, sprintf("%.2f", pval), ""))

    } else {
      tbl <- data %>%
        group_by(!!sym(by)) %>%
        summarise(
          val = case_when(
            statistic == "mean_sd" ~ paste0(
              round(mean(!!sym(colname), na.rm = TRUE), 2),
              " (",
              round(sd(!!sym(colname), na.rm = TRUE), 2),
              ")"
            ),
            TRUE ~ paste0(
              round(median(!!sym(colname), na.rm = TRUE), 2),
              " (",
              round(quantile(!!sym(colname), 0.25, na.rm = TRUE), 2),
              ", ",
              round(quantile(!!sym(colname), 0.75, na.rm = TRUE), 2),
              ")"
            )
          ),
          .groups = "drop"
        ) %>%
        mutate(Variable = colname) %>%
        pivot_wider(names_from = !!sym(by), values_from = val) %>%
        mutate(
          Characteristic = if_else(statistic == "mean_sd", "Mean (SD)", "Median (IQR)"),
          `p-value` = sprintf("%.2f", pval)  # <-- 2 decimals
        ) %>%
        select(Variable, Characteristic, everything())
    }

    summary_list[[colname]] <- tbl
  }

  summary_df <- bind_rows(summary_list) %>%
    group_by(Variable) %>%
    mutate(Variable = if_else(row_number() == 1, Variable, "")) %>%
    ungroup()

  # --- Build Footer ---
  stat_text <- case_when(
    statistic == "mean_sd" ~ "1 n (%); Mean (SD)",
    statistic == "med_iqr" ~ "1 n (%); Median (IQR)",
    TRUE ~ "1 n (%)"
  )

  test_text <- paste("P-values calculated using:", paste(unique(used_tests[used_tests != ""]), collapse = ", "))

  flextable(summary_df) %>%
    set_header_labels(
      Variable = "Variable",
      Characteristic = "Statistics",
      `p-value` = "p-value"
    ) %>%
    autofit() %>%
    bold(part = "header") %>%
    add_footer_lines(stat_text) %>%
    add_footer_lines(test_text)
}


