#' @title Function to solve for treatment effect size given target power for two-sample tests.
#' @description
#' Computes the treatment effect parameter \code{beta.trt} that achieves a specified power for a two-sided Wald-type
#' test under a normal approximation. The calculation assumes that the test statistic \eqn{Q} is asymptotically normal
#' with mean proportional to the treatment effect and variance \code{var}.
#' @param power Target power for the two-sided test, default is \code{0.8}.
#' @param power.c Positive numeric constant relating the mean of the test statistic under the alternative hypothesis to the treatment effect.
#' @param var Variance of the test statistic \eqn{Q} under the alternative hypothesis.
#'
#' @returns A numeric value giving the treatment effect parameter.
#' @export
find.beta.trt <- function(power = 0.8, power.c, var){

  # Under the alternative hypothesis, Q follows a normal distribution with mean (1- exp(beta.trt))*power.c and variance 'var'
  alpha = 0.05
  z_alpha <- qnorm(1 - alpha/2)  # for two-sided test, e.g., alpha = 0.05
  z_power <- qnorm(power)        # power you want, e.g., 0.80
  beta.trt <- log(1 - ((z_alpha + z_power)*sqrt(var)) / power.c)
  return(beta.trt)
}
