% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lua_module.R
\name{lua_module}
\alias{lua_module}
\alias{lua_import}
\title{Load Lua modules}
\usage{
lua_module(filename = NULL, package = NULL)

lua_import(module, name, argcode)
}
\arguments{
\item{filename}{Name of file from which to load the module. If this is a
character vector, the elements are concatenated together with
\code{\link[=file.path]{file.path()}}.}

\item{package}{If non-\code{NULL}, the file will be sought within this package.}

\item{module}{Module previously loaded with \code{\link[=lua_module]{lua_module()}}.}

\item{name}{Name of the function to import (character string).}

\item{argcode}{How to wrap R arguments for the Lua function; see
documentation for \code{\link[=lua_func]{lua_func()}}.}
}
\value{
\code{\link[=lua_module]{lua_module()}} returns an environment with class \code{"luajr_module"}.
}
\description{
\code{\link[=lua_module]{lua_module()}} can be used in an R project or package to declare a Lua
module in an external file. You can then use \code{\link[=lua_import]{lua_import()}} to access the
functions within the module, or provide access to those functions to your
package users. The object returned by \code{\link[=lua_module]{lua_module()}} can also be used to
set and get other (non-function) values stored in the Lua module table.
}
\section{Typical usage}{


\verb{# To load a Lua module containing myfunc(x,y)}

\code{mymod <- lua_module("Lua/mymodule.lua", package = "mypackage")}

\code{func <- function(x, y) lua_import(mymod, "myfunc", "s")}
}

\section{Module files}{


Module files should have the file extension \code{.lua} and be placed somewhere
in your project directory. If you are writing a package, the best practice
is probably to place these in the subdirectory \code{inst/Lua} of your package.

The module file itself should follow standard practice for
\href{http://lua-users.org/wiki/ModulesTutorial}{Lua modules}. In other words,
the module file should return a Lua table containing the module's functions.
A relatively minimal example would be:

\if{html}{\out{<div class="sourceCode Lua">}}\preformatted{local mymodule = \{\}
mymodule.fave_name = "Nick"

function mymodule.greet(name)
    print("Hello, " .. name .. "!")
    if name == mymodule.fave_name then
        print("Incidentally, that's a great name. Nice one.")
    end
end

return mymodule
}\if{html}{\out{</div>}}
}

\section{Loading the module}{


Before you import functions from your module, you need to create a module
object using \code{\link[=lua_module]{lua_module()}}. Supply the file name as the \code{filename} argument
to \code{\link[=lua_module]{lua_module()}}. If you are developing a package, also supply your package
name as the \code{package} argument. If \code{package} is \code{NULL}, \code{\link[=lua_module]{lua_module()}} will
look for the file relative to the current working directory. If \code{package} is
non-\code{NULL}, \code{\link[=lua_module]{lua_module()}} will look for the file relative to the
installed package directory (using \code{\link[=system.file]{system.file()}}). So, if you are
developing a package and you have put your module file in
\code{inst/Lua/mymodule.lua} as recommended above, supply \code{"Lua/mymodule.lua"}
as the filename.

The module returned by \code{\link[=lua_module]{lua_module()}} is not actually loaded until the
first time that you import a function from the module. If you want the
module to be loaded into a specific \link[=lua_open]{Lua state} in your R
project, then assign that state to the module's state right after declaring
it:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{mymod <- lua_module("path/to/file.lua", package = "mypackage")
mymod$L <- my_state
}\if{html}{\out{</div>}}

If you are creating a package and you want to load your module into a
specific Lua state, you will need to create that state and assign it to
\code{module$L} after the package is loaded, probably by using \code{\link[=.onLoad]{.onLoad()}}.
}

\section{Importing functions}{


To import a function from a module, declare it like this:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{myfunc <- function(x, y) lua_import(mymod, "funcname", "s")
}\if{html}{\out{</div>}}

where \code{mymod} is the previously-declared module object, \code{"funcname"} is the
function name within the Lua module, and \code{"s"} is whatever
\link[=lua_func]{arg code} you want to use. Note that \code{lua_import()} must be used
as the only statement in your function body and you should \strong{not} enclose
it in braces (\code{{}}). The arguments of \code{myfunc} will be passed to the
imported function in the same order as they are declared in the function
signature. You can give default values to the function arguments.

With the example above, the first time you call \code{myfunc()}, it will make
sure the module is properly loaded and then call the Lua function. It will
also overwrite the existing body of \code{myfunc()} with a direct call to the
Lua function so that subsequent calls to \code{myfunc()} execute as quickly
as possible.

In some cases, you may want to do some processing or checking of function
arguments in R before calling the Lua function. You can do that with a
"two-step" process like this:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{greet0 <- function(name) lua_import(mymod, "greet", "s")
greet <- function(name) \{
    if (!is.character(name)) \{
        stop("greet expects a character string.")
    \}
    greet0(name)
\}
}\if{html}{\out{</div>}}

In a package, you can document and export a function that uses \code{\link[=lua_import]{lua_import()}}
just like any other function.
}

\section{Setting and getting}{


Lua modules can contain more than just functions; they can also hold other
values, as shown in the example module above (under "Module files"). In this
example, the module also contains a string called \code{fave_name} which alters
the behaviour of the \code{greet} function.

You can get a value from a module by using e.g. \code{module["fave_name"]} and
set it using e.g. \code{module["fave_name"] <- "Janet"}. You must use single
brackets \verb{[]} and not double brackets \verb{[[]]} or the dollar sign \code{$} for
this, and you cannot change a function at the top level of the module. If
your module contains a table \code{x} which contains a value \code{y}, you can get or
set \code{y} by using multiple indices, e.g. \code{module["x", "y"]} or
\code{module["x", "y"] <- 1}. Using empty brackets, e.g. \code{module[]}, will return
all the contents of the module, but you cannot set the entire contents of
the module with e.g. \code{module[] = foo}.

By default, when setting a module value using \code{module[i] <- value}, the
value is passed to Lua "by simplify" (e.g. with \link[=lua_func]{arg code}
\code{"s"}). You can change this behaviour with the \code{as} argument. For example,
\code{module[i, as = "a"] <- 2} will set element \code{i} of the module to a Lua
table \code{{2}} instead of the plain value \code{2}.
}

\examples{
module <- lua_module(c("Lua", "example.lua"), package = "luajr")
greet <- function(name) lua_import(module, "greet", "s")
greet("Janet")
greet("Nick")
}
