% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sigmoid_imax.R
\name{sigmoid_imax}
\alias{sigmoid_imax}
\title{Sigmoid Imax (Inhibitory Hill Model) for Dose-Response Analysis}
\arguments{
\item{data}{A data frame containing dose-response experimental data.}

\item{dose_col}{Character string specifying the column name for dose or concentration.}

\item{response_col}{Character string specifying the column name for measured response.}

\item{group_col}{Optional character string specifying a column for grouping.}

\item{log_dose}{Logical; if TRUE, dose values are log10-transformed for plotting
(model fitting uses original dose values).}

\item{plot}{Logical; if TRUE, generates a dose-response plot with fitted Hill curves.}

\item{annotate}{Logical; if TRUE, annotates the plot with model parameters and fit metrics
(only if <=1 group).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with E0, Imax, IC50, Hill coefficient (\code{n}),
        RMSE, AIC, and BIC values for each group.}
  \item{\code{data}}{The processed dataset used for model fitting and plotting.}
}
}
\description{
Fits pharmacodynamic dose-response data to the inhibitory Hill
(Sigmoid Imax) model using nonlinear least squares regression.

The inhibitory Hill model describes decreasing pharmacological
effects with increasing dose (or concentration) according to:

\deqn{
  E = E_0 - \frac{I_{max} \cdot D^n}{IC_{50}^n + D^n}
}

An equivalent parameterization is:

\deqn{
  E = E_{min} + \frac{(E_0 - E_{min}) \cdot IC_{50}^n}{IC_{50}^n + D^n}
}

where \eqn{E_0} is the baseline response, \eqn{I_{max}} is the maximum
inhibitory effect, \eqn{IC_{50}} is the dose producing 50% of maximal
inhibition, and \eqn{n} is the Hill coefficient.

This model is appropriate when the observed response decreases
monotonically with increasing dose.
}
\examples{
# Example I: Single inhibitory dose-response dataset
df <- data.frame(
  dose = c(0.1, 0.3, 1, 3, 10, 30, 100),
  response = c(95, 90, 75, 45, 20, 8, 3)
)
sigmoid_imax(
  data = df,
  dose_col = "dose",
  response_col = "response"
)

# Example II: Two treatment groups (e.g., two inhibitors or conditions)
df2 <- data.frame(
  dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 2),
  response = c(
    92, 85, 65, 40, 18, 7,    # Group A (stronger inhibitor)
    95, 88, 72, 50, 30, 15    # Group B (weaker inhibitor)
  ),
  treatment = rep(c("Group A", "Group B"), each = 6)
)
sigmoid_imax(
  data = df2,
  dose_col = "dose",
  response_col = "response",
  group_col = "treatment",
  log_dose = TRUE
)

# Example III: Multiple subjects (population-style inhibitory pharmacodynamics)
df_subjects <- data.frame(
  dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 5),
  response = c(
    94, 86, 68, 42, 20, 9,    # Subject 1
    96, 88, 70, 45, 22, 10,   # Subject 2
    93, 84, 65, 40, 18, 8,    # Subject 3
    95, 87, 69, 44, 21, 9,    # Subject 4
    97, 89, 72, 48, 25, 12    # Subject 5
  ),
  subject = rep(paste0("S", 1:5), each = 6)
)
sigmoid_imax(
  data = df_subjects,
  dose_col = "dose",
  response_col = "response",
  group_col = "subject",
  log_dose = TRUE
)
}
\references{
Holford, N. H. G. & Sheiner, L. B. (1981) <doi:10.2165/00003088-198106060-00002>
Understanding the dose-effect relationship. Clinical Pharmacokinetics, 6(6), 429–453.
}
\author{
Paul Angelo C. Manlapaz
}
