






use core::cell::UnsafeCell;
use core::fmt;
use core::marker::PhantomData;
use core::mem;
use core::ops::{Deref, DerefMut};

#[cfg(feature = "arc_lock")]
use alloc::sync::Arc;
#[cfg(feature = "arc_lock")]
use core::mem::ManuallyDrop;
#[cfg(feature = "arc_lock")]
use core::ptr;

#[cfg(feature = "owning_ref")]
use owning_ref::StableAddress;

#[cfg(feature = "serde")]
use serde::{Deserialize, Deserializer, Serialize, Serializer};






/// # Safety





pub unsafe trait RawRwLock {
    
    
    
    #[allow(clippy::declare_interior_mutable_const)]
    const INIT: Self;

    
    
    type GuardMarker;

    
    fn lock_shared(&self);

    
    fn try_lock_shared(&self) -> bool;

    
    
    /// # Safety
    
    
    unsafe fn unlock_shared(&self);

    
    fn lock_exclusive(&self);

    
    fn try_lock_exclusive(&self) -> bool;

    
    
    /// # Safety
    
    
    unsafe fn unlock_exclusive(&self);

    
    #[inline]
    fn is_locked(&self) -> bool {
        let acquired_lock = self.try_lock_exclusive();
        if acquired_lock {
            
            unsafe {
                self.unlock_exclusive();
            }
        }
        !acquired_lock
    }

    
    fn is_locked_exclusive(&self) -> bool {
        let acquired_lock = self.try_lock_shared();
        if acquired_lock {
            
            unsafe {
                self.unlock_shared();
            }
        }
        !acquired_lock
    }
}







pub unsafe trait RawRwLockFair: RawRwLock {
    
    
    /// # Safety
    
    
    unsafe fn unlock_shared_fair(&self);

    
    
    /// # Safety
    
    
    unsafe fn unlock_exclusive_fair(&self);

    
    
    
    
    
    
    /// # Safety
    
    
    unsafe fn bump_shared(&self) {
        self.unlock_shared_fair();
        self.lock_shared();
    }

    
    
    
    
    
    
    /// # Safety
    
    
    unsafe fn bump_exclusive(&self) {
        self.unlock_exclusive_fair();
        self.lock_exclusive();
    }
}



pub unsafe trait RawRwLockDowngrade: RawRwLock {
    
    
    
    /// # Safety
    
    
    unsafe fn downgrade(&self);
}





pub unsafe trait RawRwLockTimed: RawRwLock {
    
    type Duration;

    
    type Instant;

    
    fn try_lock_shared_for(&self, timeout: Self::Duration) -> bool;

    
    fn try_lock_shared_until(&self, timeout: Self::Instant) -> bool;

    
    fn try_lock_exclusive_for(&self, timeout: Self::Duration) -> bool;

    
    fn try_lock_exclusive_until(&self, timeout: Self::Instant) -> bool;
}








pub unsafe trait RawRwLockRecursive: RawRwLock {
    
    fn lock_shared_recursive(&self);

    
    fn try_lock_shared_recursive(&self) -> bool;
}


pub unsafe trait RawRwLockRecursiveTimed: RawRwLockRecursive + RawRwLockTimed {
    
    
    fn try_lock_shared_recursive_for(&self, timeout: Self::Duration) -> bool;

    
    
    fn try_lock_shared_recursive_until(&self, timeout: Self::Instant) -> bool;
}







pub unsafe trait RawRwLockUpgrade: RawRwLock {
    
    fn lock_upgradable(&self);

    
    fn try_lock_upgradable(&self) -> bool;

    
    
    /// # Safety
    
    
    unsafe fn unlock_upgradable(&self);

    
    
    /// # Safety
    
    
    unsafe fn upgrade(&self);

    
    
    
    /// # Safety
    
    
    unsafe fn try_upgrade(&self) -> bool;
}



pub unsafe trait RawRwLockUpgradeFair: RawRwLockUpgrade + RawRwLockFair {
    
    
    /// # Safety
    
    
    unsafe fn unlock_upgradable_fair(&self);

    
    
    
    
    
    
    /// # Safety
    
    
    unsafe fn bump_upgradable(&self) {
        self.unlock_upgradable_fair();
        self.lock_upgradable();
    }
}



pub unsafe trait RawRwLockUpgradeDowngrade: RawRwLockUpgrade + RawRwLockDowngrade {
    
    
    /// # Safety
    
    
    unsafe fn downgrade_upgradable(&self);

    
    
    /// # Safety
    
    
    unsafe fn downgrade_to_upgradable(&self);
}



pub unsafe trait RawRwLockUpgradeTimed: RawRwLockUpgrade + RawRwLockTimed {
    
    fn try_lock_upgradable_for(&self, timeout: Self::Duration) -> bool;

    
    fn try_lock_upgradable_until(&self, timeout: Self::Instant) -> bool;

    
    
    
    /// # Safety
    
    
    unsafe fn try_upgrade_for(&self, timeout: Self::Duration) -> bool;

    
    
    
    /// # Safety
    
    
    unsafe fn try_upgrade_until(&self, timeout: Self::Instant) -> bool;
}













pub struct RwLock<R, T: ?Sized> {
    raw: R,
    data: UnsafeCell<T>,
}


#[cfg(feature = "serde")]
impl<R, T> Serialize for RwLock<R, T>
where
    R: RawRwLock,
    T: Serialize + ?Sized,
{
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        self.read().serialize(serializer)
    }
}

#[cfg(feature = "serde")]
impl<'de, R, T> Deserialize<'de> for RwLock<R, T>
where
    R: RawRwLock,
    T: Deserialize<'de> + ?Sized,
{
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        Deserialize::deserialize(deserializer).map(RwLock::new)
    }
}

unsafe impl<R: RawRwLock + Send, T: ?Sized + Send> Send for RwLock<R, T> {}
unsafe impl<R: RawRwLock + Sync, T: ?Sized + Send + Sync> Sync for RwLock<R, T> {}

impl<R: RawRwLock, T> RwLock<R, T> {
    
    #[cfg(has_const_fn_trait_bound)]
    #[inline]
    pub const fn new(val: T) -> RwLock<R, T> {
        RwLock {
            data: UnsafeCell::new(val),
            raw: R::INIT,
        }
    }

    
    #[cfg(not(has_const_fn_trait_bound))]
    #[inline]
    pub fn new(val: T) -> RwLock<R, T> {
        RwLock {
            data: UnsafeCell::new(val),
            raw: R::INIT,
        }
    }

    
    #[inline]
    #[allow(unused_unsafe)]
    pub fn into_inner(self) -> T {
        unsafe { self.data.into_inner() }
    }
}

impl<R, T> RwLock<R, T> {
    
    
    #[inline]
    pub const fn from_raw(raw_rwlock: R, val: T) -> RwLock<R, T> {
        RwLock {
            data: UnsafeCell::new(val),
            raw: raw_rwlock,
        }
    }

    
    
    
    
    
    
    
    #[inline]
    pub const fn const_new(raw_rwlock: R, val: T) -> RwLock<R, T> {
        Self::from_raw(raw_rwlock, val)
    }
}

impl<R: RawRwLock, T: ?Sized> RwLock<R, T> {
    
    
    /// # Safety
    
    
    
    
    
    
    #[inline]
    pub unsafe fn make_read_guard_unchecked(&self) -> RwLockReadGuard<'_, R, T> {
        RwLockReadGuard {
            rwlock: self,
            marker: PhantomData,
        }
    }

    
    
    /// # Safety
    
    
    
    
    
    #[inline]
    pub unsafe fn make_write_guard_unchecked(&self) -> RwLockWriteGuard<'_, R, T> {
        RwLockWriteGuard {
            rwlock: self,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn read(&self) -> RwLockReadGuard<'_, R, T> {
        self.raw.lock_shared();
        
        unsafe { self.make_read_guard_unchecked() }
    }

    
    
    
    
    
    
    
    #[inline]
    pub fn try_read(&self) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared() {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    
    
    #[inline]
    pub fn write(&self) -> RwLockWriteGuard<'_, R, T> {
        self.raw.lock_exclusive();
        
        unsafe { self.make_write_guard_unchecked() }
    }

    
    
    
    
    
    
    
    #[inline]
    pub fn try_write(&self) -> Option<RwLockWriteGuard<'_, R, T>> {
        if self.raw.try_lock_exclusive() {
            
            Some(unsafe { self.make_write_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[inline]
    pub fn get_mut(&mut self) -> &mut T {
        unsafe { &mut *self.data.get() }
    }

    
    #[inline]
    pub fn is_locked(&self) -> bool {
        self.raw.is_locked()
    }

    
    #[inline]
    pub fn is_locked_exclusive(&self) -> bool {
        self.raw.is_locked_exclusive()
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    #[inline]
    pub unsafe fn force_unlock_read(&self) {
        self.raw.unlock_shared();
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    #[inline]
    pub unsafe fn force_unlock_write(&self) {
        self.raw.unlock_exclusive();
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    pub unsafe fn raw(&self) -> &R {
        &self.raw
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    
    #[inline]
    pub fn data_ptr(&self) -> *mut T {
        self.data.get()
    }

    
    
    /// # Safety
    
    
    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub unsafe fn make_arc_read_guard_unchecked(self: &Arc<Self>) -> ArcRwLockReadGuard<R, T> {
        ArcRwLockReadGuard {
            rwlock: self.clone(),
            marker: PhantomData,
        }
    }

    
    
    /// # Safety
    
    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub unsafe fn make_arc_write_guard_unchecked(self: &Arc<Self>) -> ArcRwLockWriteGuard<R, T> {
        ArcRwLockWriteGuard {
            rwlock: self.clone(),
            marker: PhantomData,
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn read_arc(self: &Arc<Self>) -> ArcRwLockReadGuard<R, T> {
        self.raw.lock_shared();
        
        unsafe { self.make_arc_read_guard_unchecked() }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_arc(self: &Arc<Self>) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared() {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn write_arc(self: &Arc<Self>) -> ArcRwLockWriteGuard<R, T> {
        self.raw.lock_exclusive();
        
        unsafe { self.make_arc_write_guard_unchecked() }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_write_arc(self: &Arc<Self>) -> Option<ArcRwLockWriteGuard<R, T>> {
        if self.raw.try_lock_exclusive() {
            
            Some(unsafe { self.make_arc_write_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLockFair, T: ?Sized> RwLock<R, T> {
    
    
    
    
    
    
    /// # Safety
    
    
    
    
    #[inline]
    pub unsafe fn force_unlock_read_fair(&self) {
        self.raw.unlock_shared_fair();
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    #[inline]
    pub unsafe fn force_unlock_write_fair(&self) {
        self.raw.unlock_exclusive_fair();
    }
}

impl<R: RawRwLockTimed, T: ?Sized> RwLock<R, T> {
    
    
    
    
    
    
    #[inline]
    pub fn try_read_for(&self, timeout: R::Duration) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared_for(timeout) {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    #[inline]
    pub fn try_read_until(&self, timeout: R::Instant) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared_until(timeout) {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    #[inline]
    pub fn try_write_for(&self, timeout: R::Duration) -> Option<RwLockWriteGuard<'_, R, T>> {
        if self.raw.try_lock_exclusive_for(timeout) {
            
            Some(unsafe { self.make_write_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    #[inline]
    pub fn try_write_until(&self, timeout: R::Instant) -> Option<RwLockWriteGuard<'_, R, T>> {
        if self.raw.try_lock_exclusive_until(timeout) {
            
            Some(unsafe { self.make_write_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_arc_for(
        self: &Arc<Self>,
        timeout: R::Duration,
    ) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared_for(timeout) {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_arc_until(
        self: &Arc<Self>,
        timeout: R::Instant,
    ) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared_until(timeout) {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_write_arc_for(
        self: &Arc<Self>,
        timeout: R::Duration,
    ) -> Option<ArcRwLockWriteGuard<R, T>> {
        if self.raw.try_lock_exclusive_for(timeout) {
            
            Some(unsafe { self.make_arc_write_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_write_arc_until(
        self: &Arc<Self>,
        timeout: R::Instant,
    ) -> Option<ArcRwLockWriteGuard<R, T>> {
        if self.raw.try_lock_exclusive_until(timeout) {
            
            Some(unsafe { self.make_arc_write_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLockRecursive, T: ?Sized> RwLock<R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn read_recursive(&self) -> RwLockReadGuard<'_, R, T> {
        self.raw.lock_shared_recursive();
        
        unsafe { self.make_read_guard_unchecked() }
    }

    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_read_recursive(&self) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared_recursive() {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn read_arc_recursive(self: &Arc<Self>) -> ArcRwLockReadGuard<R, T> {
        self.raw.lock_shared_recursive();
        
        unsafe { self.make_arc_read_guard_unchecked() }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_recursive_arc(self: &Arc<Self>) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared_recursive() {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLockRecursiveTimed, T: ?Sized> RwLock<R, T> {
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_read_recursive_for(
        &self,
        timeout: R::Duration,
    ) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared_recursive_for(timeout) {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    #[inline]
    pub fn try_read_recursive_until(
        &self,
        timeout: R::Instant,
    ) -> Option<RwLockReadGuard<'_, R, T>> {
        if self.raw.try_lock_shared_recursive_until(timeout) {
            
            Some(unsafe { self.make_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_arc_recursive_for(
        self: &Arc<Self>,
        timeout: R::Duration,
    ) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared_recursive_for(timeout) {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_read_arc_recursive_until(
        self: &Arc<Self>,
        timeout: R::Instant,
    ) -> Option<ArcRwLockReadGuard<R, T>> {
        if self.raw.try_lock_shared_recursive_until(timeout) {
            
            Some(unsafe { self.make_arc_read_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLockUpgrade, T: ?Sized> RwLock<R, T> {
    
    
    /// # Safety
    
    
    
    
    
    
    #[inline]
    pub unsafe fn make_upgradable_guard_unchecked(&self) -> RwLockUpgradableReadGuard<'_, R, T> {
        RwLockUpgradableReadGuard {
            rwlock: self,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn upgradable_read(&self) -> RwLockUpgradableReadGuard<'_, R, T> {
        self.raw.lock_upgradable();
        
        unsafe { self.make_upgradable_guard_unchecked() }
    }

    
    
    
    
    
    
    
    #[inline]
    pub fn try_upgradable_read(&self) -> Option<RwLockUpgradableReadGuard<'_, R, T>> {
        if self.raw.try_lock_upgradable() {
            
            Some(unsafe { self.make_upgradable_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    /// # Safety
    
    
    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub unsafe fn make_upgradable_arc_guard_unchecked(
        self: &Arc<Self>,
    ) -> ArcRwLockUpgradableReadGuard<R, T> {
        ArcRwLockUpgradableReadGuard {
            rwlock: self.clone(),
            marker: PhantomData,
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn upgradable_read_arc(self: &Arc<Self>) -> ArcRwLockUpgradableReadGuard<R, T> {
        self.raw.lock_upgradable();
        
        unsafe { self.make_upgradable_arc_guard_unchecked() }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_upgradable_read_arc(self: &Arc<Self>) -> Option<ArcRwLockUpgradableReadGuard<R, T>> {
        if self.raw.try_lock_upgradable() {
            
            Some(unsafe { self.make_upgradable_arc_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLockUpgradeTimed, T: ?Sized> RwLock<R, T> {
    
    
    
    
    
    
    #[inline]
    pub fn try_upgradable_read_for(
        &self,
        timeout: R::Duration,
    ) -> Option<RwLockUpgradableReadGuard<'_, R, T>> {
        if self.raw.try_lock_upgradable_for(timeout) {
            
            Some(unsafe { self.make_upgradable_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    
    
    #[inline]
    pub fn try_upgradable_read_until(
        &self,
        timeout: R::Instant,
    ) -> Option<RwLockUpgradableReadGuard<'_, R, T>> {
        if self.raw.try_lock_upgradable_until(timeout) {
            
            Some(unsafe { self.make_upgradable_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_upgradable_read_arc_for(
        self: &Arc<Self>,
        timeout: R::Duration,
    ) -> Option<ArcRwLockUpgradableReadGuard<R, T>> {
        if self.raw.try_lock_upgradable_for(timeout) {
            
            Some(unsafe { self.make_upgradable_arc_guard_unchecked() })
        } else {
            None
        }
    }

    
    
    
    
    #[cfg(feature = "arc_lock")]
    #[inline]
    pub fn try_upgradable_read_arc_until(
        self: &Arc<Self>,
        timeout: R::Instant,
    ) -> Option<ArcRwLockUpgradableReadGuard<R, T>> {
        if self.raw.try_lock_upgradable_until(timeout) {
            
            Some(unsafe { self.make_upgradable_arc_guard_unchecked() })
        } else {
            None
        }
    }
}

impl<R: RawRwLock, T: ?Sized + Default> Default for RwLock<R, T> {
    #[inline]
    fn default() -> RwLock<R, T> {
        RwLock::new(Default::default())
    }
}

impl<R: RawRwLock, T> From<T> for RwLock<R, T> {
    #[inline]
    fn from(t: T) -> RwLock<R, T> {
        RwLock::new(t)
    }
}

impl<R: RawRwLock, T: ?Sized + fmt::Debug> fmt::Debug for RwLock<R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut d = f.debug_struct("RwLock");
        match self.try_read() {
            Some(guard) => d.field("data", &&*guard),
            None => {
                
                d.field("data", &format_args!("<locked>"))
            }
        };
        d.finish()
    }
}



#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct RwLockReadGuard<'a, R: RawRwLock, T: ?Sized> {
    rwlock: &'a RwLock<R, T>,
    marker: PhantomData<(&'a T, R::GuardMarker)>,
}

unsafe impl<R: RawRwLock + Sync, T: Sync + ?Sized> Sync for RwLockReadGuard<'_, R, T> {}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> RwLockReadGuard<'a, R, T> {
    
    pub fn rwlock(s: &Self) -> &'a RwLock<R, T> {
        s.rwlock
    }

    
    
    
    
    
    
    
    
    #[inline]
    pub fn map<U: ?Sized, F>(s: Self, f: F) -> MappedRwLockReadGuard<'a, R, U>
    where
        F: FnOnce(&T) -> &U,
    {
        let raw = &s.rwlock.raw;
        let data = f(unsafe { &*s.rwlock.data.get() });
        mem::forget(s);
        MappedRwLockReadGuard {
            raw,
            data,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_map<U: ?Sized, F>(s: Self, f: F) -> Result<MappedRwLockReadGuard<'a, R, U>, Self>
    where
        F: FnOnce(&T) -> Option<&U>,
    {
        let raw = &s.rwlock.raw;
        let data = match f(unsafe { &*s.rwlock.data.get() }) {
            Some(data) => data,
            None => return Err(s),
        };
        mem::forget(s);
        Ok(MappedRwLockReadGuard {
            raw,
            data,
            marker: PhantomData,
        })
    }

    
    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_shared();
        }
        defer!(s.rwlock.raw.lock_shared());
        f()
    }
}

impl<'a, R: RawRwLockFair + 'a, T: ?Sized + 'a> RwLockReadGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_shared_fair();
        }
        mem::forget(s);
    }

    
    
    
    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_shared_fair();
        }
        defer!(s.rwlock.raw.lock_shared());
        f()
    }

    
    
    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_shared();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Deref for RwLockReadGuard<'a, R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Drop for RwLockReadGuard<'a, R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_shared();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Debug + ?Sized + 'a> fmt::Debug for RwLockReadGuard<'a, R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Display + ?Sized + 'a> fmt::Display
    for RwLockReadGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

#[cfg(feature = "owning_ref")]
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> StableAddress for RwLockReadGuard<'a, R, T> {}





#[cfg(feature = "arc_lock")]
#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct ArcRwLockReadGuard<R: RawRwLock, T: ?Sized> {
    rwlock: Arc<RwLock<R, T>>,
    marker: PhantomData<R::GuardMarker>,
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> ArcRwLockReadGuard<R, T> {
    
    pub fn rwlock(s: &Self) -> &Arc<RwLock<R, T>> {
        &s.rwlock
    }

    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_shared();
        }
        defer!(s.rwlock.raw.lock_shared());
        f()
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockFair, T: ?Sized> ArcRwLockReadGuard<R, T> {
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_shared_fair();
        }

        
        let mut s = ManuallyDrop::new(s);
        unsafe { ptr::drop_in_place(&mut s.rwlock) };
    }

    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_shared_fair();
        }
        defer!(s.rwlock.raw.lock_shared());
        f()
    }

    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_shared();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> Deref for ArcRwLockReadGuard<R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> Drop for ArcRwLockReadGuard<R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_shared();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: fmt::Debug + ?Sized> fmt::Debug for ArcRwLockReadGuard<R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: fmt::Display + ?Sized> fmt::Display for ArcRwLockReadGuard<R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}



#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct RwLockWriteGuard<'a, R: RawRwLock, T: ?Sized> {
    rwlock: &'a RwLock<R, T>,
    marker: PhantomData<(&'a mut T, R::GuardMarker)>,
}

unsafe impl<R: RawRwLock + Sync, T: Sync + ?Sized> Sync for RwLockWriteGuard<'_, R, T> {}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> RwLockWriteGuard<'a, R, T> {
    
    pub fn rwlock(s: &Self) -> &'a RwLock<R, T> {
        s.rwlock
    }

    
    
    
    
    
    
    
    
    #[inline]
    pub fn map<U: ?Sized, F>(s: Self, f: F) -> MappedRwLockWriteGuard<'a, R, U>
    where
        F: FnOnce(&mut T) -> &mut U,
    {
        let raw = &s.rwlock.raw;
        let data = f(unsafe { &mut *s.rwlock.data.get() });
        mem::forget(s);
        MappedRwLockWriteGuard {
            raw,
            data,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_map<U: ?Sized, F>(s: Self, f: F) -> Result<MappedRwLockWriteGuard<'a, R, U>, Self>
    where
        F: FnOnce(&mut T) -> Option<&mut U>,
    {
        let raw = &s.rwlock.raw;
        let data = match f(unsafe { &mut *s.rwlock.data.get() }) {
            Some(data) => data,
            None => return Err(s),
        };
        mem::forget(s);
        Ok(MappedRwLockWriteGuard {
            raw,
            data,
            marker: PhantomData,
        })
    }

    
    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive();
        }
        defer!(s.rwlock.raw.lock_exclusive());
        f()
    }
}

impl<'a, R: RawRwLockDowngrade + 'a, T: ?Sized + 'a> RwLockWriteGuard<'a, R, T> {
    
    
    
    
    
    
    pub fn downgrade(s: Self) -> RwLockReadGuard<'a, R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade();
        }
        let rwlock = s.rwlock;
        mem::forget(s);
        RwLockReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }
}

impl<'a, R: RawRwLockUpgradeDowngrade + 'a, T: ?Sized + 'a> RwLockWriteGuard<'a, R, T> {
    
    
    
    
    
    
    pub fn downgrade_to_upgradable(s: Self) -> RwLockUpgradableReadGuard<'a, R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade_to_upgradable();
        }
        let rwlock = s.rwlock;
        mem::forget(s);
        RwLockUpgradableReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }
}

impl<'a, R: RawRwLockFair + 'a, T: ?Sized + 'a> RwLockWriteGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive_fair();
        }
        mem::forget(s);
    }

    
    
    
    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive_fair();
        }
        defer!(s.rwlock.raw.lock_exclusive());
        f()
    }

    
    
    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_exclusive();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Deref for RwLockWriteGuard<'a, R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> DerefMut for RwLockWriteGuard<'a, R, T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut T {
        unsafe { &mut *self.rwlock.data.get() }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Drop for RwLockWriteGuard<'a, R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_exclusive();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Debug + ?Sized + 'a> fmt::Debug for RwLockWriteGuard<'a, R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Display + ?Sized + 'a> fmt::Display
    for RwLockWriteGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

#[cfg(feature = "owning_ref")]
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> StableAddress for RwLockWriteGuard<'a, R, T> {}




#[cfg(feature = "arc_lock")]
#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct ArcRwLockWriteGuard<R: RawRwLock, T: ?Sized> {
    rwlock: Arc<RwLock<R, T>>,
    marker: PhantomData<R::GuardMarker>,
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> ArcRwLockWriteGuard<R, T> {
    
    pub fn rwlock(s: &Self) -> &Arc<RwLock<R, T>> {
        &s.rwlock
    }

    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive();
        }
        defer!(s.rwlock.raw.lock_exclusive());
        f()
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockDowngrade, T: ?Sized> ArcRwLockWriteGuard<R, T> {
    
    
    
    
    pub fn downgrade(s: Self) -> ArcRwLockReadGuard<R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade();
        }

        
        let s = ManuallyDrop::new(s);
        let rwlock = unsafe { ptr::read(&s.rwlock) };

        ArcRwLockReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgradeDowngrade, T: ?Sized> ArcRwLockWriteGuard<R, T> {
    
    
    
    
    pub fn downgrade_to_upgradable(s: Self) -> ArcRwLockUpgradableReadGuard<R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade_to_upgradable();
        }

        
        let s = ManuallyDrop::new(s);
        let rwlock = unsafe { ptr::read(&s.rwlock) };

        ArcRwLockUpgradableReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockFair, T: ?Sized> ArcRwLockWriteGuard<R, T> {
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive_fair();
        }

        
        let mut s = ManuallyDrop::new(s);
        unsafe { ptr::drop_in_place(&mut s.rwlock) };
    }

    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_exclusive_fair();
        }
        defer!(s.rwlock.raw.lock_exclusive());
        f()
    }

    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_exclusive();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> Deref for ArcRwLockWriteGuard<R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> DerefMut for ArcRwLockWriteGuard<R, T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut T {
        unsafe { &mut *self.rwlock.data.get() }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: ?Sized> Drop for ArcRwLockWriteGuard<R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_exclusive();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: fmt::Debug + ?Sized> fmt::Debug for ArcRwLockWriteGuard<R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLock, T: fmt::Display + ?Sized> fmt::Display for ArcRwLockWriteGuard<R, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}



#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct RwLockUpgradableReadGuard<'a, R: RawRwLockUpgrade, T: ?Sized> {
    rwlock: &'a RwLock<R, T>,
    marker: PhantomData<(&'a T, R::GuardMarker)>,
}

unsafe impl<'a, R: RawRwLockUpgrade + 'a, T: ?Sized + Sync + 'a> Sync
    for RwLockUpgradableReadGuard<'a, R, T>
{
}

impl<'a, R: RawRwLockUpgrade + 'a, T: ?Sized + 'a> RwLockUpgradableReadGuard<'a, R, T> {
    
    pub fn rwlock(s: &Self) -> &'a RwLock<R, T> {
        s.rwlock
    }

    
    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable();
        }
        defer!(s.rwlock.raw.lock_upgradable());
        f()
    }

    
    
    pub fn upgrade(s: Self) -> RwLockWriteGuard<'a, R, T> {
        
        unsafe {
            s.rwlock.raw.upgrade();
        }
        let rwlock = s.rwlock;
        mem::forget(s);
        RwLockWriteGuard {
            rwlock,
            marker: PhantomData,
        }
    }

    
    
    
    pub fn try_upgrade(s: Self) -> Result<RwLockWriteGuard<'a, R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade() } {
            let rwlock = s.rwlock;
            mem::forget(s);
            Ok(RwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }
}

impl<'a, R: RawRwLockUpgradeFair + 'a, T: ?Sized + 'a> RwLockUpgradableReadGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable_fair();
        }
        mem::forget(s);
    }

    
    
    
    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable_fair();
        }
        defer!(s.rwlock.raw.lock_upgradable());
        f()
    }

    
    
    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_upgradable();
        }
    }
}

impl<'a, R: RawRwLockUpgradeDowngrade + 'a, T: ?Sized + 'a> RwLockUpgradableReadGuard<'a, R, T> {
    
    
    
    
    
    
    
    pub fn downgrade(s: Self) -> RwLockReadGuard<'a, R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade_upgradable();
        }
        let rwlock = s.rwlock;
        mem::forget(s);
        RwLockReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    
    pub fn with_upgraded<Ret, F: FnOnce(&mut T) -> Ret>(&mut self, f: F) -> Ret {
        unsafe {
            self.rwlock.raw.upgrade();
        }

        
        
        defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

        
        
        
        f(unsafe { &mut *self.rwlock.data.get() })
    }

    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded<Ret, F: FnOnce(&mut T) -> Ret>(&mut self, f: F) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade() } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }
}

impl<'a, R: RawRwLockUpgradeTimed + 'a, T: ?Sized + 'a> RwLockUpgradableReadGuard<'a, R, T> {
    
    
    
    
    
    pub fn try_upgrade_for(
        s: Self,
        timeout: R::Duration,
    ) -> Result<RwLockWriteGuard<'a, R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade_for(timeout) } {
            let rwlock = s.rwlock;
            mem::forget(s);
            Ok(RwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }

    
    
    
    
    
    #[inline]
    pub fn try_upgrade_until(
        s: Self,
        timeout: R::Instant,
    ) -> Result<RwLockWriteGuard<'a, R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade_until(timeout) } {
            let rwlock = s.rwlock;
            mem::forget(s);
            Ok(RwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }
}

impl<'a, R: RawRwLockUpgradeTimed + RawRwLockUpgradeDowngrade + 'a, T: ?Sized + 'a>
    RwLockUpgradableReadGuard<'a, R, T>
{
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded_for<Ret, F: FnOnce(&mut T) -> Ret>(
        &mut self,
        timeout: R::Duration,
        f: F,
    ) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade_for(timeout) } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded_until<Ret, F: FnOnce(&mut T) -> Ret>(
        &mut self,
        timeout: R::Instant,
        f: F,
    ) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade_until(timeout) } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }
}

impl<'a, R: RawRwLockUpgrade + 'a, T: ?Sized + 'a> Deref for RwLockUpgradableReadGuard<'a, R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

impl<'a, R: RawRwLockUpgrade + 'a, T: ?Sized + 'a> Drop for RwLockUpgradableReadGuard<'a, R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_upgradable();
        }
    }
}

impl<'a, R: RawRwLockUpgrade + 'a, T: fmt::Debug + ?Sized + 'a> fmt::Debug
    for RwLockUpgradableReadGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

impl<'a, R: RawRwLockUpgrade + 'a, T: fmt::Display + ?Sized + 'a> fmt::Display
    for RwLockUpgradableReadGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

#[cfg(feature = "owning_ref")]
unsafe impl<'a, R: RawRwLockUpgrade + 'a, T: ?Sized + 'a> StableAddress
    for RwLockUpgradableReadGuard<'a, R, T>
{
}





#[cfg(feature = "arc_lock")]
#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct ArcRwLockUpgradableReadGuard<R: RawRwLockUpgrade, T: ?Sized> {
    rwlock: Arc<RwLock<R, T>>,
    marker: PhantomData<R::GuardMarker>,
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgrade, T: ?Sized> ArcRwLockUpgradableReadGuard<R, T> {
    
    pub fn rwlock(s: &Self) -> &Arc<RwLock<R, T>> {
        &s.rwlock
    }

    
    
    
    #[inline]
    pub fn unlocked<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable();
        }
        defer!(s.rwlock.raw.lock_upgradable());
        f()
    }

    
    
    pub fn upgrade(s: Self) -> ArcRwLockWriteGuard<R, T> {
        
        unsafe {
            s.rwlock.raw.upgrade();
        }

        
        
        let s = ManuallyDrop::new(s);
        let rwlock = unsafe { ptr::read(&s.rwlock) };

        ArcRwLockWriteGuard {
            rwlock,
            marker: PhantomData,
        }
    }

    
    
    
    pub fn try_upgrade(s: Self) -> Result<ArcRwLockWriteGuard<R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade() } {
            
            let s = ManuallyDrop::new(s);
            let rwlock = unsafe { ptr::read(&s.rwlock) };

            Ok(ArcRwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgradeFair, T: ?Sized> ArcRwLockUpgradableReadGuard<R, T> {
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable_fair();
        }

        
        let mut s = ManuallyDrop::new(s);
        unsafe { ptr::drop_in_place(&mut s.rwlock) };
    }

    
    
    
    #[inline]
    pub fn unlocked_fair<F, U>(s: &mut Self, f: F) -> U
    where
        F: FnOnce() -> U,
    {
        
        unsafe {
            s.rwlock.raw.unlock_upgradable_fair();
        }
        defer!(s.rwlock.raw.lock_upgradable());
        f()
    }

    
    
    
    #[inline]
    pub fn bump(s: &mut Self) {
        
        unsafe {
            s.rwlock.raw.bump_upgradable();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgradeDowngrade, T: ?Sized> ArcRwLockUpgradableReadGuard<R, T> {
    
    
    
    
    
    
    
    pub fn downgrade(s: Self) -> ArcRwLockReadGuard<R, T> {
        
        unsafe {
            s.rwlock.raw.downgrade_upgradable();
        }

        
        let s = ManuallyDrop::new(s);
        let rwlock = unsafe { ptr::read(&s.rwlock) };

        ArcRwLockReadGuard {
            rwlock,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    
    pub fn with_upgraded<Ret, F: FnOnce(&mut T) -> Ret>(&mut self, f: F) -> Ret {
        unsafe {
            self.rwlock.raw.upgrade();
        }

        
        
        defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

        
        
        
        f(unsafe { &mut *self.rwlock.data.get() })
    }

    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded<Ret, F: FnOnce(&mut T) -> Ret>(&mut self, f: F) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade() } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgradeTimed, T: ?Sized> ArcRwLockUpgradableReadGuard<R, T> {
    
    
    
    
    
    pub fn try_upgrade_for(
        s: Self,
        timeout: R::Duration,
    ) -> Result<ArcRwLockWriteGuard<R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade_for(timeout) } {
            
            let s = ManuallyDrop::new(s);
            let rwlock = unsafe { ptr::read(&s.rwlock) };

            Ok(ArcRwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }

    
    
    
    
    
    #[inline]
    pub fn try_upgrade_until(
        s: Self,
        timeout: R::Instant,
    ) -> Result<ArcRwLockWriteGuard<R, T>, Self> {
        
        if unsafe { s.rwlock.raw.try_upgrade_until(timeout) } {
            
            let s = ManuallyDrop::new(s);
            let rwlock = unsafe { ptr::read(&s.rwlock) };

            Ok(ArcRwLockWriteGuard {
                rwlock,
                marker: PhantomData,
            })
        } else {
            Err(s)
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgradeTimed + RawRwLockUpgradeDowngrade, T: ?Sized>
    ArcRwLockUpgradableReadGuard<R, T>
{
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded_for<Ret, F: FnOnce(&mut T) -> Ret>(
        &mut self,
        timeout: R::Duration,
        f: F,
    ) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade_for(timeout) } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    pub fn try_with_upgraded_until<Ret, F: FnOnce(&mut T) -> Ret>(
        &mut self,
        timeout: R::Instant,
        f: F,
    ) -> Option<Ret> {
        if unsafe { self.rwlock.raw.try_upgrade_until(timeout) } {
            
            
            defer!(unsafe { self.rwlock.raw.downgrade_to_upgradable() });

            
            
            
            Some(f(unsafe { &mut *self.rwlock.data.get() }))
        } else {
            None
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgrade, T: ?Sized> Deref for ArcRwLockUpgradableReadGuard<R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.rwlock.data.get() }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgrade, T: ?Sized> Drop for ArcRwLockUpgradableReadGuard<R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.rwlock.raw.unlock_upgradable();
        }
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgrade, T: fmt::Debug + ?Sized> fmt::Debug
    for ArcRwLockUpgradableReadGuard<R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

#[cfg(feature = "arc_lock")]
impl<R: RawRwLockUpgrade, T: fmt::Display + ?Sized> fmt::Display
    for ArcRwLockUpgradableReadGuard<R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}








#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct MappedRwLockReadGuard<'a, R: RawRwLock, T: ?Sized> {
    raw: &'a R,
    data: *const T,
    marker: PhantomData<&'a T>,
}

unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + Sync + 'a> Sync for MappedRwLockReadGuard<'a, R, T> {}
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + Sync + 'a> Send for MappedRwLockReadGuard<'a, R, T> where
    R::GuardMarker: Send
{
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> MappedRwLockReadGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    #[inline]
    pub fn map<U: ?Sized, F>(s: Self, f: F) -> MappedRwLockReadGuard<'a, R, U>
    where
        F: FnOnce(&T) -> &U,
    {
        let raw = s.raw;
        let data = f(unsafe { &*s.data });
        mem::forget(s);
        MappedRwLockReadGuard {
            raw,
            data,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_map<U: ?Sized, F>(s: Self, f: F) -> Result<MappedRwLockReadGuard<'a, R, U>, Self>
    where
        F: FnOnce(&T) -> Option<&U>,
    {
        let raw = s.raw;
        let data = match f(unsafe { &*s.data }) {
            Some(data) => data,
            None => return Err(s),
        };
        mem::forget(s);
        Ok(MappedRwLockReadGuard {
            raw,
            data,
            marker: PhantomData,
        })
    }
}

impl<'a, R: RawRwLockFair + 'a, T: ?Sized + 'a> MappedRwLockReadGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.raw.unlock_shared_fair();
        }
        mem::forget(s);
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Deref for MappedRwLockReadGuard<'a, R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.data }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Drop for MappedRwLockReadGuard<'a, R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.raw.unlock_shared();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Debug + ?Sized + 'a> fmt::Debug
    for MappedRwLockReadGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Display + ?Sized + 'a> fmt::Display
    for MappedRwLockReadGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

#[cfg(feature = "owning_ref")]
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> StableAddress
    for MappedRwLockReadGuard<'a, R, T>
{
}








#[clippy::has_significant_drop]
#[must_use = "if unused the RwLock will immediately unlock"]
pub struct MappedRwLockWriteGuard<'a, R: RawRwLock, T: ?Sized> {
    raw: &'a R,
    data: *mut T,
    marker: PhantomData<&'a mut T>,
}

unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + Sync + 'a> Sync
    for MappedRwLockWriteGuard<'a, R, T>
{
}
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + Send + 'a> Send for MappedRwLockWriteGuard<'a, R, T> where
    R::GuardMarker: Send
{
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> MappedRwLockWriteGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    #[inline]
    pub fn map<U: ?Sized, F>(s: Self, f: F) -> MappedRwLockWriteGuard<'a, R, U>
    where
        F: FnOnce(&mut T) -> &mut U,
    {
        let raw = s.raw;
        let data = f(unsafe { &mut *s.data });
        mem::forget(s);
        MappedRwLockWriteGuard {
            raw,
            data,
            marker: PhantomData,
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn try_map<U: ?Sized, F>(s: Self, f: F) -> Result<MappedRwLockWriteGuard<'a, R, U>, Self>
    where
        F: FnOnce(&mut T) -> Option<&mut U>,
    {
        let raw = s.raw;
        let data = match f(unsafe { &mut *s.data }) {
            Some(data) => data,
            None => return Err(s),
        };
        mem::forget(s);
        Ok(MappedRwLockWriteGuard {
            raw,
            data,
            marker: PhantomData,
        })
    }
}

impl<'a, R: RawRwLockFair + 'a, T: ?Sized + 'a> MappedRwLockWriteGuard<'a, R, T> {
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    pub fn unlock_fair(s: Self) {
        
        unsafe {
            s.raw.unlock_exclusive_fair();
        }
        mem::forget(s);
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Deref for MappedRwLockWriteGuard<'a, R, T> {
    type Target = T;
    #[inline]
    fn deref(&self) -> &T {
        unsafe { &*self.data }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> DerefMut for MappedRwLockWriteGuard<'a, R, T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut T {
        unsafe { &mut *self.data }
    }
}

impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> Drop for MappedRwLockWriteGuard<'a, R, T> {
    #[inline]
    fn drop(&mut self) {
        
        unsafe {
            self.raw.unlock_exclusive();
        }
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Debug + ?Sized + 'a> fmt::Debug
    for MappedRwLockWriteGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&**self, f)
    }
}

impl<'a, R: RawRwLock + 'a, T: fmt::Display + ?Sized + 'a> fmt::Display
    for MappedRwLockWriteGuard<'a, R, T>
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        (**self).fmt(f)
    }
}

#[cfg(feature = "owning_ref")]
unsafe impl<'a, R: RawRwLock + 'a, T: ?Sized + 'a> StableAddress
    for MappedRwLockWriteGuard<'a, R, T>
{
}
