
invisible( utils::globalVariables( c( "vertex" ) ) )

#' gsnSubset
#'
#' @description Create a subset GSNData object, containing only a subset of vertices or subnets.
#'
#' @param object A \code{GSNData} object.
#' @param distance Specifies a distance metric to use for subsetting. Defaults to the \code{default_distance}.
#' @param vertex_names A character vector specifying the vertex names/gene sets to include in the \code{GSNData}
#' subset object.
#' @param subnet_names  A character vector specifying the names of the subnets to include in the \code{GSNData}
#' subset object.
#'
#' @return A new \code{GSNData} object is returned containing a subset of the vertices and subnets from the original
#' \code{GSNData} object. For a given distance, the following data are subsetted for the included vertices and
#' copied:
#'
#' \item{\code{$distances[[distance]]$matrix}}{The raw distance matrix, subsetted.}
#' \item{\code{$distances[[distance]]$pared}}{The pared distance matrix, subsetted.}
#' \item{\code{$distances[[distance]]$edges}}{The edge list, subsetted}
#' \item{\code{$distances[[distance]]$vertex_subnets}}{The vertex assignments for each subnet, subsetted.}
#' \item{\code{$distances[[distance]]$clusters}}{The cluster assignments for each subnet, subsetted (for
#'  hierarchical clustering).}
#' \item{\code{$distances[[distance]]$optimal_extreme}}{Character vector of length 1 indicating whether min or max distances
#' are close in the raw distance matrix.}
#' \item{\code{$distances[[distance]]$pared_optimal_extreme}}{Character vector of length 1  indicating whether min
#' or max distances are close in the pared distance matrix.}
#'
#' The hclust object generated by hierarchical clustering is not currently subsetted or copied.
#'
#' The \code{default_distance} is set as whichever distance matrix is copied. Currently, this function only supports
#' copying a single distance matrix.
#'
#' The following pathways data are copied:
#'
#' \item{\code{$pathways$data}}{Pathways results, subsetted.}
#' \item{\code{$pathways$type}}{The type of pathways results, copied.}
#' \item{\code{$pathways$id_col}}{The identifier, copied.}
#' \item{\code{$pathways$stat_col}}{Statistical column name, copied.}
#' \item{\code{$pathways$sig_order}}{The significance order of the pathways results, based on the stat_column; are
#' the pathways results to be sorted by significance from 'loToHi' (most significant values are low) or 'hiToLo'
#' (most significant values high)?}
#'
#'
#' @details This function is useful for subsetting a single subnet, or a small set of subnets for the purpose of
#' plotting just that subnet.
#'
#' @export
#'
#'
#' @seealso \code{\link{GSNData}()}
#'
gsnSubset<- function( object, distance = NULL, vertex_names = c(), subnet_names = NULL ){
  stopifnot( "GSNData" %in% class( object ) )
  if( is.null( distance ) ) distance <- object$default_distance
  if( is.null( distance ) ) stop( "Need distance argument" )

  if( ! is.null( subnet_names ) ){
    subnetVertexNames.l <- object$distances[[distance]]$subnets[[subnet_names]]
    vertex_names <- unlist(unique( c( vertex_names, subnetVertexNames.l ) ) )
  } else {
    vertex_names <- unlist(unique( vertex_names ))
  }
  GSN.subset <- GSNData()
  GSN.subset$default_distance <- distance
  GSN.subset$distances[[distance]] <- list()
  GSN.subset$distances[[distance]]$vertices <- vertex_names

  if( ! is.null(object$distances[[distance]]$matrix))
    GSN.subset$distances[[distance]]$matrix <- object$distances[[distance]]$matrix[vertex_names, vertex_names]

  if( ! is.null(object$distances[[distance]]$pared))
    GSN.subset$distances[[distance]]$pared <- object$distances[[distance]]$pared[vertex_names, vertex_names]

  if( ! is.null(object$distances[[distance]]$edges))
    GSN.subset$distances[[distance]]$edges <- with( object$distances[[distance]], edges[edges[,1] %in% vertex_names & edges[,2] %in% vertex_names,])

  if( ! is.null(object$distances[[distance]]$vertex_subnets)){
    GSN.subset$distances[[distance]]$vertex_subnets <- subset(object$distances[[distance]]$vertex_subnets, vertex %in% vertex_names )
    GSN.subset$distances[[distance]]$subnets <- with(GSN.subset$distances[[distance]]$vertex_subnets, split( x = vertex, f = subnet ))
  }

  if( ! is.null(object$distances[[distance]]$clusters))
    GSN.subset$distances[[distance]]$clusters <- object$distances[[distance]]$clusters[vertex_names]

  if( ! is.null(object$distances[[distance]]$optimal_extreme ) )
    GSN.subset$distances[[distance]]$optimal_extreme <- object$distances[[distance]]$optimal_extreme

  if( ! is.null(object$distances[[distance]]$pared_optimal_extreme ) )
    GSN.subset$distances[[distance]]$pared_optimal_extreme <- object$distances[[distance]]$pared_optimal_extreme

  if( ! is.null( object$pathways ) ){
    GSN.subset$pathways <- object$pathways
    if( ! is.null( GSN.subset$pathways$data ) && ! is.null( GSN.subset$pathways$id_col ) )
      GSN.subset$pathways$data <- GSN.subset$pathways$data[GSN.subset$pathways$data[,object$pathways$id_col] %in% vertex_names,]
  }
  GSN.subset
}
