% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ALMC.R
\name{ALMC_RNAmf}
\alias{ALMC_RNAmf}
\title{find the next point by ALMC criterion}
\usage{
ALMC_RNAmf(Xref = NULL, Xcand = NULL, fit, mc.sample = 100,
cost = NULL, optim = TRUE, parallel = FALSE, ncore = 1, trace=TRUE)
}
\arguments{
\item{Xref}{vector or matrix of reference location to approximate the integral of ALC. If \code{Xref=NULL}, \eqn{100 \times d} points from 0 to 1 are generated by Latin hypercube design. Default is \code{NULL}.}

\item{Xcand}{vector or matrix of candidate set which could be added into the current design only when \code{optim=FALSE}. \code{Xcand} is the set of the points where ALM criterion is evaluated. If \code{Xcand=NULL}, \code{Xref} is used. Default is \code{NULL}.}

\item{fit}{object of class \code{RNAmf}.}

\item{mc.sample}{a number of mc samples generated for the imputation through MC approximation. Default is \code{100}.}

\item{cost}{vector of the costs for each level of fidelity. If \code{cost=NULL}, total costs at all levels would be 1. \code{cost} is encouraged to have a ascending order of positive value. Default is \code{NULL}.}

\item{optim}{logical indicating whether to optimize AL criterion by \code{optim}'s gradient-based \code{L-BFGS-B} method. If \code{optim=TRUE}, \eqn{5 \times d} starting points are generated by Latin hypercube design for optimization. If \code{optim=FALSE}, AL criterion is optimized on the \code{Xcand}. Default is \code{TRUE}.}

\item{parallel}{logical indicating whether to compute the AL criterion in parallel or not. If \code{parallel=TRUE}, parallel computation is utilized. Default is \code{FALSE}.}

\item{ncore}{a number of core for parallel. It is only used if \code{parallel=TRUE}. Default is 1.}

\item{trace}{logical indicating whether to print the computational time for each step. If \code{trace=TRUE}, the computation time for each step is printed. Default is \code{TRUE}.}
}
\value{
\itemize{
  \item \code{ALMC}: vector of ALMC criterion \eqn{ \frac{\Delta \sigma_L^{2}(l,\bm{x^*})}{\sum^l_{j=1}C_j}} for \eqn{1\leq l\leq L}.
  \item \code{ALM}: vector of ALM criterion computed at each point of \code{Xcand} at the highest fidelity level if \code{optim=FALSE}. If \code{optim=TRUE}, \code{ALM} returns \code{NULL}.
  \item \code{ALC}: list of ALC criterion integrated on \code{Xref} when each data point on \code{Xcand} is added at each level \eqn{l} if \code{optim=FALSE}. If \code{optim=TRUE}, \code{ALC} returns \code{NULL}.
  \item \code{cost}: a copy of \code{cost}.
  \item \code{Xcand}: a copy of \code{Xcand}.
  \item \code{chosen}: list of chosen level and point.
  \item \code{time}: a scalar of the time for the computation.
}
}
\description{
The function acquires the new point by the hybrid approach,
referred to as Active learning MacKay-Cohn (ALMC) criterion.
It finds the optimal input location \eqn{\bm{x^*}}
by maximizing \eqn{\sigma^{*2}_L(\bm{x})},
the posterior predictive variance at the highest-fidelity level \eqn{L}.
After selecting \eqn{\bm{x^*}},
it finds the optimal fidelity level by maximizing ALC criterion at \eqn{\bm{x^*}},
\eqn{\text{argmax}_{l\in\{1,\ldots,L\}} \frac{\Delta \sigma_L^{2}(l,\bm{x^*})}{\sum^l_{j=1}C_j}},
where \eqn{C_j} is the simulation cost at level \eqn{j}.
See \code{\link{ALC_RNAmf}}.
For details, see Heo and Sung (2025, <\doi{https://doi.org/10.1080/00401706.2024.2376173}>).

A new point is acquired on \code{Xcand}. If \code{Xcand=NULL} and \code{Xref=NULL}, a new point is acquired on unit hypercube \eqn{[0,1]^d}.
}
\examples{
\donttest{
library(lhs)
library(doParallel)
library(foreach)

### simulation costs ###
cost <- c(1, 3)

### 1-d Perdikaris function in Perdikaris, et al. (2017) ###
# low-fidelity function
f1 <- function(x) {
  sin(8 * pi * x)
}

# high-fidelity function
f2 <- function(x) {
  (x - sqrt(2)) * (sin(8 * pi * x))^2
}

### training data ###
n1 <- 13
n2 <- 8

### fix seed to reproduce the result ###
set.seed(1)

### generate initial nested design ###
X <- NestedX(c(n1, n2), 1)
X1 <- X[[1]]
X2 <- X[[2]]

### n1 and n2 might be changed from NestedX ###
### assign n1 and n2 again ###
n1 <- nrow(X1)
n2 <- nrow(X2)

y1 <- f1(X1)
y2 <- f2(X2)

### n=100 uniform test data ###
x <- seq(0, 1, length.out = 100)

### fit an RNAmf ###
fit.RNAmf <- RNAmf(list(X1, X2), list(y1, y2), kernel = "sqex", constant=TRUE)

### predict ###
predy <- predict(fit.RNAmf, x)$mu
predsig2 <- predict(fit.RNAmf, x)$sig2

### active learning with optim=TRUE ###
almc.RNAmf.optim <- ALMC_RNAmf(
  Xref = x, Xcand = x, fit.RNAmf, cost = cost,
  optim = TRUE, parallel = TRUE, ncore = 2
)
print(almc.RNAmf.optim$time) # computation time of optim=TRUE

### active learning with optim=FALSE ###
almc.RNAmf <- ALMC_RNAmf(
  Xref = x, Xcand = x, fit.RNAmf, cost = cost,
  optim = FALSE, parallel = TRUE, ncore = 2
)
print(almc.RNAmf$time) # computation time of optim=FALSE

### visualize ALMC ###
oldpar <- par(mfrow = c(1, 2))
plot(x, almc.RNAmf$ALM,
  type = "l", lty = 2,
  xlab = "x", ylab = "ALM criterion at the high-fidelity level"
)
points(almc.RNAmf$chosen$Xnext,
  almc.RNAmf$ALM[which(x == drop(almc.RNAmf$chosen$Xnext))],
  pch = 16, cex = 1, col = "red"
)
plot(x, almc.RNAmf$ALC$ALC1,
  type = "l", lty = 2,
  ylim = c(min(c(almc.RNAmf$ALC$ALC1, almc.RNAmf$ALC$ALC2)),
  max(c(almc.RNAmf$ALC$ALC1, almc.RNAmf$ALC$ALC2))),
  xlab = "x", ylab = "ALC criterion augmented at each level on the optimal input location"
)
lines(x, almc.RNAmf$ALC$ALC2, type = "l", lty = 2)
points(almc.RNAmf$chosen$Xnext,
  almc.RNAmf$ALC$ALC1[which(x == drop(almc.RNAmf$chosen$Xnext))],
  pch = 16, cex = 1, col = "red"
)
points(almc.RNAmf$chosen$Xnext,
  almc.RNAmf$ALC$ALC2[which(x == drop(almc.RNAmf$chosen$Xnext))],
  pch = 16, cex = 1, col = "red"
)
par(oldpar)}

}
