#' Draw a next (batch) of Monte-Carlo samples to update the sequential estimate
#' of the p-value
#'
#' @param R an object of class `avseqmc_progress` containing previous progress on
#'   the sequential estimation.
#' @param compute_lower TRUE if the lower limit of the confidence sequence of
#'   significance epsilon (as embedded in parameter R) should be computed.
#'
#' @return An (updated) object of class `avseqmc_progress` containing the progress of the
#'  sequentially estimated p-value. The object is a list containing the
#'  following elements:
#'
#'  - `$epsilon`: risk of overestimated significance used in the sequential estimation.
#'  - `$sample_G`: function that samples (batches) from the Monte-Carlo distribution $G^*(X)$ as in Equation (5).
#'  - `$ptilde`: sequence of sequential $p$-value estimates. The final value in this sequence is the most recent estimate of the $p$-value.
#'  - `$Ltilde`: sequence of lower bounds of the confidence sequence based on the construction by Robbins (1970). Contains NA values if these were
#'  not computed by default through `stopcrit =
#'  list("type"="futility","param"=...)` or requested using
#'  `compute_lower=TRUE`.
#'  - `$n`: total number of samples drawn from the MC sampler.
#'  - `$S`: total number of ones observed from the MC sampler.
#'  - `$B`: sequence of number of ones observed at each sampling timepoint (which can be greater than 1 if `sample_G` samples in batches)
#'  - `$Bn`: sequence of number of samples drawn from MC sampler at each timepoint (which can be greater than 1 if `sample_G` samples in batches)
#' 
#' @keywords internal
nextsample <- function(R, compute_lower = FALSE) {
  # Draw new MC sample
  Bnew <- R$sample_G()
  # Update avseqmc_progress object
  R$B <- c(R$B, sum(Bnew))
  R$Bn <- c(R$Bn, length(Bnew))
  R$S <- R$S + sum(Bnew)
  R$n <- R$n + length(Bnew)
  R$ptilde <- cummin(c(R$ptilde, Un_tilde(R$S, R$n, R$epsilon) + R$epsilon))

  if(compute_lower){
    R$Ltilde <- c(R$Ltilde, max(c(R$Ltilde, Ln_tilde(R$S, R$n, R$epsilon)), na.rm = TRUE))
  } else {
    R$Ltilde <- c(R$Ltilde, rep(NA, length(Bnew)))
  }
  return(R)
}
