\docType{methods}
\name{demi.wilcox.test.fast}
\alias{demi.wilcox.test.fast}
\title{Cluster probes into higher and lower clusters based on their differential signalling}
\usage{
  demi.wilcox.test.fast(x = "DEMIClust")
}
\arguments{
  \item{x}{A \code{DEMIClust} object. The \code{DEMIClust}
  object containing normalized expression values used for
  statistical significance test on differential signalling
  of probes. The object contains the column indexes of
  groups (e.g. 'test' and 'control') used in the analysis.}
}
\value{
  A \code{list}. Returns a \code{list} containing different
  sets of probes that behave similarly under current
  statistical test (e.g. up- or down-regulated probes).
}
\description{
  Performs a modified \code{wilcox.test} on normalized
  expression value matrix defined in \code{DEMIClust}
  object. It precalculates the probabilities of the rank
  sums and makes the algorithm run a lot quicker.
}
\examples{
\dontrun{

# To use the example we need to download a subset of CEL files from http://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?acc=GSE9819 published by Pradervand et al. 2008.

# Set the destination folder where the downloaded files fill be located. It can be any folder of your choosing.
destfolder <- "demitest/testdata/"

# Download packed CEL files and change the names according to the feature they represent (for example to include UHR or BRAIN in them to denote the features).
# It is a good practice to name the files according to their features which allows easier identification of the files later.
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247694/suppl/GSM247694.CEL.gz", destfile = paste( destfolder, "UHR01_GSM247694.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247695/suppl/GSM247695.CEL.gz", destfile = paste( destfolder, "UHR02_GSM247695.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247698/suppl/GSM247698.CEL.gz", destfile = paste( destfolder, "UHR03_GSM247698.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247699/suppl/GSM247699.CEL.gz", destfile = paste( destfolder, "UHR04_GSM247699.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247696/suppl/GSM247696.CEL.gz", destfile = paste( destfolder, "BRAIN01_GSM247696.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247697/suppl/GSM247697.CEL.gz", destfile = paste( destfolder, "BRAIN02_GSM247697.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247700/suppl/GSM247700.CEL.gz", destfile = paste( destfolder, "BRAIN03_GSM247700.CEL.gz", sep = "" ) )
download.file( "ftp://ftp.ncbi.nlm.nih.gov/geo/samples/GSM247nnn/GSM247701/suppl/GSM247701.CEL.gz", destfile = paste( destfolder, "BRAIN04_GSM247701.CEL.gz", sep = "" ) )

# We need the gunzip function (located in the R.utils package) to unpack the gz files.
# Also we will remove the original unpacked files for we won't need them.
library( R.utils )
for( i in list.files( destfolder ) ) {
	gunzip( paste( destfolder, i, sep = "" ), remove = TRUE )
}

# Now we can continue the example of the function demi.wilcox.test.fast

# Basic experiment set up.
demiexp <- DEMIExperiment(analysis = 'gene', celpath = destfolder,
			experiment = 'myexperiment', organism = 'homo_sapiens')

# Create clusters with default behaviour
demiclust <- DEMIClust( demiexp, group = c( "BRAIN", "UHR" ) )

# Retrieve probes whose differential signalling was statistically significant
sigprobes <- demi.wilcox.test.fast( demiclust )

# However it makes more sense to incorporate the method straight into \\code{DEMIClust} object
demiclust <- DEMIClust( demiexp, group = c( "BRAIN", "UHR" ), clust.method = demi.wilcox.test.fast )

# Retrieve the probes whose differential signalling was statistically significant
sigprobes <- getCluster( demiclust )

# Retrieve the cluster names since we have both up-regulated and down-regulated probe clusters
names( sigprobes )

# Retrieve the up-regulated probes whose cluster names contain the sign '[H]'
head( sigprobes[[grep("\\\\[H\\\\]", names( sigprobes ))]] )

# Retrieve the down-regulated probes whose cluster names contain the sign '[L]'
head( sigprobes[[grep("\\\\[L\\\\]", names( sigprobes ))]] )

}
}
\author{
  Sten Ilmjarv
}
\seealso{
  \code{\link{wilcox.test}} which this function mimics and
  \code{\link{wprob}} which this function implements.
}

