# Preamble code
data("traits_birds")
simple_site_sp <- matrix(1, nrow = 1, ncol = nrow(traits_birds),
                        dimnames = list("s1", row.names(traits_birds)))

# Actual tests
test_that("Rao's entropy output format", {

  rq <- expect_silent(fd_raoq(traits_birds, sp_com = simple_site_sp))

  expect_s3_class(rq, "data.frame")
  expect_length(rq, 2)
  expect_equal(nrow(rq), 1)
  expect_equal(colnames(rq), c("site", "Q"))


  rq <- expect_silent(fd_raoq(traits_birds))

  expect_s3_class(rq, "data.frame")
  expect_length(rq, 2)
  expect_equal(nrow(rq), 1)
  expect_equal(colnames(rq), c("site", "Q"))

})

test_that("Rao's entropy computation are in line with other packages", {
  expect_equal(fd_raoq(traits_birds, simple_site_sp)$Q, 170.0519,
               tolerance = 1e-6)
})

test_that("Rao's entropy works in 1D", {

  expect_identical(
    fd_raoq(traits_birds[, 1], sp_com = simple_site_sp),
    fd_raoq(traits_birds[, 1, drop = FALSE], sp_com = simple_site_sp)
  )

})

test_that("Rao's Quadratic Entropy works on subset of site/species", {
  site_sp <- matrix(1, ncol = nrow(traits_birds))
  colnames(site_sp) <-  rownames(traits_birds)
  rownames(site_sp) <- "s1"

  expect_message(fd_raoq(traits_birds, site_sp[, 2:ncol(site_sp),
                                               drop = FALSE]),
                 paste0("Differing number of species between trait dataset ",
                        "and site-species matrix\nTaking subset of species"))

  expect_message(fd_raoq(traits_birds[2:nrow(traits_birds),], site_sp),
                 paste0("Differing number of species between trait dataset ",
                        "and site-species matrix\nTaking subset of species"))
})

test_that("Rao's quadratric entropy works for sites with no species", {
  data("traits_plants")
  data("site_sp_plants")

  raoq <- expect_silent(
    fd_raoq(traits_plants, site_sp_plants[10,, drop = FALSE])
  )

  expect_equal(raoq$Q[[1]], 0)
})

test_that("Rao's Quadratic Entropy works on sparse matrices", {

  skip_if_not_installed("Matrix")

  site_sp <- matrix(1, ncol = nrow(traits_birds))
  colnames(site_sp) <-  rownames(traits_birds)
  rownames(site_sp) <- "s1"

  sparse_site_sp <- Matrix(site_sp, sparse = TRUE)

  sparse_dist_mat <- Matrix(as.matrix(dist(traits_birds)), sparse = TRUE)

  # Only site-species matrix is sparse
  expect_silent(fd_raoq(traits_birds, sparse_site_sp))

  expect_equal(fd_raoq(traits_birds, sparse_site_sp)$Q, 170.0519,
               tolerance = 1e-6)

  # Only distance matrix is sparse
  expect_silent(fd_raoq(sp_com = site_sp, dist_matrix = sparse_dist_mat))

  expect_equal(fd_raoq(sp_com = site_sp, dist_matrix = sparse_dist_mat)$Q,
               170.0519, tolerance = 1e-6)

  # Both site-species and distance matrix are sparse
  expect_silent(fd_raoq(sp_com = sparse_site_sp, dist_matrix = sparse_dist_mat))

  expect_equal(
    fd_raoq(sp_com = sparse_site_sp, dist_matrix = sparse_dist_mat)$Q,
    170.0519, tolerance = 1e-6)

})

test_that("Rao's entropy fails gracefully", {
  # No traits and no dissimilarity
  expect_error(
    fd_raoq(NULL, matrix(1), NULL),
    "Please provide either a trait dataset or a dissimilarity matrix",
    fixed = TRUE
  )

  # Both traits AND dissimilarity
  expect_error(
    fd_raoq(data.frame(a = 1), matrix(1), matrix(1)),
    "Please provide either a trait dataset or a dissimilarity matrix",
    fixed = TRUE
  )

  # Species matrix doesn't contain species from trait data
  expect_error(
    fd_raoq(data.frame(a = 1, row.names = "sp1"), matrix(1)),
    paste0("No species in common found between trait dataset ",
           "and site-species matrix"),
    fixed = TRUE
  )
})
