% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesopt_mpcl.R
\name{mlr_loop_functions_mpcl}
\alias{mlr_loop_functions_mpcl}
\alias{bayesopt_mpcl}
\title{Single-Objective Bayesian Optimization via Multipoint Constant Liar}
\usage{
bayesopt_mpcl(
  instance,
  surrogate,
  acq_function,
  acq_optimizer,
  init_design_size = NULL,
  q = 2L,
  liar = mean,
  random_interleave_iter = 0L
)
}
\arguments{
\item{instance}{(\link[bbotk:OptimInstanceBatchSingleCrit]{bbotk::OptimInstanceBatchSingleCrit})\cr
The \link[bbotk:OptimInstanceBatchSingleCrit]{bbotk::OptimInstanceBatchSingleCrit} to be optimized.}

\item{surrogate}{(\link{Surrogate})\cr
\link{Surrogate} to be used as a surrogate.
Typically a \link{SurrogateLearner}.}

\item{acq_function}{(\link{AcqFunction})\cr
\link{AcqFunction} to be used as acquisition function.}

\item{acq_optimizer}{(\link{AcqOptimizer})\cr
\link{AcqOptimizer} to be used as acquisition function optimizer.}

\item{init_design_size}{(\code{NULL} | \code{integer(1)})\cr
Size of the initial design.
If \code{NULL} and the \link[bbotk:ArchiveBatch]{bbotk::ArchiveBatch} contains no evaluations, \code{4 * d} is used with \code{d} being the
dimensionality of the search space.
Points are generated via a Sobol sequence.}

\item{q}{(\code{integer(1)})\cr
Batch size > \code{1}.
Default is \code{2}.}

\item{liar}{(\code{function})\cr
Any function accepting a numeric vector as input and returning a single numeric output.
Default is \code{mean}. Other sensible functions include \code{min} (or \code{max}, depending on the optimization direction).}

\item{random_interleave_iter}{(\code{integer(1)})\cr
Every \code{random_interleave_iter} iteration (starting after the initial design), a point is
sampled uniformly at random and evaluated (instead of a model based proposal).
For example, if \code{random_interleave_iter = 2}, random interleaving is performed in the second,
fourth, sixth, ... iteration.
Default is \code{0}, i.e., no random interleaving is performed at all.}
}
\value{
invisible(instance)\cr
The original instance is modified in-place and returned invisible.
}
\description{
Loop function for single-objective Bayesian Optimization via multipoint constant liar.
Normally used inside an \link{OptimizerMbo}.

In each iteration after the initial design, the surrogate and acquisition function are updated.
The acquisition function is then optimized, to find a candidate but instead of evaluating this candidate, the
objective function value is obtained by applying the \code{liar} function to all previously obtained objective function values.
This is repeated \code{q - 1} times to obtain a total of \code{q} candidates that are then evaluated in a single batch.
}
\note{
\itemize{
\item The \code{acq_function$surrogate}, even if already populated, will always be overwritten by the \code{surrogate}.
\item The \code{acq_optimizer$acq_function}, even if already populated, will always be overwritten by \code{acq_function}.
\item The \code{surrogate$archive}, even if already populated, will always be overwritten by the \link[bbotk:ArchiveBatch]{bbotk::ArchiveBatch} of the \link[bbotk:OptimInstanceBatchSingleCrit]{bbotk::OptimInstanceBatchSingleCrit}.
\item To make use of parallel evaluations in the case of `q > 1, the objective
function of the \link[bbotk:OptimInstanceBatchSingleCrit]{bbotk::OptimInstanceBatchSingleCrit} must be implemented accordingly.
}
}
\examples{
\donttest{
if (requireNamespace("mlr3learners") &
    requireNamespace("DiceKriging") &
    requireNamespace("rgenoud")) {

  library(bbotk)
  library(paradox)
  library(mlr3learners)

  fun = function(xs) {
    list(y = xs$x ^ 2)
  }
  domain = ps(x = p_dbl(lower = -10, upper = 10))
  codomain = ps(y = p_dbl(tags = "minimize"))
  objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

  instance = OptimInstanceBatchSingleCrit$new(
    objective = objective,
    terminator = trm("evals", n_evals = 7))

  surrogate = default_surrogate(instance)

  acq_function = acqf("ei")

  acq_optimizer = acqo(
    optimizer = opt("random_search", batch_size = 100),
    terminator = trm("evals", n_evals = 100))

  optimizer = opt("mbo",
    loop_function = bayesopt_mpcl,
    surrogate = surrogate,
    acq_function = acq_function,
    acq_optimizer = acq_optimizer,
    args = list(q = 3))

  optimizer$optimize(instance)
}
}
}
\references{
\itemize{
\item Ginsbourger, David, Le Riche, Rodolphe, Carraro, Laurent (2008).
\dQuote{A Multi-Points Criterion for Deterministic Parallel Global Optimization Based on Gaussian Processes.}
\item Wang, Jialei, Clark, C. S, Liu, Eric, Frazier, I. P (2020).
\dQuote{Parallel Bayesian Global Optimization of Expensive Functions.}
\emph{Operations Research}, \bold{68}(6), 1850--1865.
}
}
\seealso{
Other Loop Function: 
\code{\link{loop_function}},
\code{\link{mlr_loop_functions}},
\code{\link{mlr_loop_functions_ego}},
\code{\link{mlr_loop_functions_emo}},
\code{\link{mlr_loop_functions_parego}},
\code{\link{mlr_loop_functions_smsego}}
}
\concept{Loop Function}
