#' P-value Combination Methods for High-Dimensional Data
#'
#' @description
#' Combines individual \eqn{p}-values across multiple studies for each outcome
#' using \eqn{p}-value combination methods applied independently per outcome.
#' Includes traditional and weighted \eqn{p}-value combination approaches and a vote counting method.
#'
#' @usage
#' MetaHDpval(
#'   pmat,
#'   method = c("Fisher", "Stouffer", "wZ", "Lancaster", "wFisher", "Vote counting"),
#'   weight = NULL,
#'   is.onetail = TRUE,
#'   eff.sign = NULL,
#'   alpha = 0.5
#' )
#'
#' @param pmat A \eqn{K \times N} matrix of individual \eqn{p}-values, where \eqn{K} is the number of studies and \eqn{N} is the number of outcomes.
#' @param method Character string specifying the \eqn{p}-value combination method. One of \code{"Fisher"}, \code{"Stouffer"}, \code{"wZ"}, \code{"Lancaster"}, \code{"wFisher"}, or \code{"Vote counting"}. See Details for more information.
#' @param weight An optional \eqn{K \times N} matrix of weights or sample sizes for each outcome in each study. Not relevant for \code{"Vote counting"}.
#' @param is.onetail Logical. If \code{TRUE}, \eqn{p}-values are combined without considering effect directions. If \code{FALSE}, effect directions are used via \code{eff.sign}. Default is \code{TRUE}. Not relevant for \code{"Vote counting"}.
#' @param eff.sign An optional \eqn{K \times N} matrix indicating the signs of effect sizes (e.g., \code{+1} or \code{-1}). Only used when \code{is.onetail = FALSE}. Not relevant for \code{"Vote counting"}.
#' @param alpha Numeric value defining the \eqn{p}-value cutoff for the \code{"Vote counting"} method. By default, \code{alpha = 0.5} splits \eqn{p}-values at 0.5. Values between \code{alpha} and \code{1 - alpha} are treated as neutral. If \code{alpha > 1}, it is interpreted as a percentage.
#'
#' @details
#' The \code{MetaHDpval} function offers five traditional and more recent
#' \eqn{p}-value combination methods implemented using the \code{metapro}
#' R package, as well as a vote counting method implemented using the
#' \code{metap} R package:
#' \itemize{
#'   \item \strong{Fisher's method} (Fisher, 1932), which combines logarithmically
#'   transformed \eqn{p}-values from individual studies for each outcome using
#'   Fisher’s statistic.
#'   \item \strong{Stouffer's method} (Stouffer et al., 1949), which combines
#'   inverse normal–transformed \eqn{p}-values derived from individual study test
#'   statistics for each outcome.
#'   \item \strong{Weighted Z-method (wZ)} (Mosteller and Bush, 1954), an extension
#'   of Stouffer’s method that incorporates study-specific weights, resulting in
#'   a weighted inverse normal combination.
#'   \item \strong{Lancaster's method} (Lancaster, 1961), which generalizes
#'   Fisher’s method by introducing weights and exploits the additive property of
#'   the \eqn{\chi^2}-distribution.
#'   \item \strong{Weighted Fisher's method (wFisher)} (Yoon et al., 2021), which extends
#'   Fisher’s method by allowing non-integer weights reflecting study-specific
#'   information (e.g., sample sizes). This approach replaces the
#'   \eqn{\chi^2}-distribution with the gamma distribution to accommodate
#'   non-integer degrees of freedom.
#'   \item \strong{Vote counting method} (Becker, 1994), that classifies a study
#'   as positive if its \eqn{p}-value is less than \code{alpha} and as negative
#'   if it exceeds \code{1 - alpha}, with studies falling in between treated as
#'   neutral and excluded. The number of positive studies is then counted, and a
#'   one-sided binomial test is applied to the non-neutral studies to obtain a
#'   combined \eqn{p}-value for each outcome.
#' }
#'
#' @return A numeric vector of length \eqn{N} containing the combined \eqn{p}-values for each outcome.
#'
#' @examples
#' ## Example with 5 studies and 12 outcomes
#' set.seed(123)
#' pmat <- matrix(runif(15), nrow = 5, ncol = 12)
#' eff.sign <- matrix(sample(c(-1, 1), 60, replace = TRUE), nrow = 5, ncol = 12)
#' wmat <- matrix(sample(50:200, 60, replace = TRUE), nrow = 5, ncol = 12)
#'
#' ## Fisher's method
#' MetaHDpval(pmat, method = "Fisher", is.onetail = FALSE, eff.sign = eff.sign)
#'
#' ## Weighted Z method
#' MetaHDpval(pmat, method = "wZ", weight = wmat, is.onetail = FALSE, eff.sign = eff.sign)
#'
#' ## Vote counting
#' MetaHDpval(pmat, method = "Vote counting", alpha = 0.4)
#'
#' @references
#' Yoon, S., Baik, B., Park, T., et al. (2021). \emph{Powerful p-value combination methods to detect incomplete association}.
#' Scientific Reports, 11, 6980. \doi{10.1038/s41598-021-86465-y}
#'
#' @references
#' Yoon, S. (2023). \emph{metapro: Robust P-Value Combination Methods}
#' (R package version 1.5.11). Comprehensive R Archive Network (CRAN).
#' \doi{10.32614/CRAN.package.metapro}
#'
#' @references
#' Becker, B.J. (1994). \emph{Combining significance levels. In Cooper H, Hedges LV (eds.), A handbook of research synthesis}, 215–230. Russell Sage, New York.
#'
#' @references
#' Dewey, M. (2025). \emph{metap: Meta-Analysis of Significance Values}
#' (R package version 1.13). Comprehensive R Archive Network (CRAN).
#' \doi{10.32614/CRAN.package.metap}
#'
#' @export MetaHDpval
#'
#' @importFrom metapro wFisher wZ lancaster
#' @importFrom metap votep


MetaHDpval <- function(pmat,method = c("Fisher", "Stouffer", "wZ", "Lancaster", "wFisher", "Vote counting"),weight = NULL,is.onetail = TRUE,eff.sign = NULL,alpha = 0.5) {
  method <- match.arg(method)
  N <- ncol(pmat)
  cmb_p  <- numeric(N)
  if (method != "Vote counting" && !is.onetail && is.null(eff.sign)) {
    warning("Individual p-values are two-sided but effect direction is unknown. Switching to one-tailed.",call. = FALSE)
    is.onetail <- TRUE
  }
  if (method == "Vote counting" && (!is.null(weight) || !is.null(eff.sign))) {
    warning("Arguments 'weight' and 'eff.sign' are ignored for Vote counting.", call. = FALSE)
  }
  for (n in seq_len(N)) {
    p <- as.numeric(pmat[, n])
    eff.sign_n <- if (!is.null(eff.sign)) as.numeric(eff.sign[, n]) else NULL
    wght <- if (!is.null(weight)) as.numeric(weight[, n]) else NULL
    res <- switch(
      method,
      "Fisher" =
        metapro::wFisher(p, weight = NULL, is.onetail = is.onetail, eff.sign = eff.sign_n),
      "wFisher" =
        metapro::wFisher(p, weight = wght, is.onetail = is.onetail, eff.sign = eff.sign_n),
      "Stouffer" =
        metapro::wZ(p, weight = NULL, is.onetail = is.onetail, eff.sign = eff.sign_n),
      "wZ" =
        metapro::wZ(p, weight = wght, is.onetail = is.onetail, eff.sign = eff.sign_n),
      "Lancaster" =
        metapro::lancaster(p, weight = wght, is.onetail = is.onetail, eff.sign = eff.sign_n),
      "Vote counting" =
        metap::votep(p, alpha = alpha)
    )
    cmb_p[n] <- res$p
  }
  return(cmb_p)
}

