#' rb
#'
#' @description
#' Calculates the proportional rate of change in an abundance or biomass index (\code{rb})
#' between consecutive data points using one of three methods:
#' \describe{
#'   \item{\code{"annual"}}{Change between the two most recent data points:
#'     \eqn{(x_1 - x_2)/x_2}. Requires at least 2 values.}
#'   \item{\code{"1over2"}}{Change between the most recent value and the mean of the
#'     two values prior: \eqn{(x_1 - \bar{x}_{2:3})/\bar{x}_{2:3}}. Requires at least 3 values.}
#'   \item{\code{"2over3"}}{Change between the mean of the two most recent values and
#'     the mean of the three values prior: \eqn{(\bar{x}_{1:2} - \bar{x}_{3:5})/\bar{x}_{3:5}}.
#'     Requires at least 5 values.}
#' }
#'
#' @param b_index Numeric vector of abundance or biomass indices in descending time order (most recent first).
#' @param method Character string; one of \code{"annual"} (default), \code{"1over2"}, or \code{"2over3"}.
#' @param na.rm Logical; if \code{TRUE}, \code{NA}s are removed before computing.
#'   If \code{FALSE} (default) and \code{NA}s are present in the needed positions, the result may be \code{NA}.
#' @param digits Optional integer. If supplied, the result is rounded using
#'   \code{round(x, digits)}. If \code{NULL} (default), full precision is returned.
#'
#' @return A numeric scalar: the proportional rate of change \code{rb}. Positive values
#'   indicate an increase; negative values indicate a decrease.
#'
#' @details
#' Validates that sufficient data are available for the chosen method and guards
#' against (near-)zero denominators. If a needed denominator is \code{NA} (after
#' \code{na.rm}) or numerically zero, an error is thrown.
#'
#' @note \code{b_index} must be in descending time order (most recent first). Indices
#'   should be non-negative (e.g., CPUE).
#'
#' @examples
#' cpue <- c(0.75, 0.70, 1.49, 1.20, 1.10)  # most recent first
#' rb(b_index = cpue) # annual method by default
#' rb(b_index = cpue, method = "1over2")
#' rb(b_index = cpue, method = "2over3")
#'
#' cpue2 <- c(0.75, NA, 1.49, 1.20, 1.10)
#' rb(cpue2, method = "1over2", na.rm = TRUE, digits = 2)
#'
#' @seealso \code{\link{TBA}}
#' @export
rb <- function(b_index = NULL,
               method = c("annual", "1over2", "2over3"),
               na.rm = FALSE,
               digits = NULL) {
  method <- match.arg(method)

  # ---- validation ----
  if (!is.numeric(b_index))
    stop("`b_index` must be a numeric vector.", call. = FALSE)
  if (length(b_index) == 0L)
    stop("`b_index` must have length >= 1.", call. = FALSE)
  if (!is.logical(na.rm) || length(na.rm) != 1L || is.na(na.rm))
    stop("`na.rm` must be a single TRUE/FALSE.", call. = FALSE)
  if (!is.null(digits)) {
    if (!is.numeric(digits) || length(digits) != 1L || is.na(digits) ||
        digits < 0 || digits %% 1 != 0)
      stop("`digits` must be a single non-negative integer or NULL.", call. = FALSE)
    digits <- as.integer(digits)
  }
  # allow NA in b_index but reject other non-finite values
  if (any(!is.finite(b_index[!is.na(b_index)])))
    stop("Non-finite values (Inf/-Inf/NaN) found in `b_index`.", call. = FALSE)

  required_len <- switch(method, "annual" = 2L, "1over2" = 3L, "2over3" = 5L)
  if (length(b_index) < required_len)
    stop(sprintf("`method = \"%s\"` requires at least %d data points.",
                 method, required_len), call. = FALSE)

  # helper: safe mean with optional na.rm and informative error if all NA
  safe_mean <- function(x, na.rm) {
    m <- mean(x, na.rm = na.rm)
    if (!na.rm && any(is.na(x)) && is.na(m)) return(m)      # propagate NA
    if (na.rm && all(is.na(x)))
      stop("All values needed for the mean are NA after `na.rm = TRUE`.", call. = FALSE)
    m
  }

  # numeric zero tolerance
  tol <- .Machine$double.eps * 16
  check_den <- function(den, label) {
    if (is.na(den)) stop(sprintf("Denominator is NA for %s.", label), call. = FALSE)
    if (abs(den) <= tol) stop(sprintf("Zero (or near-zero) denominator for %s.", label), call. = FALSE)
    den
  }

  # ---- compute rb by method ----
  rb_value <- switch(
    method,
    "annual" = {
      den <- check_den(b_index[2], "`annual` (b_index[2])")
      num <- b_index[1] - b_index[2]
      if (na.rm && is.na(num)) return(NA_real_)
      num / den
    },
    "1over2" = {
      m23 <- safe_mean(b_index[2:3], na.rm = na.rm)
      den <- check_den(m23, "`1over2` (mean of b_index[2:3])")
      num <- b_index[1] - m23
      num / den
    },
    "2over3" = {
      m12 <- safe_mean(b_index[1:2], na.rm = na.rm)
      m35 <- safe_mean(b_index[3:5], na.rm = na.rm)
      den <- check_den(m35, "`2over3` (mean of b_index[3:5])")
      num <- m12 - m35
      num / den
    }
  )

  if (!is.null(digits)) rb_value <- round(rb_value, digits)
  rb_value
}
