#' Identify and Analyze Network Components
#'
#' Detects connected components in a citation network and computes summary statistics
#' for each component. Returns both the component information and an updated network
#' with component labels.
#'
#' @param net A network object (tbl_graph or igraph) generated by `sniff_network()`
#'
#' @return A list with two elements:
#' \describe{
#'   \item{components}{A tibble with component statistics containing:
#'     \itemize{
#'       \item \code{component}: Component identifier (e.g., "c1")
#'       \item \code{quantity_publications}: Number of publications in component
#'       \item \code{average_age}: Mean publication year of component
#'     }
#'   }
#'   \item{network}{The input network with added component labels}
#' }
#'
#' @importFrom rlang .data
#' @importFrom utils globalVariables
#' @importFrom igraph V vertex_attr_names components
#' @importFrom dplyr n desc
#' @importFrom tidygraph activate as_tbl_graph
#' @export
#'
#' @examples
#' \dontrun{
#' # Create a network first
#' data <- read_wos("savedrecs.txt")
#' net <- sniff_network(data)
#'
#' # Analyze components
#' result <- sniff_components(net)
#'
#' # Access component information
#' result$components
#'
#' # Get network with component labels
#' component_net <- result$network
#' }
sniff_components <- function(net) {

  # Input validation
  if (!inherits(net, c("tbl_graph", "igraph"))) {
    stop("Input must be a network object (tbl_graph or igraph)", call. = FALSE)
  }

  if (!"PY" %in% igraph::vertex_attr_names(net)) {
    stop("Network must contain 'PY' (publication year) vertex attribute", call. = FALSE)
  }

  # Calculate components
  cfg <- igraph::components(net)
  igraph::V(net)$component <- cfg$membership

  tryCatch(
    {
      net |>
        tidygraph::activate(nodes) |>
        tibble::as_tibble() |>
        dplyr::mutate(PY = as.numeric(.data$PY)) |>
        dplyr::select(.data$PY, .data$component) |>
        dplyr::group_by(.data$component) |>
        dplyr::summarise(
          qtde_publi = dplyr::n(),
          idade_media = mean(.data$PY, na.rm = TRUE)
        ) |>
        dplyr::arrange(dplyr::desc(.data$qtde_publi)) |>
        dplyr::mutate(
          component_name = paste0("c", 1:dplyr::n())
        ) ->
        componentes

      componentes |>
        dplyr::select(.data$component, .data$component_name) ->
        componentes_v2

      net |>
        tidygraph::as_tbl_graph() |>
        tidygraph::activate("nodes") |>
        dplyr::left_join(componentes_v2, by = "component") |>
        dplyr::select(-.data$component) |>
        dplyr::rename(component = .data$component_name) ->
        net_comp

      componentes |>
        dplyr::select(-.data$component) |>
        dplyr::select(
          component = .data$component_name,
          quantity_publications = .data$qtde_publi,
          average_age = .data$idade_media
        ) ->
        componentes_v3

      list(components = componentes_v3, network = net_comp)
    },
    error = function(e) {
      stop("Error analyzing components: ", e$message, call. = FALSE)
    }
  )
}
