% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/acp.R
\name{acp}
\alias{acp}
\alias{print.acp}
\alias{summary.acp}
\alias{print.summary.acp}
\title{Adaptive conformal prediction method}
\usage{
acp(
  object,
  alpha = 1 - 0.01 * object$level,
  gamma = 0.005,
  symmetric = FALSE,
  ncal = 10,
  rolling = FALSE,
  quantiletype = 1,
  update = FALSE,
  na.rm = TRUE,
  ...
)
}
\arguments{
\item{object}{An object of class \code{"cvforecast"}. It must have an argument
\code{x} for original univariate time series, an argument \code{MEAN} for
point forecasts and \code{ERROR} for forecast errors on validation set.
See the results of a call to \code{\link{cvforecast}}.}

\item{alpha}{A numeric vector of significance levels to achieve a desired
coverage level \eqn{1-\alpha}.}

\item{gamma}{The step size parameter \eqn{\gamma>0} for \eqn{\alpha} updating.}

\item{symmetric}{If \code{TRUE}, symmetric nonconformity scores (i.e. \eqn{|e_{t+h|t}|})
are used. If \code{FALSE}, asymmetric nonconformity scores (i.e. \eqn{e_{t+h|t}})
are used, and then upper bounds and lower bounds are produced separately.}

\item{ncal}{Length of the calibration set. If \code{rolling = FALSE}, it denotes
the initial period of calibration sets. Otherwise, it indicates
the period of every rolling calibration set.}

\item{rolling}{If \code{TRUE}, a rolling window strategy will be adopted to
form the calibration set. Otherwise, expanding window strategy will be used.}

\item{quantiletype}{An integer between 1 and 9 determining the type of
quantile estimator to be used. Types 1 to 3 are for discontinuous quantiles,
types 4 to 9 are for continuous quantiles. See the
\code{\link[ggdist]{weighted_quantile}} function in the ggdist package.}

\item{update}{If \code{TRUE}, the function will be compatible with the
\code{update}(\link{update.cpforecast}) function, allowing for easy updates of conformal prediction.}

\item{na.rm}{If \code{TRUE}, corresponding entries in sample values are removed
if it is \code{NA} when calculating sample quantile.}

\item{...}{Other arguments are passed to the
\code{\link[ggdist]{weighted_quantile}} function for quantile computation.}
}
\value{
A list of class \code{c("acp", "cpforecast", "forecast")}
with the following components:
\item{x}{The original time series.}
\item{series}{The name of the series \code{x}.}
\item{method}{A character string "acp".}
\item{cp_times}{The number of times the conformal prediction is performed in
cross-validation.}
\item{MEAN}{Point forecasts as a multivariate time series, where the \eqn{h}th column
holds the point forecasts for forecast horizon \eqn{h}. The time index
corresponds to the period for which the forecast is produced.}
\item{ERROR}{Forecast errors given by
\eqn{e_{t+h|t} = y_{t+h}-\hat{y}_{t+h|t}}{e[t+h] = y[t+h]-f[t+h]}.}
\item{LOWER}{A list containing lower bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{UPPER}{A list containing upper bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{level}{The confidence values associated with the prediction intervals.}
\item{call}{The matched call.}
\item{model}{A list containing information abouth the conformal prediction model.}
If \code{mean} is included in the \code{object}, the components \code{mean},
\code{lower}, and \code{upper} will also be returned, showing the information
about the forecasts generated using all available observations.
}
\description{
Compute prediction intervals and other information by
applying the adaptive conformal prediction (ACP) method.
}
\details{
The ACP method considers the online update:
\deqn{\alpha_{t+h|t}:=\alpha_{t+h-1|t-1}+\gamma(\alpha-\mathrm{err}_{t|t-h}),}
for each individual forecast horizon \code{h}, respectively,
where \eqn{\mathrm{err}_{t|t-h}=1} if \eqn{s_{t|t-h}>q_{t|t-h}}, and
\eqn{\mathrm{err}_{t|t-h}=0} if \eqn{s_{t|t-h} \leq q_{t|t-h}}.
}
\examples{
# Simulate time series from an AR(2) model
library(forecast)
series <- arima.sim(n = 200, list(ar = c(0.8, -0.5)), sd = sqrt(1))

# Cross-validation forecasting
far2 <- function(x, h, level) {
  Arima(x, order = c(2, 0, 0)) |>
    forecast(h = h, level)
}
fc <- cvforecast(series, forecastfun = far2, h = 3, level = 95,
                 forward = TRUE, initial = 1, window = 50)

# ACP with asymmetric nonconformity scores and rolling calibration sets
acpfc <- acp(fc, symmetric = FALSE, gamma = 0.005, ncal = 50, rolling = TRUE)
print(acpfc)
summary(acpfc)

}
\references{
Gibbs, I., and Candes, E. (2021). "Adaptive conformal inference under
distribution shift", \emph{Advances in Neural Information Processing Systems},
\bold{34}, 1660--1672.
}
