#' IBGE subject codes lookup
#'
#' Returns a tibble with IBGE subject (theme) codes and descriptions. These
#' codes can be used to understand the thematic organization of IBGE surveys
#' and aggregates.
#'
#' The table is built-in (no API call is made) and contains all subject codes
#' currently defined by IBGE.
#'
#' @param pattern Optional character string. A regular expression used to
#'   filter subject descriptions. If `NULL` (default), returns the full table.
#'   Matching is case-insensitive by default.
#' @param ignore_case Logical. If `TRUE` (default), pattern matching ignores
#'   case.
#'
#' @return A [tibble][tibble::tibble] with columns:
#'   \describe{
#'     \item{id}{Integer subject code.}
#'     \item{name}{Character subject description.}
#'   }
#'
#' @examples
#' # All subjects
#' ibge_subjects()
#'
#' # Find sanitation-related themes
#' ibge_subjects("saneamento|esgoto|lixo")
#'
#' # Find themes mentioning "internet"
#' ibge_subjects("internet")
#'
#' @export
ibge_subjects <- function(pattern = NULL, ignore_case = TRUE) {
  
  subjects <- tibble::tribble(
    ~id, ~name,
    148L, "Abastecimento de \u00e1gua",
    70L, "Abate de animais",
    110L, "Acesso a esgotamento sanit\u00e1rio",
    147L, "Acesso \u00e0 internet",
    107L, "Acesso a servi\u00e7o de coleta de lixo dom\u00e9stico",
    146L, "Acesso a servi\u00e7os de telefonia",
    109L, "Acesso a sistema de abastecimento de \u00e1gua",
    316L, "Acidentes",
    221L, "Acidentes, viol\u00eancia e seguran\u00e7a",
    128L, "Adequa\u00e7\u00e3o de moradia",
    245L, "Agenda 21 Local",
    208L, "Aglomerado subnormal",
    88L, "Agricultura Familiar",
    78L, "Agroind\u00fastria rural",
    296L, "\u00c1gua pot\u00e1vel e saneamento",
    163L, "Alimenta\u00e7\u00e3o",
    188L, "Alojamento e/ou alimenta\u00e7\u00e3o",
    181L, "Antropometria",
    49L, "Aquisi\u00e7\u00e3o",
    230L, "Aquisi\u00e7\u00e3o de produtos aliment\u00edcios",
    97L, "\u00c1rea remanescente e desflorestamento na Mata Atl\u00e2ntica e nas forma\u00e7\u00f5es vegetais litor\u00e2neas",
    104L, "\u00c1reas protegidas",
    248L, "Articula\u00e7\u00f5es interinstitucionais",
    223L, "Asma",
    315L, "Aten\u00e7\u00e3o prim\u00e1ria \u00e0 sa\u00fade",
    62L, "Atividade",
    165L, "Atividade f\u00edsica",
    189L, "Atividades imobili\u00e1rias",
    133L, "Balan\u00e7a comercial",
    100L, "Balneabilidade",
    306L, "Boa sa\u00fade e bem-estar",
    310L, "Caracter\u00edsticas adicionais do mercado de trabalho",
    214L, "Caracter\u00edsticas b\u00e1sicas",
    290L, "Caracter\u00edsticas da escola",
    161L, "Caracter\u00edsticas da pessoa",
    162L, "Caracter\u00edsticas do domic\u00edlio",
    313L, "Caracter\u00edsticas do domic\u00edlio e servi\u00e7os domiciliares de sa\u00fade",
    317L, "Caracter\u00edsticas do trabalho e apoio social",
    43L, "Caracter\u00edsticas dos estabelecimentos agropecu\u00e1rios",
    44L, "Caracter\u00edsticas dos produtores agropecu\u00e1rios",
    175L, "Caracter\u00edsticas gerais da popula\u00e7\u00e3o",
    210L, "Caracter\u00edsticas gerais dos ind\u00edgenas",
    4L, "Casamentos",
    297L, "Cidades e comunidades sustent\u00e1veis",
    314L, "Cobertura de plano de sa\u00fade",
    129L, "Coeficiente de mortalidade por homic\u00eddios ou por acidentes de transporte",
    141L, "Coleta seletiva de lixo",
    83L, "Com\u00e9rcio",
    243L, "Comit\u00eas de Bacias Hidrogr\u00e1ficas",
    218L, "Comportamento alimentar",
    92L, "Concentra\u00e7\u00e3o de poluentes no ar em \u00e1reas urbanas",
    309L, "Confedera\u00e7\u00e3o da Agricultura e Pecu\u00e1ria do Brasil (CNA)",
    79L, "Constru\u00e7\u00e3o civil",
    51L, "Consumo alimentar",
    135L, "Consumo de energia per capita",
    138L, "Consumo mineral per capita",
    81L, "Contas Nacionais e Regionais",
    217L, "Contexto familiar",
    216L, "Contexto socioecon\u00f4mico",
    59L, "Cor ou ra\u00e7a",
    184L, "Correio e/ou telecomunica\u00e7\u00f5es",
    71L, "Couro",
    330L, "Covid-19",
    277L, "Crian\u00e7as com menos de 2 anos",
    60L, "Defici\u00eancia",
    80L, "Desenvolvimento sustent\u00e1vel",
    98L, "Desertifica\u00e7\u00e3o e areniza\u00e7\u00e3o",
    96L, "Desflorestamento na Amaz\u00f4nia Legal",
    213L, "Deslocamento",
    207L, "Desmatamento do Cerrado",
    249L, "Desmatamento nos Biomas",
    50L, "Despesa",
    108L, "Destina\u00e7\u00e3o final do lixo",
    6L, "Div\u00f3rcios",
    268L, "Doen\u00e7as cr\u00f4nicas",
    124L, "Doen\u00e7as relacionadas ao saneamento ambiental inadequado",
    318L, "Doen\u00e7as transmiss\u00edveis",
    20L, "Domic\u00edlios",
    27L, "Educa\u00e7\u00e3o",
    302L, "Educa\u00e7\u00e3o de qualidade",
    156L, "Emigra\u00e7\u00e3o",
    91L, "Emiss\u00f5es de origem antr\u00f3pica dos gases associados ao efeito estufa",
    173L, "Emprego",
    198L, "Empresarial n\u00e3o-financeiro",
    12L, "Empresas",
    54L, "Empresas e outras organiza\u00e7\u00f5es",
    298L, "Energia limpa e acess\u00edvel",
    186L, "Engenharia e arquitetura",
    41L, "Entidades sem fins lucrativos",
    229L, "Equipamentos de ensino",
    304L, "Erradica\u00e7\u00e3o da pobreza",
    127L, "Escolaridade",
    215L, "Escolaridade materna e paterna",
    149L, "Esgotamento sanit\u00e1rio",
    103L, "Esp\u00e9cies extintas e amea\u00e7adas de extin\u00e7\u00e3o",
    106L, "Esp\u00e9cies invasoras",
    119L, "Esperan\u00e7a de vida ao nascer",
    85L, "Estoques agropecu\u00e1rios",
    157L, "Etnia",
    143L, "Exist\u00eancia de conselhos municipais",
    75L, "Extra\u00e7\u00e3o vegetal",
    331L, "Faces de logradouros",
    19L, "Fam\u00edlias",
    18L, "Fecundidade",
    77L, "Floricultura",
    58L, "Fome zero e agricultura sustent\u00e1vel",
    235L, "Frequ\u00eancia escolar",
    42L, "Funda\u00e7\u00f5es e Associa\u00e7\u00f5es sem fins lucrativos",
    320L, "Furto e roubo",
    145L, "Gasto p\u00fablico com prote\u00e7\u00e3o ao meio ambiente",
    144L, "Gastos com Pesquisa e Desenvolvimento - P D",
    295L, "Geral",
    153L, "Gest\u00e3o municipal de saneamento b\u00e1sico",
    134L, "Grau de endividamento",
    222L, "Higiene e sa\u00fade bucal",
    76L, "Horticultura",
    209L, "Hospedagem",
    273L, "Idosos",
    130L, "Igualdade de g\u00eanero",
    293L, "Imagem corporal",
    224L, "Imagem corporal, atitude em rela\u00e7\u00e3o ao peso corporal e estado nutricional",
    122L, "Imuniza\u00e7\u00e3o contra doen\u00e7as infecciosas infantis",
    234L, "Incid\u00eancia de AIDS",
    254L, "Indicadores demogr\u00e1ficos",
    115L, "\u00cdndice de Gini da distribui\u00e7\u00e3o do rendimento",
    200L, "\u00cdndice de Pre\u00e7os ao Produtor",
    82L, "\u00cdndices de pre\u00e7os",
    87L, "Ind\u00fastria",
    303L, "Ind\u00fastria, inova\u00e7\u00e3o e infraestrutura",
    195L, "Informa\u00e7\u00e3o e comunica\u00e7\u00e3o",
    86L, "Informa\u00e7\u00f5es b\u00e1sicas municipais",
    291L, "Informa\u00e7\u00f5es gerais",
    183L, "Inform\u00e1tica",
    179L, "Instru\u00e7\u00e3o e Educa\u00e7\u00e3o",
    136L, "Intensidade energ\u00e9tica",
    201L, "Internet",
    24L, "Lavouras permanentes",
    23L, "Lavouras tempor\u00e1rias",
    25L, "Lavouras tempor\u00e1rias e permanentes",
    72L, "Leite",
    151L, "Limpeza urbana e Coleta de lixo",
    158L, "L\u00edngua falada",
    154L, "Manejo de \u00e1guas pluviais",
    155L, "Manejo de res\u00edduos s\u00f3lidos",
    197L, "Manuten\u00e7\u00e3o e repara\u00e7\u00e3o",
    274L, "Meio ambiente",
    329L, "Mercado de trabalho",
    15L, "Migra\u00e7\u00e3o",
    61L, "Migra\u00e7\u00e3o e deslocamento",
    160L, "Mortalidade",
    152L, "Movimenta\u00e7\u00e3o financeira",
    233L, "Mulheres em trabalhos formais",
    1L, "Nascidos vivos",
    65L, "Nupcialidade",
    177L, "Nupcialidade e fecundidade",
    2L, "\u00d3bitos",
    3L, "\u00d3bitos fetais",
    123L, "Oferta de servi\u00e7os b\u00e1sicos de sa\u00fade",
    244L, "Organiza\u00e7\u00f5es da Sociedade Civil",
    191L, "Outras atividades de servi\u00e7os",
    308L, "Outras formas de trabalho",
    73L, "Ovos de galinha",
    299L, "Parcerias e meios de implementa\u00e7\u00e3o",
    137L, "Participa\u00e7\u00e3o de fontes renov\u00e1veis na oferta de energia",
    319L, "Paternidade e pr\u00e9-natal do parceiro",
    275L, "Patrim\u00f4nio cultural",
    300L, "Paz, justi\u00e7a e institui\u00e7\u00f5es eficazes",
    40L, "Pecu\u00e1ria",
    269L, "Percep\u00e7\u00e3o do estado de sa\u00fade",
    45L, "Pessoal ocupado",
    52L, "Pessoas",
    262L, "Popula\u00e7\u00e3o desocupada",
    114L, "Popula\u00e7\u00e3o e terras ind\u00edgenas",
    327L, "Popula\u00e7\u00e3o ind\u00edgena",
    328L, "Popula\u00e7\u00e3o quilombola",
    102L, "Popula\u00e7\u00e3o residente em \u00e1reas costeiras",
    276L, "Pr\u00e9-natal",
    121L, "Preval\u00eancia de desnutri\u00e7\u00e3o total",
    101L, "Produ\u00e7\u00e3o de pescado mar\u00edtima e continental",
    47L, "Produ\u00e7\u00e3o vegetal",
    131L, "Produto Interno Bruto per capita",
    66L, "Programas Sociais do Governo",
    187L, "Publicidade",
    99L, "Qualidade das \u00e1guas interiores",
    95L, "Queimadas e inc\u00eandios florestais",
    227L, "Raz\u00e3o de depend\u00eancia",
    192L, "Receitas",
    140L, "Reciclagem",
    46L, "Redu\u00e7\u00e3o das desigualdades",
    253L, "Registro civil",
    159L, "Registro de nascimento",
    142L, "Rejeitos radioativos: gera\u00e7\u00e3o e armazenamento",
    57L, "Religi\u00e3o",
    28L, "Rendimento",
    312L, "Rendimento de todas as fontes",
    228L, "Rendimento domiciliar per capita",
    117L, "Rendimento familiar per capita",
    118L, "Rendimento m\u00e9dio mensal",
    53L, "Respons\u00e1vel pelo domic\u00edlio/fam\u00edlia",
    67L, "Sa\u00fade",
    272L, "Sa\u00fade da mulher",
    225L, "Sa\u00fade mental",
    219L, "Sa\u00fade sexual e reprodutiva",
    172L, "Seguran\u00e7a",
    68L, "Seguran\u00e7a alimentar",
    321L, "Sensa\u00e7\u00e3o de seguran\u00e7a",
    5L, "Separa\u00e7\u00f5es",
    84L, "Servi\u00e7os",
    185L, "Servi\u00e7os audiovisuais",
    193L, "Servi\u00e7os empresariais n\u00e3o-financeiros",
    190L, "Servi\u00e7os prestados \u00e0s empresas",
    194L, "Servi\u00e7os prestados \u00e0s fam\u00edlias",
    196L, "Servi\u00e7os profissionais, administrativos e complementares",
    74L, "Silvicultura",
    292L, "Situa\u00e7\u00f5es em casa e na escola",
    232L, "Tabaco na escola",
    203L, "Tabagismo",
    126L, "Taxa de alfabetiza\u00e7\u00e3o",
    112L, "Taxa de crescimento da popula\u00e7\u00e3o",
    116L, "Taxa de desocupa\u00e7\u00e3o",
    125L, "Taxa de escolariza\u00e7\u00e3o",
    113L, "Taxa de fecundidade",
    132L, "Taxa de investimento",
    120L, "Taxa de mortalidade infantil",
    180L, "Tecnologia da Informa\u00e7\u00e3o",
    212L, "Tecnologia de informa\u00e7\u00e3o e comunica\u00e7\u00e3o",
    322L, "Teletrabalho",
    94L, "Terras em uso agrossilvipastoril",
    211L, "Terras ind\u00edgenas",
    63L, "Territ\u00f3rio",
    326L, "Territ\u00f3rios quilombolas",
    17L, "Trabalho",
    325L, "Trabalho de crian\u00e7as e adolescentes",
    301L, "Trabalho decente e crescimento econ\u00f4mico",
    176L, "Trabalho e rendimento",
    199L, "Trabalho infantil",
    323L, "Trabalho por meio de plataformas digitais",
    105L, "Tr\u00e1fico, cria\u00e7\u00e3o e com\u00e9rcio de animais silvestres",
    182L, "Transportes",
    111L, "Tratamento de esgoto",
    311L, "Turismo",
    13L, "Unidades locais",
    93L, "Uso de agrot\u00f3xicos",
    168L, "Uso de \u00e1lcool",
    167L, "Uso de cigarro",
    169L, "Uso de drogas",
    90L, "Uso de fertilizantes",
    226L, "Uso de servi\u00e7o de sa\u00fade",
    69L, "Valores monet\u00e1rios na agropecu\u00e1ria",
    305L, "Vida na \u00e1gua",
    170L, "Vida sexual",
    307L, "Vida terrestre",
    139L, "Vida \u00fatil das reservas minerais",
    171L, "Viol\u00eancia"
  )
  
  if (!is.null(pattern) && nzchar(pattern)) {
    subjects <- subjects[grepl(pattern, subjects$name, ignore.case = ignore_case), ]
    cli::cli_alert_success("{nrow(subjects)} subject{?s} matched pattern {.val {pattern}}.")
  } else {
    cli::cli_alert_success("{nrow(subjects)} subjects loaded.")
  }
  
  subjects
}