#' Fit Fairfield-Smith's variance law to matrix data with ggplot2 plots
#'
#' This function fits the Fairfield-Smith variance law, computes weighted R^2 on
#' the log-log scale, identifies the optimum plot size and recommended shape,
#' and produces two ggplot2 visualisations (original and log scale).
#'
#' @param df_mat numeric matrix of data
#' @param plot_curve logical, if TRUE returns ggplot objects
#' @return list with results:
#' \itemize{
#'   \item df_shapes: data.frame of plot shapes
#'   \item V1: variance at 1x1
#'   \item b_hat: estimated variance law coefficient
#'   \item R2_log: weighted R^2 on log-log scale
#'   \item x_opt: optimum plot size (units)
#'   \item Vx_opt: predicted variance at optimum
#'   \item best_shape: recommended shape for optimum plot size
#'   \item plots: list of ggplot objects if plot_curve = TRUE
#' }
#' @importFrom stats aggregate
#' @import ggplot2
#' @export
fit_variance_law <- function(df_mat, plot_curve = TRUE) {
  df_shapes <- generate_plot_shapes(df_mat)

  avg_var <- aggregate(
    V_per_unit_area ~ plot_size_units,
    data = df_shapes,
    FUN = mean
  )
  counts <- aggregate(plot_size_units ~ plot_size_units, data = df_shapes, FUN = length)
  avg_var$w <- counts$plot_size_units
  colnames(avg_var)[2] <- "Vx"

  if (1 %in% avg_var$plot_size_units) {
    V1 <- avg_var$Vx[avg_var$plot_size_units == 1]
  } else {
    V1 <- min(avg_var$Vx)
  }

  avg_var$X <- log(avg_var$plot_size_units)
  avg_var$Y <- log(avg_var$Vx) - log(V1)

  c_hat <- sum(avg_var$w * avg_var$X * avg_var$Y) / sum(avg_var$w * avg_var$X^2)
  b_hat <- -c_hat

  variance_law <- function(x, V1, b_hat) {
    V1 / (x^b_hat)
  }

  avg_var$Fitted <- variance_law(avg_var$plot_size_units, V1, b_hat)

  Y_obs_log <- log(avg_var$Vx)
  Y_fit_log <- log(avg_var$Fitted)
  w <- avg_var$w
  y_bar_log <- sum(w * Y_obs_log) / sum(w)
  SST_log <- sum(w * (Y_obs_log - y_bar_log)^2)
  SSR_log <- sum(w * (Y_obs_log - Y_fit_log)^2)
  R2_log <- 1 - SSR_log / SST_log

  x_opt <- avg_var$plot_size_units[which.min(avg_var$Fitted)]
  Vx_opt <- min(avg_var$Fitted)
  best_shape <- df_shapes[df_shapes$plot_size_units == x_opt, ]
  best_shape <- best_shape[order(best_shape$V_per_unit_area), ][1, ]

  plots <- NULL
  if (plot_curve) {
    df_long <- data.frame(
      plot_size_units = rep(avg_var$plot_size_units, 2),
      Variance = c(avg_var$Vx, avg_var$Fitted),
      Type = rep(c("Vx", "Fitted"), each = nrow(avg_var))
    )

    p1 <- ggplot(df_long, aes(x = plot_size_units, y = Variance,
                              color = Type, shape = Type)) +
      geom_point(size = 3) +
      geom_line() +
      geom_vline(xintercept = x_opt, linetype = "dashed", color = "darkgreen") +
      geom_text(aes(x = x_opt, y = max(Variance)*0.95,
                    label = paste0("Optimum = ", x_opt)),
                color = "darkgreen", angle = 90, vjust = -0.5) +
      labs(
        title = "Observed vs Fitted (Original Scale)",
        x = "Plot size (units)",
        y = "Variance per unit area (Vx)"
      ) +
      theme_minimal()

    # Log-log scale plot
    df_long_log <- df_long
    df_long_log$log_x <- log(df_long_log$plot_size_units)
    df_long_log$log_y <- log(df_long_log$Variance)
    p2 <- ggplot(df_long_log, aes(x = log_x, y = log_y,
                                  color = Type, shape = Type)) +
      geom_point(size = 3) +
      geom_line() +
      labs(
        title = paste0("Observed vs Fitted (Log-Log Scale)\nWeighted R^2 = ",
                       round(R2_log, 3)),
        x = "log(Plot size)",
        y = "log(Variance per unit area)"
      ) +
      theme_minimal()

    plots <- list(original = p1, loglog = p2)
  }

  return(list(
    df_shapes = df_shapes,
    V1 = V1,
    b_hat = b_hat,
    R2_log = R2_log,
    x_opt = x_opt,
    Vx_opt = Vx_opt,
    best_shape = best_shape,
    plots = plots
  ))
}
