"""
The `ast` module helps Python applications to process trees of the Python
abstract syntax grammar.  The abstract syntax itself might change with
each Python release; this module helps to find out programmatically what
the current grammar looks like and allows modifications of it.

An abstract syntax tree can be generated by passing `ast.PyCF_ONLY_AST` as
a flag to the `compile()` builtin function or by using the `parse()`
function from this module.  The result will be a tree of objects whose
classes all inherit from `ast.AST`.

A modified abstract syntax tree can be compiled into a Python code object
using the built-in `compile()` function.

Additionally various helper functions are provided that make working with
the trees simpler.  The main intention of the helper functions and this
module in general is to provide an easy to use interface for libraries
that work tightly with the python syntax (template engines for example).

:copyright: Copyright 2008 by Armin Ronacher.
:license: Python License.
"""

import ast
import builtins
import os
import sys
import typing_extensions
from _ast import (
    PyCF_ALLOW_TOP_LEVEL_AWAIT as PyCF_ALLOW_TOP_LEVEL_AWAIT,
    PyCF_ONLY_AST as PyCF_ONLY_AST,
    PyCF_TYPE_COMMENTS as PyCF_TYPE_COMMENTS,
)
from _typeshed import ReadableBuffer, Unused
from collections.abc import Iterable, Iterator, Sequence
from typing import Any, ClassVar, Generic, Literal, TypedDict, TypeVar as _TypeVar, overload, type_check_only
from typing_extensions import Self, Unpack, deprecated, disjoint_base

if sys.version_info >= (3, 13):
    from _ast import PyCF_OPTIMIZED_AST as PyCF_OPTIMIZED_AST

# Used for node end positions in constructor keyword arguments
_EndPositionT = typing_extensions.TypeVar("_EndPositionT", int, int | None, default=int | None)

# Corresponds to the names in the `_attributes` class variable which is non-empty in certain AST nodes
@type_check_only
class _Attributes(TypedDict, Generic[_EndPositionT], total=False):
    lineno: int
    col_offset: int
    end_lineno: _EndPositionT
    end_col_offset: _EndPositionT

# The various AST classes are implemented in C, and imported from _ast at runtime,
# but they consider themselves to live in the ast module,
# so we'll define the stubs in this file.
if sys.version_info >= (3, 12):
    @disjoint_base
    class AST:
        __match_args__ = ()
        _attributes: ClassVar[tuple[str, ...]]
        _fields: ClassVar[tuple[str, ...]]
        if sys.version_info >= (3, 13):
            _field_types: ClassVar[dict[str, Any]]

        if sys.version_info >= (3, 14):
            def __replace__(self) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

else:
    class AST:
        if sys.version_info >= (3, 10):
            __match_args__ = ()
        _attributes: ClassVar[tuple[str, ...]]
        _fields: ClassVar[tuple[str, ...]]

class mod(AST):
    """mod = Module(stmt* body, type_ignore* type_ignores)
    | Interactive(stmt* body)
    | Expression(expr body)
    | FunctionType(expr* argtypes, expr returns)
    """

class Module(mod):
    """Module(stmt* body, type_ignore* type_ignores)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("body", "type_ignores")
    body: list[stmt]
    type_ignores: list[TypeIgnore]
    if sys.version_info >= (3, 13):
        def __init__(self, body: list[stmt] = ..., type_ignores: list[TypeIgnore] = ...) -> None: ...
    else:
        def __init__(self, body: list[stmt], type_ignores: list[TypeIgnore]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, body: list[stmt] = ..., type_ignores: list[TypeIgnore] = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Interactive(mod):
    """Interactive(stmt* body)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("body",)
    body: list[stmt]
    if sys.version_info >= (3, 13):
        def __init__(self, body: list[stmt] = ...) -> None: ...
    else:
        def __init__(self, body: list[stmt]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, body: list[stmt] = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Expression(mod):
    """Expression(expr body)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("body",)
    body: expr
    def __init__(self, body: expr) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, body: expr = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class FunctionType(mod):
    """FunctionType(expr* argtypes, expr returns)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("argtypes", "returns")
    argtypes: list[expr]
    returns: expr
    if sys.version_info >= (3, 13):
        @overload
        def __init__(self, argtypes: list[expr], returns: expr) -> None: ...
        @overload
        def __init__(self, argtypes: list[expr] = ..., *, returns: expr) -> None: ...
    else:
        def __init__(self, argtypes: list[expr], returns: expr) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, argtypes: list[expr] = ..., returns: expr = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class stmt(AST):
    """stmt = FunctionDef(identifier name, arguments args, stmt* body, expr* decorator_list, expr? returns, string? type_comment, type_param* type_params)
    | AsyncFunctionDef(identifier name, arguments args, stmt* body, expr* decorator_list, expr? returns, string? type_comment, type_param* type_params)
    | ClassDef(identifier name, expr* bases, keyword* keywords, stmt* body, expr* decorator_list, type_param* type_params)
    | Return(expr? value)
    | Delete(expr* targets)
    | Assign(expr* targets, expr value, string? type_comment)
    | TypeAlias(expr name, type_param* type_params, expr value)
    | AugAssign(expr target, operator op, expr value)
    | AnnAssign(expr target, expr annotation, expr? value, int simple)
    | For(expr target, expr iter, stmt* body, stmt* orelse, string? type_comment)
    | AsyncFor(expr target, expr iter, stmt* body, stmt* orelse, string? type_comment)
    | While(expr test, stmt* body, stmt* orelse)
    | If(expr test, stmt* body, stmt* orelse)
    | With(withitem* items, stmt* body, string? type_comment)
    | AsyncWith(withitem* items, stmt* body, string? type_comment)
    | Match(expr subject, match_case* cases)
    | Raise(expr? exc, expr? cause)
    | Try(stmt* body, excepthandler* handlers, stmt* orelse, stmt* finalbody)
    | TryStar(stmt* body, excepthandler* handlers, stmt* orelse, stmt* finalbody)
    | Assert(expr test, expr? msg)
    | Import(alias* names)
    | ImportFrom(identifier? module, alias* names, int? level)
    | Global(identifier* names)
    | Nonlocal(identifier* names)
    | Expr(expr value)
    | Pass
    | Break
    | Continue
    """

    lineno: int
    col_offset: int
    end_lineno: int | None
    end_col_offset: int | None
    def __init__(self, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class FunctionDef(stmt):
    """FunctionDef(identifier name, arguments args, stmt* body, expr* decorator_list, expr? returns, string? type_comment, type_param* type_params)"""

    if sys.version_info >= (3, 12):
        __match_args__ = ("name", "args", "body", "decorator_list", "returns", "type_comment", "type_params")
    elif sys.version_info >= (3, 10):
        __match_args__ = ("name", "args", "body", "decorator_list", "returns", "type_comment")
    name: str
    args: arguments
    body: list[stmt]
    decorator_list: list[expr]
    returns: expr | None
    type_comment: str | None
    if sys.version_info >= (3, 12):
        type_params: list[type_param]
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            returns: expr | None = None,
            type_comment: str | None = None,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    elif sys.version_info >= (3, 12):
        @overload
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None,
            type_comment: str | None,
            type_params: list[type_param],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
        @overload
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None = None,
            type_comment: str | None = None,
            *,
            type_params: list[type_param],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None = None,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            name: str = ...,
            args: arguments = ...,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            returns: expr | None = ...,
            type_comment: str | None = ...,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class AsyncFunctionDef(stmt):
    """AsyncFunctionDef(identifier name, arguments args, stmt* body, expr* decorator_list, expr? returns, string? type_comment, type_param* type_params)"""

    if sys.version_info >= (3, 12):
        __match_args__ = ("name", "args", "body", "decorator_list", "returns", "type_comment", "type_params")
    elif sys.version_info >= (3, 10):
        __match_args__ = ("name", "args", "body", "decorator_list", "returns", "type_comment")
    name: str
    args: arguments
    body: list[stmt]
    decorator_list: list[expr]
    returns: expr | None
    type_comment: str | None
    if sys.version_info >= (3, 12):
        type_params: list[type_param]
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            returns: expr | None = None,
            type_comment: str | None = None,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    elif sys.version_info >= (3, 12):
        @overload
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None,
            type_comment: str | None,
            type_params: list[type_param],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
        @overload
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None = None,
            type_comment: str | None = None,
            *,
            type_params: list[type_param],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            name: str,
            args: arguments,
            body: list[stmt],
            decorator_list: list[expr],
            returns: expr | None = None,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            name: str = ...,
            args: arguments = ...,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            returns: expr | None = ...,
            type_comment: str | None = ...,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class ClassDef(stmt):
    """ClassDef(identifier name, expr* bases, keyword* keywords, stmt* body, expr* decorator_list, type_param* type_params)"""

    if sys.version_info >= (3, 12):
        __match_args__ = ("name", "bases", "keywords", "body", "decorator_list", "type_params")
    elif sys.version_info >= (3, 10):
        __match_args__ = ("name", "bases", "keywords", "body", "decorator_list")
    name: str
    bases: list[expr]
    keywords: list[keyword]
    body: list[stmt]
    decorator_list: list[expr]
    if sys.version_info >= (3, 12):
        type_params: list[type_param]
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            name: str,
            bases: list[expr] = ...,
            keywords: list[keyword] = ...,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    elif sys.version_info >= (3, 12):
        def __init__(
            self,
            name: str,
            bases: list[expr],
            keywords: list[keyword],
            body: list[stmt],
            decorator_list: list[expr],
            type_params: list[type_param],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            name: str,
            bases: list[expr],
            keywords: list[keyword],
            body: list[stmt],
            decorator_list: list[expr],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            name: str = ...,
            bases: list[expr] = ...,
            keywords: list[keyword] = ...,
            body: list[stmt] = ...,
            decorator_list: list[expr] = ...,
            type_params: list[type_param] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Return(stmt):
    """Return(expr? value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value",)
    value: expr | None
    def __init__(self, value: expr | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Delete(stmt):
    """Delete(expr* targets)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("targets",)
    targets: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(self, targets: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, targets: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, targets: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Assign(stmt):
    """Assign(expr* targets, expr value, string? type_comment)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("targets", "value", "type_comment")
    targets: list[expr]
    value: expr
    type_comment: str | None
    if sys.version_info >= (3, 13):
        @overload
        def __init__(
            self, targets: list[expr], value: expr, type_comment: str | None = None, **kwargs: Unpack[_Attributes]
        ) -> None: ...
        @overload
        def __init__(
            self, targets: list[expr] = ..., *, value: expr, type_comment: str | None = None, **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(
            self, targets: list[expr], value: expr, type_comment: str | None = None, **kwargs: Unpack[_Attributes]
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, targets: list[expr] = ..., value: expr = ..., type_comment: str | None = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 12):
    class TypeAlias(stmt):
        """TypeAlias(expr name, type_param* type_params, expr value)"""

        __match_args__ = ("name", "type_params", "value")
        name: Name
        type_params: list[type_param]
        value: expr
        if sys.version_info >= (3, 13):
            @overload
            def __init__(
                self, name: Name, type_params: list[type_param], value: expr, **kwargs: Unpack[_Attributes[int]]
            ) -> None: ...
            @overload
            def __init__(
                self, name: Name, type_params: list[type_param] = ..., *, value: expr, **kwargs: Unpack[_Attributes[int]]
            ) -> None: ...
        else:
            def __init__(
                self, name: Name, type_params: list[type_param], value: expr, **kwargs: Unpack[_Attributes[int]]
            ) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(  # type: ignore[override]
                self,
                *,
                name: Name = ...,
                type_params: list[type_param] = ...,
                value: expr = ...,
                **kwargs: Unpack[_Attributes[int]],
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

class AugAssign(stmt):
    """AugAssign(expr target, operator op, expr value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "op", "value")
    target: Name | Attribute | Subscript
    op: operator
    value: expr
    def __init__(
        self, target: Name | Attribute | Subscript, op: operator, value: expr, **kwargs: Unpack[_Attributes]
    ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            target: Name | Attribute | Subscript = ...,
            op: operator = ...,
            value: expr = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class AnnAssign(stmt):
    """AnnAssign(expr target, expr annotation, expr? value, int simple)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "annotation", "value", "simple")
    target: Name | Attribute | Subscript
    annotation: expr
    value: expr | None
    simple: int
    @overload
    def __init__(
        self,
        target: Name | Attribute | Subscript,
        annotation: expr,
        value: expr | None,
        simple: int,
        **kwargs: Unpack[_Attributes],
    ) -> None: ...
    @overload
    def __init__(
        self,
        target: Name | Attribute | Subscript,
        annotation: expr,
        value: expr | None = None,
        *,
        simple: int,
        **kwargs: Unpack[_Attributes],
    ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            target: Name | Attribute | Subscript = ...,
            annotation: expr = ...,
            value: expr | None = ...,
            simple: int = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class For(stmt):
    """For(expr target, expr iter, stmt* body, stmt* orelse, string? type_comment)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "iter", "body", "orelse", "type_comment")
    target: expr
    iter: expr
    body: list[stmt]
    orelse: list[stmt]
    type_comment: str | None
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            target: expr,
            iter: expr,
            body: list[stmt] = ...,
            orelse: list[stmt] = ...,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            target: expr,
            iter: expr,
            body: list[stmt],
            orelse: list[stmt],
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            target: expr = ...,
            iter: expr = ...,
            body: list[stmt] = ...,
            orelse: list[stmt] = ...,
            type_comment: str | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class AsyncFor(stmt):
    """AsyncFor(expr target, expr iter, stmt* body, stmt* orelse, string? type_comment)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "iter", "body", "orelse", "type_comment")
    target: expr
    iter: expr
    body: list[stmt]
    orelse: list[stmt]
    type_comment: str | None
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            target: expr,
            iter: expr,
            body: list[stmt] = ...,
            orelse: list[stmt] = ...,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            target: expr,
            iter: expr,
            body: list[stmt],
            orelse: list[stmt],
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            target: expr = ...,
            iter: expr = ...,
            body: list[stmt] = ...,
            orelse: list[stmt] = ...,
            type_comment: str | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class While(stmt):
    """While(expr test, stmt* body, stmt* orelse)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("test", "body", "orelse")
    test: expr
    body: list[stmt]
    orelse: list[stmt]
    if sys.version_info >= (3, 13):
        def __init__(
            self, test: expr, body: list[stmt] = ..., orelse: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(self, test: expr, body: list[stmt], orelse: list[stmt], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, test: expr = ..., body: list[stmt] = ..., orelse: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class If(stmt):
    """If(expr test, stmt* body, stmt* orelse)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("test", "body", "orelse")
    test: expr
    body: list[stmt]
    orelse: list[stmt]
    if sys.version_info >= (3, 13):
        def __init__(
            self, test: expr, body: list[stmt] = ..., orelse: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(self, test: expr, body: list[stmt], orelse: list[stmt], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, test: expr = ..., body: list[stmt] = ..., orelse: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class With(stmt):
    """With(withitem* items, stmt* body, string? type_comment)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("items", "body", "type_comment")
    items: list[withitem]
    body: list[stmt]
    type_comment: str | None
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            items: list[withitem] = ...,
            body: list[stmt] = ...,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self, items: list[withitem], body: list[stmt], type_comment: str | None = None, **kwargs: Unpack[_Attributes]
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            items: list[withitem] = ...,
            body: list[stmt] = ...,
            type_comment: str | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class AsyncWith(stmt):
    """AsyncWith(withitem* items, stmt* body, string? type_comment)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("items", "body", "type_comment")
    items: list[withitem]
    body: list[stmt]
    type_comment: str | None
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            items: list[withitem] = ...,
            body: list[stmt] = ...,
            type_comment: str | None = None,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self, items: list[withitem], body: list[stmt], type_comment: str | None = None, **kwargs: Unpack[_Attributes]
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            items: list[withitem] = ...,
            body: list[stmt] = ...,
            type_comment: str | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Raise(stmt):
    """Raise(expr? exc, expr? cause)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("exc", "cause")
    exc: expr | None
    cause: expr | None
    def __init__(self, exc: expr | None = None, cause: expr | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, exc: expr | None = ..., cause: expr | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Try(stmt):
    """Try(stmt* body, excepthandler* handlers, stmt* orelse, stmt* finalbody)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("body", "handlers", "orelse", "finalbody")
    body: list[stmt]
    handlers: list[ExceptHandler]
    orelse: list[stmt]
    finalbody: list[stmt]
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            body: list[stmt] = ...,
            handlers: list[ExceptHandler] = ...,
            orelse: list[stmt] = ...,
            finalbody: list[stmt] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
    else:
        def __init__(
            self,
            body: list[stmt],
            handlers: list[ExceptHandler],
            orelse: list[stmt],
            finalbody: list[stmt],
            **kwargs: Unpack[_Attributes],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            body: list[stmt] = ...,
            handlers: list[ExceptHandler] = ...,
            orelse: list[stmt] = ...,
            finalbody: list[stmt] = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 11):
    class TryStar(stmt):
        """TryStar(stmt* body, excepthandler* handlers, stmt* orelse, stmt* finalbody)"""

        __match_args__ = ("body", "handlers", "orelse", "finalbody")
        body: list[stmt]
        handlers: list[ExceptHandler]
        orelse: list[stmt]
        finalbody: list[stmt]
        if sys.version_info >= (3, 13):
            def __init__(
                self,
                body: list[stmt] = ...,
                handlers: list[ExceptHandler] = ...,
                orelse: list[stmt] = ...,
                finalbody: list[stmt] = ...,
                **kwargs: Unpack[_Attributes],
            ) -> None: ...
        else:
            def __init__(
                self,
                body: list[stmt],
                handlers: list[ExceptHandler],
                orelse: list[stmt],
                finalbody: list[stmt],
                **kwargs: Unpack[_Attributes],
            ) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self,
                *,
                body: list[stmt] = ...,
                handlers: list[ExceptHandler] = ...,
                orelse: list[stmt] = ...,
                finalbody: list[stmt] = ...,
                **kwargs: Unpack[_Attributes],
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

class Assert(stmt):
    """Assert(expr test, expr? msg)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("test", "msg")
    test: expr
    msg: expr | None
    def __init__(self, test: expr, msg: expr | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, test: expr = ..., msg: expr | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Import(stmt):
    """Import(alias* names)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("names",)
    names: list[alias]
    if sys.version_info >= (3, 13):
        def __init__(self, names: list[alias] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, names: list[alias], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, names: list[alias] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class ImportFrom(stmt):
    """ImportFrom(identifier? module, alias* names, int? level)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("module", "names", "level")
    module: str | None
    names: list[alias]
    level: int
    if sys.version_info >= (3, 13):
        @overload
        def __init__(self, module: str | None, names: list[alias], level: int, **kwargs: Unpack[_Attributes]) -> None: ...
        @overload
        def __init__(
            self, module: str | None = None, names: list[alias] = ..., *, level: int, **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        @overload
        def __init__(self, module: str | None, names: list[alias], level: int, **kwargs: Unpack[_Attributes]) -> None: ...
        @overload
        def __init__(
            self, module: str | None = None, *, names: list[alias], level: int, **kwargs: Unpack[_Attributes]
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, module: str | None = ..., names: list[alias] = ..., level: int = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Global(stmt):
    """Global(identifier* names)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("names",)
    names: list[str]
    if sys.version_info >= (3, 13):
        def __init__(self, names: list[str] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, names: list[str], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, names: list[str] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Nonlocal(stmt):
    """Nonlocal(identifier* names)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("names",)
    names: list[str]
    if sys.version_info >= (3, 13):
        def __init__(self, names: list[str] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, names: list[str], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, names: list[str] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Expr(stmt):
    """Expr(expr value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value",)
    value: expr
    def __init__(self, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Pass(stmt):
    """Pass"""

class Break(stmt):
    """Break"""

class Continue(stmt):
    """Continue"""

class expr(AST):
    """expr = BoolOp(boolop op, expr* values)
    | NamedExpr(expr target, expr value)
    | BinOp(expr left, operator op, expr right)
    | UnaryOp(unaryop op, expr operand)
    | Lambda(arguments args, expr body)
    | IfExp(expr test, expr body, expr orelse)
    | Dict(expr?* keys, expr* values)
    | Set(expr* elts)
    | ListComp(expr elt, comprehension* generators)
    | SetComp(expr elt, comprehension* generators)
    | DictComp(expr key, expr value, comprehension* generators)
    | GeneratorExp(expr elt, comprehension* generators)
    | Await(expr value)
    | Yield(expr? value)
    | YieldFrom(expr value)
    | Compare(expr left, cmpop* ops, expr* comparators)
    | Call(expr func, expr* args, keyword* keywords)
    | FormattedValue(expr value, int conversion, expr? format_spec)
    | Interpolation(expr value, constant str, int conversion, expr? format_spec)
    | JoinedStr(expr* values)
    | TemplateStr(expr* values)
    | Constant(constant value, string? kind)
    | Attribute(expr value, identifier attr, expr_context ctx)
    | Subscript(expr value, expr slice, expr_context ctx)
    | Starred(expr value, expr_context ctx)
    | Name(identifier id, expr_context ctx)
    | List(expr* elts, expr_context ctx)
    | Tuple(expr* elts, expr_context ctx)
    | Slice(expr? lower, expr? upper, expr? step)
    """

    lineno: int
    col_offset: int
    end_lineno: int | None
    end_col_offset: int | None
    def __init__(self, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class BoolOp(expr):
    """BoolOp(boolop op, expr* values)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("op", "values")
    op: boolop
    values: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(self, op: boolop, values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, op: boolop, values: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, op: boolop = ..., values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class NamedExpr(expr):
    """NamedExpr(expr target, expr value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "value")
    target: Name
    value: expr
    def __init__(self, target: Name, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, target: Name = ..., value: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class BinOp(expr):
    """BinOp(expr left, operator op, expr right)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("left", "op", "right")
    left: expr
    op: operator
    right: expr
    def __init__(self, left: expr, op: operator, right: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, left: expr = ..., op: operator = ..., right: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class UnaryOp(expr):
    """UnaryOp(unaryop op, expr operand)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("op", "operand")
    op: unaryop
    operand: expr
    def __init__(self, op: unaryop, operand: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, op: unaryop = ..., operand: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Lambda(expr):
    """Lambda(arguments args, expr body)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("args", "body")
    args: arguments
    body: expr
    def __init__(self, args: arguments, body: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, args: arguments = ..., body: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class IfExp(expr):
    """IfExp(expr test, expr body, expr orelse)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("test", "body", "orelse")
    test: expr
    body: expr
    orelse: expr
    def __init__(self, test: expr, body: expr, orelse: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, test: expr = ..., body: expr = ..., orelse: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Dict(expr):
    """Dict(expr?* keys, expr* values)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("keys", "values")
    keys: list[expr | None]
    values: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(self, keys: list[expr | None] = ..., values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, keys: list[expr | None], values: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, keys: list[expr | None] = ..., values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Set(expr):
    """Set(expr* elts)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elts",)
    elts: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(self, elts: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elts: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elts: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class ListComp(expr):
    """ListComp(expr elt, comprehension* generators)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elt", "generators")
    elt: expr
    generators: list[comprehension]
    if sys.version_info >= (3, 13):
        def __init__(self, elt: expr, generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elt: expr, generators: list[comprehension], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elt: expr = ..., generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class SetComp(expr):
    """SetComp(expr elt, comprehension* generators)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elt", "generators")
    elt: expr
    generators: list[comprehension]
    if sys.version_info >= (3, 13):
        def __init__(self, elt: expr, generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elt: expr, generators: list[comprehension], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elt: expr = ..., generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class DictComp(expr):
    """DictComp(expr key, expr value, comprehension* generators)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("key", "value", "generators")
    key: expr
    value: expr
    generators: list[comprehension]
    if sys.version_info >= (3, 13):
        def __init__(
            self, key: expr, value: expr, generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(self, key: expr, value: expr, generators: list[comprehension], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, key: expr = ..., value: expr = ..., generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class GeneratorExp(expr):
    """GeneratorExp(expr elt, comprehension* generators)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elt", "generators")
    elt: expr
    generators: list[comprehension]
    if sys.version_info >= (3, 13):
        def __init__(self, elt: expr, generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elt: expr, generators: list[comprehension], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elt: expr = ..., generators: list[comprehension] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Await(expr):
    """Await(expr value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value",)
    value: expr
    def __init__(self, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Yield(expr):
    """Yield(expr? value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value",)
    value: expr | None
    def __init__(self, value: expr | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class YieldFrom(expr):
    """YieldFrom(expr value)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value",)
    value: expr
    def __init__(self, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Compare(expr):
    """Compare(expr left, cmpop* ops, expr* comparators)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("left", "ops", "comparators")
    left: expr
    ops: list[cmpop]
    comparators: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(
            self, left: expr, ops: list[cmpop] = ..., comparators: list[expr] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(self, left: expr, ops: list[cmpop], comparators: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, left: expr = ..., ops: list[cmpop] = ..., comparators: list[expr] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Call(expr):
    """Call(expr func, expr* args, keyword* keywords)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("func", "args", "keywords")
    func: expr
    args: list[expr]
    keywords: list[keyword]
    if sys.version_info >= (3, 13):
        def __init__(
            self, func: expr, args: list[expr] = ..., keywords: list[keyword] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        def __init__(self, func: expr, args: list[expr], keywords: list[keyword], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, func: expr = ..., args: list[expr] = ..., keywords: list[keyword] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class FormattedValue(expr):
    """FormattedValue(expr value, int conversion, expr? format_spec)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value", "conversion", "format_spec")
    value: expr
    conversion: int
    format_spec: expr | None
    def __init__(self, value: expr, conversion: int, format_spec: expr | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, value: expr = ..., conversion: int = ..., format_spec: expr | None = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class JoinedStr(expr):
    """JoinedStr(expr* values)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("values",)
    values: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(self, values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, values: list[expr], **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 14):
    class TemplateStr(expr):
        """TemplateStr(expr* values)"""

        __match_args__ = ("values",)
        values: list[expr]
        def __init__(self, values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
        def __replace__(self, *, values: list[expr] = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

    class Interpolation(expr):
        """Interpolation(expr value, constant str, int conversion, expr? format_spec)"""

        __match_args__ = ("value", "str", "conversion", "format_spec")
        value: expr
        str: builtins.str
        conversion: int
        format_spec: expr | None = None
        def __init__(
            self,
            value: expr = ...,
            str: builtins.str = ...,
            conversion: int = ...,
            format_spec: expr | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> None: ...
        def __replace__(
            self,
            *,
            value: expr = ...,
            str: builtins.str = ...,
            conversion: int = ...,
            format_spec: expr | None = ...,
            **kwargs: Unpack[_Attributes],
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 10):
    from types import EllipsisType

    _ConstantValue: typing_extensions.TypeAlias = str | bytes | bool | int | float | complex | None | EllipsisType
else:
    # Rely on builtins.ellipsis
    _ConstantValue: typing_extensions.TypeAlias = str | bytes | bool | int | float | complex | None | ellipsis  # noqa: F821

class Constant(expr):
    """Constant(constant value, string? kind)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value", "kind")
    value: _ConstantValue
    kind: str | None
    if sys.version_info < (3, 14):
        # Aliases for value, for backwards compatibility
        @property
        @deprecated("Removed in Python 3.14. Use `value` instead.")
        def n(self) -> _ConstantValue:
            """Deprecated. Use value instead."""

        @n.setter
        @deprecated("Removed in Python 3.14. Use `value` instead.")
        def n(self, value: _ConstantValue) -> None: ...
        @property
        @deprecated("Removed in Python 3.14. Use `value` instead.")
        def s(self) -> _ConstantValue:
            """Deprecated. Use value instead."""

        @s.setter
        @deprecated("Removed in Python 3.14. Use `value` instead.")
        def s(self, value: _ConstantValue) -> None: ...

    def __init__(self, value: _ConstantValue, kind: str | None = None, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: _ConstantValue = ..., kind: str | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Attribute(expr):
    """Attribute(expr value, identifier attr, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value", "attr", "ctx")
    value: expr
    attr: str
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    def __init__(self, value: expr, attr: str, ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, value: expr = ..., attr: str = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Subscript(expr):
    """Subscript(expr value, expr slice, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value", "slice", "ctx")
    value: expr
    slice: expr
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    def __init__(self, value: expr, slice: expr, ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, value: expr = ..., slice: expr = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Starred(expr):
    """Starred(expr value, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("value", "ctx")
    value: expr
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    def __init__(self, value: expr, ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, value: expr = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Name(expr):
    """Name(identifier id, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("id", "ctx")
    id: str
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    def __init__(self, id: str, ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, id: str = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class List(expr):
    """List(expr* elts, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elts", "ctx")
    elts: list[expr]
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    if sys.version_info >= (3, 13):
        def __init__(self, elts: list[expr] = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elts: list[expr], ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elts: list[expr] = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class Tuple(expr):
    """Tuple(expr* elts, expr_context ctx)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("elts", "ctx")
    elts: list[expr]
    ctx: expr_context  # Not present in Python < 3.13 if not passed to `__init__`
    dims: list[expr]
    """Deprecated. Use elts instead."""

    if sys.version_info >= (3, 13):
        def __init__(self, elts: list[expr] = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, elts: list[expr], ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, elts: list[expr] = ..., ctx: expr_context = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

@deprecated("Deprecated since Python 3.9.")
class slice(AST):
    """Deprecated AST node class."""

class Slice(expr):
    """Slice(expr? lower, expr? upper, expr? step)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("lower", "upper", "step")
    lower: expr | None
    upper: expr | None
    step: expr | None
    def __init__(
        self, lower: expr | None = None, upper: expr | None = None, step: expr | None = None, **kwargs: Unpack[_Attributes]
    ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, lower: expr | None = ..., upper: expr | None = ..., step: expr | None = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

@deprecated("Deprecated since Python 3.9. Use `ast.Tuple` instead.")
class ExtSlice(slice):
    """Deprecated AST node class. Use ast.Tuple instead."""

    def __new__(cls, dims: Iterable[slice] = (), **kwargs: Unpack[_Attributes]) -> Tuple: ...  # type: ignore[misc]

@deprecated("Deprecated since Python 3.9. Use the index value directly instead.")
class Index(slice):
    """Deprecated AST node class. Use the index value directly instead."""

    def __new__(cls, value: expr, **kwargs: Unpack[_Attributes]) -> expr: ...  # type: ignore[misc]

class expr_context(AST):
    """expr_context = Load | Store | Del"""

@deprecated("Deprecated since Python 3.9. Unused in Python 3.")
class AugLoad(expr_context):
    """Deprecated AST node class.  Unused in Python 3."""

@deprecated("Deprecated since Python 3.9. Unused in Python 3.")
class AugStore(expr_context):
    """Deprecated AST node class.  Unused in Python 3."""

@deprecated("Deprecated since Python 3.9. Unused in Python 3.")
class Param(expr_context):
    """Deprecated AST node class.  Unused in Python 3."""

@deprecated("Deprecated since Python 3.9. Unused in Python 3.")
class Suite(mod):
    """Deprecated AST node class.  Unused in Python 3."""

class Load(expr_context):
    """Load"""

class Store(expr_context):
    """Store"""

class Del(expr_context):
    """Del"""

class boolop(AST):
    """boolop = And | Or"""

class And(boolop):
    """And"""

class Or(boolop):
    """Or"""

class operator(AST):
    """operator = Add | Sub | Mult | MatMult | Div | Mod | Pow | LShift | RShift | BitOr | BitXor | BitAnd | FloorDiv"""

class Add(operator):
    """Add"""

class Sub(operator):
    """Sub"""

class Mult(operator):
    """Mult"""

class MatMult(operator):
    """MatMult"""

class Div(operator):
    """Div"""

class Mod(operator):
    """Mod"""

class Pow(operator):
    """Pow"""

class LShift(operator):
    """LShift"""

class RShift(operator):
    """RShift"""

class BitOr(operator):
    """BitOr"""

class BitXor(operator):
    """BitXor"""

class BitAnd(operator):
    """BitAnd"""

class FloorDiv(operator):
    """FloorDiv"""

class unaryop(AST):
    """unaryop = Invert | Not | UAdd | USub"""

class Invert(unaryop):
    """Invert"""

class Not(unaryop):
    """Not"""

class UAdd(unaryop):
    """UAdd"""

class USub(unaryop):
    """USub"""

class cmpop(AST):
    """cmpop = Eq | NotEq | Lt | LtE | Gt | GtE | Is | IsNot | In | NotIn"""

class Eq(cmpop):
    """Eq"""

class NotEq(cmpop):
    """NotEq"""

class Lt(cmpop):
    """Lt"""

class LtE(cmpop):
    """LtE"""

class Gt(cmpop):
    """Gt"""

class GtE(cmpop):
    """GtE"""

class Is(cmpop):
    """Is"""

class IsNot(cmpop):
    """IsNot"""

class In(cmpop):
    """In"""

class NotIn(cmpop):
    """NotIn"""

class comprehension(AST):
    """comprehension(expr target, expr iter, expr* ifs, int is_async)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("target", "iter", "ifs", "is_async")
    target: expr
    iter: expr
    ifs: list[expr]
    is_async: int
    if sys.version_info >= (3, 13):
        @overload
        def __init__(self, target: expr, iter: expr, ifs: list[expr], is_async: int) -> None: ...
        @overload
        def __init__(self, target: expr, iter: expr, ifs: list[expr] = ..., *, is_async: int) -> None: ...
    else:
        def __init__(self, target: expr, iter: expr, ifs: list[expr], is_async: int) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, target: expr = ..., iter: expr = ..., ifs: list[expr] = ..., is_async: int = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class excepthandler(AST):
    """excepthandler = ExceptHandler(expr? type, identifier? name, stmt* body)"""

    lineno: int
    col_offset: int
    end_lineno: int | None
    end_col_offset: int | None
    def __init__(self, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, lineno: int = ..., col_offset: int = ..., end_lineno: int | None = ..., end_col_offset: int | None = ...
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class ExceptHandler(excepthandler):
    """ExceptHandler(expr? type, identifier? name, stmt* body)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("type", "name", "body")
    type: expr | None
    name: str | None
    body: list[stmt]
    if sys.version_info >= (3, 13):
        def __init__(
            self, type: expr | None = None, name: str | None = None, body: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> None: ...
    else:
        @overload
        def __init__(self, type: expr | None, name: str | None, body: list[stmt], **kwargs: Unpack[_Attributes]) -> None: ...
        @overload
        def __init__(
            self, type: expr | None = None, name: str | None = None, *, body: list[stmt], **kwargs: Unpack[_Attributes]
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, type: expr | None = ..., name: str | None = ..., body: list[stmt] = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class arguments(AST):
    """arguments(arg* posonlyargs, arg* args, arg? vararg, arg* kwonlyargs, expr?* kw_defaults, arg? kwarg, expr* defaults)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("posonlyargs", "args", "vararg", "kwonlyargs", "kw_defaults", "kwarg", "defaults")
    posonlyargs: list[arg]
    args: list[arg]
    vararg: arg | None
    kwonlyargs: list[arg]
    kw_defaults: list[expr | None]
    kwarg: arg | None
    defaults: list[expr]
    if sys.version_info >= (3, 13):
        def __init__(
            self,
            posonlyargs: list[arg] = ...,
            args: list[arg] = ...,
            vararg: arg | None = None,
            kwonlyargs: list[arg] = ...,
            kw_defaults: list[expr | None] = ...,
            kwarg: arg | None = None,
            defaults: list[expr] = ...,
        ) -> None: ...
    else:
        @overload
        def __init__(
            self,
            posonlyargs: list[arg],
            args: list[arg],
            vararg: arg | None,
            kwonlyargs: list[arg],
            kw_defaults: list[expr | None],
            kwarg: arg | None,
            defaults: list[expr],
        ) -> None: ...
        @overload
        def __init__(
            self,
            posonlyargs: list[arg],
            args: list[arg],
            vararg: arg | None,
            kwonlyargs: list[arg],
            kw_defaults: list[expr | None],
            kwarg: arg | None = None,
            *,
            defaults: list[expr],
        ) -> None: ...
        @overload
        def __init__(
            self,
            posonlyargs: list[arg],
            args: list[arg],
            vararg: arg | None = None,
            *,
            kwonlyargs: list[arg],
            kw_defaults: list[expr | None],
            kwarg: arg | None = None,
            defaults: list[expr],
        ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self,
            *,
            posonlyargs: list[arg] = ...,
            args: list[arg] = ...,
            vararg: arg | None = ...,
            kwonlyargs: list[arg] = ...,
            kw_defaults: list[expr | None] = ...,
            kwarg: arg | None = ...,
            defaults: list[expr] = ...,
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class arg(AST):
    """arg(identifier arg, expr? annotation, string? type_comment)"""

    lineno: int
    col_offset: int
    end_lineno: int | None
    end_col_offset: int | None
    if sys.version_info >= (3, 10):
        __match_args__ = ("arg", "annotation", "type_comment")
    arg: str
    annotation: expr | None
    type_comment: str | None
    def __init__(
        self, arg: str, annotation: expr | None = None, type_comment: str | None = None, **kwargs: Unpack[_Attributes]
    ) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(
            self, *, arg: str = ..., annotation: expr | None = ..., type_comment: str | None = ..., **kwargs: Unpack[_Attributes]
        ) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class keyword(AST):
    """keyword(identifier? arg, expr value)"""

    lineno: int
    col_offset: int
    end_lineno: int | None
    end_col_offset: int | None
    if sys.version_info >= (3, 10):
        __match_args__ = ("arg", "value")
    arg: str | None
    value: expr
    @overload
    def __init__(self, arg: str | None, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...
    @overload
    def __init__(self, arg: str | None = None, *, value: expr, **kwargs: Unpack[_Attributes]) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, arg: str | None = ..., value: expr = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class alias(AST):
    """alias(identifier name, identifier? asname)"""

    name: str
    asname: str | None
    if sys.version_info >= (3, 10):
        lineno: int
        col_offset: int
        end_lineno: int | None
        end_col_offset: int | None
    if sys.version_info >= (3, 10):
        __match_args__ = ("name", "asname")
    if sys.version_info >= (3, 10):
        def __init__(self, name: str, asname: str | None = None, **kwargs: Unpack[_Attributes]) -> None: ...
    else:
        def __init__(self, name: str, asname: str | None = None) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, name: str = ..., asname: str | None = ..., **kwargs: Unpack[_Attributes]) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

class withitem(AST):
    """withitem(expr context_expr, expr? optional_vars)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("context_expr", "optional_vars")
    context_expr: expr
    optional_vars: expr | None
    def __init__(self, context_expr: expr, optional_vars: expr | None = None) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, context_expr: expr = ..., optional_vars: expr | None = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 10):
    class pattern(AST):
        """pattern = MatchValue(expr value)
        | MatchSingleton(constant value)
        | MatchSequence(pattern* patterns)
        | MatchMapping(expr* keys, pattern* patterns, identifier? rest)
        | MatchClass(expr cls, pattern* patterns, identifier* kwd_attrs, pattern* kwd_patterns)
        | MatchStar(identifier? name)
        | MatchAs(pattern? pattern, identifier? name)
        | MatchOr(pattern* patterns)
        """

        lineno: int
        col_offset: int
        end_lineno: int
        end_col_offset: int
        def __init__(self, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self, *, lineno: int = ..., col_offset: int = ..., end_lineno: int = ..., end_col_offset: int = ...
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class match_case(AST):
        """match_case(pattern pattern, expr? guard, stmt* body)"""

        __match_args__ = ("pattern", "guard", "body")
        pattern: ast.pattern
        guard: expr | None
        body: list[stmt]
        if sys.version_info >= (3, 13):
            def __init__(self, pattern: ast.pattern, guard: expr | None = None, body: list[stmt] = ...) -> None: ...
        elif sys.version_info >= (3, 10):
            @overload
            def __init__(self, pattern: ast.pattern, guard: expr | None, body: list[stmt]) -> None: ...
            @overload
            def __init__(self, pattern: ast.pattern, guard: expr | None = None, *, body: list[stmt]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, pattern: ast.pattern = ..., guard: expr | None = ..., body: list[stmt] = ...) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class Match(stmt):
        """Match(expr subject, match_case* cases)"""

        __match_args__ = ("subject", "cases")
        subject: expr
        cases: list[match_case]
        if sys.version_info >= (3, 13):
            def __init__(self, subject: expr, cases: list[match_case] = ..., **kwargs: Unpack[_Attributes]) -> None: ...
        else:
            def __init__(self, subject: expr, cases: list[match_case], **kwargs: Unpack[_Attributes]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, subject: expr = ..., cases: list[match_case] = ..., **kwargs: Unpack[_Attributes]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchValue(pattern):
        """MatchValue(expr value)"""

        __match_args__ = ("value",)
        value: expr
        def __init__(self, value: expr, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, value: expr = ..., **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchSingleton(pattern):
        """MatchSingleton(constant value)"""

        __match_args__ = ("value",)
        value: bool | None
        def __init__(self, value: bool | None, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, value: bool | None = ..., **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchSequence(pattern):
        """MatchSequence(pattern* patterns)"""

        __match_args__ = ("patterns",)
        patterns: list[pattern]
        if sys.version_info >= (3, 13):
            def __init__(self, patterns: list[pattern] = ..., **kwargs: Unpack[_Attributes[int]]) -> None: ...
        else:
            def __init__(self, patterns: list[pattern], **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, patterns: list[pattern] = ..., **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchMapping(pattern):
        """MatchMapping(expr* keys, pattern* patterns, identifier? rest)"""

        __match_args__ = ("keys", "patterns", "rest")
        keys: list[expr]
        patterns: list[pattern]
        rest: str | None
        if sys.version_info >= (3, 13):
            def __init__(
                self,
                keys: list[expr] = ...,
                patterns: list[pattern] = ...,
                rest: str | None = None,
                **kwargs: Unpack[_Attributes[int]],
            ) -> None: ...
        else:
            def __init__(
                self, keys: list[expr], patterns: list[pattern], rest: str | None = None, **kwargs: Unpack[_Attributes[int]]
            ) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self,
                *,
                keys: list[expr] = ...,
                patterns: list[pattern] = ...,
                rest: str | None = ...,
                **kwargs: Unpack[_Attributes[int]],
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchClass(pattern):
        """MatchClass(expr cls, pattern* patterns, identifier* kwd_attrs, pattern* kwd_patterns)"""

        __match_args__ = ("cls", "patterns", "kwd_attrs", "kwd_patterns")
        cls: expr
        patterns: list[pattern]
        kwd_attrs: list[str]
        kwd_patterns: list[pattern]
        if sys.version_info >= (3, 13):
            def __init__(
                self,
                cls: expr,
                patterns: list[pattern] = ...,
                kwd_attrs: list[str] = ...,
                kwd_patterns: list[pattern] = ...,
                **kwargs: Unpack[_Attributes[int]],
            ) -> None: ...
        else:
            def __init__(
                self,
                cls: expr,
                patterns: list[pattern],
                kwd_attrs: list[str],
                kwd_patterns: list[pattern],
                **kwargs: Unpack[_Attributes[int]],
            ) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self,
                *,
                cls: expr = ...,
                patterns: list[pattern] = ...,
                kwd_attrs: list[str] = ...,
                kwd_patterns: list[pattern] = ...,
                **kwargs: Unpack[_Attributes[int]],
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchStar(pattern):
        """MatchStar(identifier? name)"""

        __match_args__ = ("name",)
        name: str | None
        def __init__(self, name: str | None = None, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, name: str | None = ..., **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchAs(pattern):
        """MatchAs(pattern? pattern, identifier? name)"""

        __match_args__ = ("pattern", "name")
        pattern: ast.pattern | None
        name: str | None
        def __init__(
            self, pattern: ast.pattern | None = None, name: str | None = None, **kwargs: Unpack[_Attributes[int]]
        ) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self, *, pattern: ast.pattern | None = ..., name: str | None = ..., **kwargs: Unpack[_Attributes[int]]
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class MatchOr(pattern):
        """MatchOr(pattern* patterns)"""

        __match_args__ = ("patterns",)
        patterns: list[pattern]
        if sys.version_info >= (3, 13):
            def __init__(self, patterns: list[pattern] = ..., **kwargs: Unpack[_Attributes[int]]) -> None: ...
        else:
            def __init__(self, patterns: list[pattern], **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, *, patterns: list[pattern] = ..., **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

class type_ignore(AST):
    """type_ignore = TypeIgnore(int lineno, string tag)"""

class TypeIgnore(type_ignore):
    """TypeIgnore(int lineno, string tag)"""

    if sys.version_info >= (3, 10):
        __match_args__ = ("lineno", "tag")
    lineno: int
    tag: str
    def __init__(self, lineno: int, tag: str) -> None: ...

    if sys.version_info >= (3, 14):
        def __replace__(self, *, lineno: int = ..., tag: str = ...) -> Self:
            """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 12):
    class type_param(AST):
        """type_param = TypeVar(identifier name, expr? bound, expr? default_value)
        | ParamSpec(identifier name, expr? default_value)
        | TypeVarTuple(identifier name, expr? default_value)
        """

        lineno: int
        col_offset: int
        end_lineno: int
        end_col_offset: int
        def __init__(self, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(self, **kwargs: Unpack[_Attributes[int]]) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class TypeVar(type_param):
        """TypeVar(identifier name, expr? bound, expr? default_value)"""

        if sys.version_info >= (3, 13):
            __match_args__ = ("name", "bound", "default_value")
        else:
            __match_args__ = ("name", "bound")
        name: str
        bound: expr | None
        if sys.version_info >= (3, 13):
            default_value: expr | None
            def __init__(
                self, name: str, bound: expr | None = None, default_value: expr | None = None, **kwargs: Unpack[_Attributes[int]]
            ) -> None: ...
        else:
            def __init__(self, name: str, bound: expr | None = None, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self,
                *,
                name: str = ...,
                bound: expr | None = ...,
                default_value: expr | None = ...,
                **kwargs: Unpack[_Attributes[int]],
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class ParamSpec(type_param):
        """ParamSpec(identifier name, expr? default_value)"""

        if sys.version_info >= (3, 13):
            __match_args__ = ("name", "default_value")
        else:
            __match_args__ = ("name",)
        name: str
        if sys.version_info >= (3, 13):
            default_value: expr | None
            def __init__(self, name: str, default_value: expr | None = None, **kwargs: Unpack[_Attributes[int]]) -> None: ...
        else:
            def __init__(self, name: str, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self, *, name: str = ..., default_value: expr | None = ..., **kwargs: Unpack[_Attributes[int]]
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

    class TypeVarTuple(type_param):
        """TypeVarTuple(identifier name, expr? default_value)"""

        if sys.version_info >= (3, 13):
            __match_args__ = ("name", "default_value")
        else:
            __match_args__ = ("name",)
        name: str
        if sys.version_info >= (3, 13):
            default_value: expr | None
            def __init__(self, name: str, default_value: expr | None = None, **kwargs: Unpack[_Attributes[int]]) -> None: ...
        else:
            def __init__(self, name: str, **kwargs: Unpack[_Attributes[int]]) -> None: ...

        if sys.version_info >= (3, 14):
            def __replace__(
                self, *, name: str = ..., default_value: expr | None = ..., **kwargs: Unpack[_Attributes[int]]
            ) -> Self:
                """Return a copy of the AST node with new values for the specified fields."""

if sys.version_info >= (3, 14):
    @type_check_only
    class _ABC(type):
        def __init__(cls, *args: Unused) -> None: ...

else:
    class _ABC(type):
        def __init__(cls, *args: Unused) -> None: ...

if sys.version_info < (3, 14):
    @deprecated("Removed in Python 3.14. Use `ast.Constant` instead.")
    class Num(Constant, metaclass=_ABC):
        """Deprecated AST node class. Use ast.Constant instead"""

        def __new__(cls, n: complex, **kwargs: Unpack[_Attributes]) -> Constant: ...  # type: ignore[misc]  # pyright: ignore[reportInconsistentConstructor]

    @deprecated("Removed in Python 3.14. Use `ast.Constant` instead.")
    class Str(Constant, metaclass=_ABC):
        """Deprecated AST node class. Use ast.Constant instead"""

        def __new__(cls, s: str, **kwargs: Unpack[_Attributes]) -> Constant: ...  # type: ignore[misc]  # pyright: ignore[reportInconsistentConstructor]

    @deprecated("Removed in Python 3.14. Use `ast.Constant` instead.")
    class Bytes(Constant, metaclass=_ABC):
        """Deprecated AST node class. Use ast.Constant instead"""

        def __new__(cls, s: bytes, **kwargs: Unpack[_Attributes]) -> Constant: ...  # type: ignore[misc]  # pyright: ignore[reportInconsistentConstructor]

    @deprecated("Removed in Python 3.14. Use `ast.Constant` instead.")
    class NameConstant(Constant, metaclass=_ABC):
        """Deprecated AST node class. Use ast.Constant instead"""

        def __new__(cls, value: _ConstantValue, kind: str | None, **kwargs: Unpack[_Attributes]) -> Constant: ...  # type: ignore[misc]  # pyright: ignore[reportInconsistentConstructor]

    @deprecated("Removed in Python 3.14. Use `ast.Constant` instead.")
    class Ellipsis(Constant, metaclass=_ABC):
        """Deprecated AST node class. Use ast.Constant instead"""

        def __new__(cls, **kwargs: Unpack[_Attributes]) -> Constant: ...  # type: ignore[misc]  # pyright: ignore[reportInconsistentConstructor]

# everything below here is defined in ast.py

_T = _TypeVar("_T", bound=AST)

if sys.version_info >= (3, 13):
    @overload
    def parse(
        source: _T,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: Literal["exec", "eval", "func_type", "single"] = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> _T:
        """
        Parse the source into an AST node.
        Equivalent to compile(source, filename, mode, PyCF_ONLY_AST).
        Pass type_comments=True to get back type comments where the syntax allows.
        """

    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: Literal["exec"] = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> Module: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["eval"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> Expression: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["func_type"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> FunctionType: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["single"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> Interactive: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["eval"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> Expression: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["func_type"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> FunctionType: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["single"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> Interactive: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: str = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
        optimize: Literal[-1, 0, 1, 2] = -1,
    ) -> mod: ...

else:
    @overload
    def parse(
        source: _T,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: Literal["exec", "eval", "func_type", "single"] = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> _T:
        """
        Parse the source into an AST node.
        Equivalent to compile(source, filename, mode, PyCF_ONLY_AST).
        Pass type_comments=True to get back type comments where the syntax allows.
        """

    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: Literal["exec"] = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> Module: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["eval"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> Expression: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["func_type"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> FunctionType: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any],
        mode: Literal["single"],
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> Interactive: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["eval"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> Expression: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["func_type"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> FunctionType: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        *,
        mode: Literal["single"],
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> Interactive: ...
    @overload
    def parse(
        source: str | ReadableBuffer,
        filename: str | bytes | os.PathLike[Any] = "<unknown>",
        mode: str = "exec",
        *,
        type_comments: bool = False,
        feature_version: None | int | tuple[int, int] = None,
    ) -> mod: ...

def literal_eval(node_or_string: str | AST) -> Any:
    """
    Evaluate an expression node or a string containing only a Python
    expression.  The string or node provided may only consist of the following
    Python literal structures: strings, bytes, numbers, tuples, lists, dicts,
    sets, booleans, and None.

    Caution: A complex expression can overflow the C stack and cause a crash.
    """

if sys.version_info >= (3, 13):
    def dump(
        node: AST,
        annotate_fields: bool = True,
        include_attributes: bool = False,
        *,
        indent: int | str | None = None,
        show_empty: bool = False,
    ) -> str:
        """
        Return a formatted dump of the tree in node.  This is mainly useful for
        debugging purposes.  If annotate_fields is true (by default),
        the returned string will show the names and the values for fields.
        If annotate_fields is false, the result string will be more compact by
        omitting unambiguous field names.  Attributes such as line
        numbers and column offsets are not dumped by default.  If this is wanted,
        include_attributes can be set to true.  If indent is a non-negative
        integer or string, then the tree will be pretty-printed with that indent
        level. None (the default) selects the single line representation.
        If show_empty is False, then empty lists and fields that are None
        will be omitted from the output for better readability.
        """

else:
    def dump(
        node: AST, annotate_fields: bool = True, include_attributes: bool = False, *, indent: int | str | None = None
    ) -> str:
        """
        Return a formatted dump of the tree in node.  This is mainly useful for
        debugging purposes.  If annotate_fields is true (by default),
        the returned string will show the names and the values for fields.
        If annotate_fields is false, the result string will be more compact by
        omitting unambiguous field names.  Attributes such as line
        numbers and column offsets are not dumped by default.  If this is wanted,
        include_attributes can be set to true.  If indent is a non-negative
        integer or string, then the tree will be pretty-printed with that indent
        level. None (the default) selects the single line representation.
        """

def copy_location(new_node: _T, old_node: AST) -> _T:
    """
    Copy source location (`lineno`, `col_offset`, `end_lineno`, and `end_col_offset`
    attributes) from *old_node* to *new_node* if possible, and return *new_node*.
    """

def fix_missing_locations(node: _T) -> _T:
    """
    When you compile a node tree with compile(), the compiler expects lineno and
    col_offset attributes for every node that supports them.  This is rather
    tedious to fill in for generated nodes, so this helper adds these attributes
    recursively where not already set, by setting them to the values of the
    parent node.  It works recursively starting at *node*.
    """

def increment_lineno(node: _T, n: int = 1) -> _T:
    """
    Increment the line number and end line number of each node in the tree
    starting at *node* by *n*. This is useful to "move code" to a different
    location in a file.
    """

def iter_fields(node: AST) -> Iterator[tuple[str, Any]]:
    """
    Yield a tuple of ``(fieldname, value)`` for each field in ``node._fields``
    that is present on *node*.
    """

def iter_child_nodes(node: AST) -> Iterator[AST]:
    """
    Yield all direct child nodes of *node*, that is, all fields that are nodes
    and all items of fields that are lists of nodes.
    """

def get_docstring(node: AsyncFunctionDef | FunctionDef | ClassDef | Module, clean: bool = True) -> str | None:
    """
    Return the docstring for the given node or None if no docstring can
    be found.  If the node provided does not have docstrings a TypeError
    will be raised.

    If *clean* is `True`, all tabs are expanded to spaces and any whitespace
    that can be uniformly removed from the second line onwards is removed.
    """

def get_source_segment(source: str, node: AST, *, padded: bool = False) -> str | None:
    """Get source code segment of the *source* that generated *node*.

    If some location information (`lineno`, `end_lineno`, `col_offset`,
    or `end_col_offset`) is missing, return None.

    If *padded* is `True`, the first line of a multi-line statement will
    be padded with spaces to match its original position.
    """

def walk(node: AST) -> Iterator[AST]:
    """
    Recursively yield all descendant nodes in the tree starting at *node*
    (including *node* itself), in no specified order.  This is useful if you
    only want to modify nodes in place and don't care about the context.
    """

if sys.version_info >= (3, 14):
    def compare(left: AST, right: AST, /, *, compare_attributes: bool = False) -> bool:
        """Recursively compares two ASTs.

        compare_attributes affects whether AST attributes are considered
        in the comparison. If compare_attributes is False (default), then
        attributes are ignored. Otherwise they must all be equal. This
        option is useful to check whether the ASTs are structurally equal but
        might differ in whitespace or similar details.
        """

class NodeVisitor:
    """
    A node visitor base class that walks the abstract syntax tree and calls a
    visitor function for every node found.  This function may return a value
    which is forwarded by the `visit` method.

    This class is meant to be subclassed, with the subclass adding visitor
    methods.

    Per default the visitor functions for the nodes are ``'visit_'`` +
    class name of the node.  So a `TryFinally` node visit function would
    be `visit_TryFinally`.  This behavior can be changed by overriding
    the `visit` method.  If no visitor function exists for a node
    (return value `None`) the `generic_visit` visitor is used instead.

    Don't use the `NodeVisitor` if you want to apply changes to nodes during
    traversing.  For this a special visitor exists (`NodeTransformer`) that
    allows modifications.
    """

    # All visit methods below can be overwritten by subclasses and return an
    # arbitrary value, which is passed to the caller.
    def visit(self, node: AST) -> Any:
        """Visit a node."""

    def generic_visit(self, node: AST) -> Any:
        """Called if no explicit visitor function exists for a node."""
    # The following visit methods are not defined on NodeVisitor, but can
    # be implemented by subclasses and are called during a visit if defined.
    def visit_Module(self, node: Module) -> Any: ...
    def visit_Interactive(self, node: Interactive) -> Any: ...
    def visit_Expression(self, node: Expression) -> Any: ...
    def visit_FunctionDef(self, node: FunctionDef) -> Any: ...
    def visit_AsyncFunctionDef(self, node: AsyncFunctionDef) -> Any: ...
    def visit_ClassDef(self, node: ClassDef) -> Any: ...
    def visit_Return(self, node: Return) -> Any: ...
    def visit_Delete(self, node: Delete) -> Any: ...
    def visit_Assign(self, node: Assign) -> Any: ...
    def visit_AugAssign(self, node: AugAssign) -> Any: ...
    def visit_AnnAssign(self, node: AnnAssign) -> Any: ...
    def visit_For(self, node: For) -> Any: ...
    def visit_AsyncFor(self, node: AsyncFor) -> Any: ...
    def visit_While(self, node: While) -> Any: ...
    def visit_If(self, node: If) -> Any: ...
    def visit_With(self, node: With) -> Any: ...
    def visit_AsyncWith(self, node: AsyncWith) -> Any: ...
    def visit_Raise(self, node: Raise) -> Any: ...
    def visit_Try(self, node: Try) -> Any: ...
    def visit_Assert(self, node: Assert) -> Any: ...
    def visit_Import(self, node: Import) -> Any: ...
    def visit_ImportFrom(self, node: ImportFrom) -> Any: ...
    def visit_Global(self, node: Global) -> Any: ...
    def visit_Nonlocal(self, node: Nonlocal) -> Any: ...
    def visit_Expr(self, node: Expr) -> Any: ...
    def visit_Pass(self, node: Pass) -> Any: ...
    def visit_Break(self, node: Break) -> Any: ...
    def visit_Continue(self, node: Continue) -> Any: ...
    def visit_Slice(self, node: Slice) -> Any: ...
    def visit_BoolOp(self, node: BoolOp) -> Any: ...
    def visit_BinOp(self, node: BinOp) -> Any: ...
    def visit_UnaryOp(self, node: UnaryOp) -> Any: ...
    def visit_Lambda(self, node: Lambda) -> Any: ...
    def visit_IfExp(self, node: IfExp) -> Any: ...
    def visit_Dict(self, node: Dict) -> Any: ...
    def visit_Set(self, node: Set) -> Any: ...
    def visit_ListComp(self, node: ListComp) -> Any: ...
    def visit_SetComp(self, node: SetComp) -> Any: ...
    def visit_DictComp(self, node: DictComp) -> Any: ...
    def visit_GeneratorExp(self, node: GeneratorExp) -> Any: ...
    def visit_Await(self, node: Await) -> Any: ...
    def visit_Yield(self, node: Yield) -> Any: ...
    def visit_YieldFrom(self, node: YieldFrom) -> Any: ...
    def visit_Compare(self, node: Compare) -> Any: ...
    def visit_Call(self, node: Call) -> Any: ...
    def visit_FormattedValue(self, node: FormattedValue) -> Any: ...
    def visit_JoinedStr(self, node: JoinedStr) -> Any: ...
    def visit_Constant(self, node: Constant) -> Any: ...
    def visit_NamedExpr(self, node: NamedExpr) -> Any: ...
    def visit_TypeIgnore(self, node: TypeIgnore) -> Any: ...
    def visit_Attribute(self, node: Attribute) -> Any: ...
    def visit_Subscript(self, node: Subscript) -> Any: ...
    def visit_Starred(self, node: Starred) -> Any: ...
    def visit_Name(self, node: Name) -> Any: ...
    def visit_List(self, node: List) -> Any: ...
    def visit_Tuple(self, node: Tuple) -> Any: ...
    def visit_Del(self, node: Del) -> Any: ...
    def visit_Load(self, node: Load) -> Any: ...
    def visit_Store(self, node: Store) -> Any: ...
    def visit_And(self, node: And) -> Any: ...
    def visit_Or(self, node: Or) -> Any: ...
    def visit_Add(self, node: Add) -> Any: ...
    def visit_BitAnd(self, node: BitAnd) -> Any: ...
    def visit_BitOr(self, node: BitOr) -> Any: ...
    def visit_BitXor(self, node: BitXor) -> Any: ...
    def visit_Div(self, node: Div) -> Any: ...
    def visit_FloorDiv(self, node: FloorDiv) -> Any: ...
    def visit_LShift(self, node: LShift) -> Any: ...
    def visit_Mod(self, node: Mod) -> Any: ...
    def visit_Mult(self, node: Mult) -> Any: ...
    def visit_MatMult(self, node: MatMult) -> Any: ...
    def visit_Pow(self, node: Pow) -> Any: ...
    def visit_RShift(self, node: RShift) -> Any: ...
    def visit_Sub(self, node: Sub) -> Any: ...
    def visit_Invert(self, node: Invert) -> Any: ...
    def visit_Not(self, node: Not) -> Any: ...
    def visit_UAdd(self, node: UAdd) -> Any: ...
    def visit_USub(self, node: USub) -> Any: ...
    def visit_Eq(self, node: Eq) -> Any: ...
    def visit_Gt(self, node: Gt) -> Any: ...
    def visit_GtE(self, node: GtE) -> Any: ...
    def visit_In(self, node: In) -> Any: ...
    def visit_Is(self, node: Is) -> Any: ...
    def visit_IsNot(self, node: IsNot) -> Any: ...
    def visit_Lt(self, node: Lt) -> Any: ...
    def visit_LtE(self, node: LtE) -> Any: ...
    def visit_NotEq(self, node: NotEq) -> Any: ...
    def visit_NotIn(self, node: NotIn) -> Any: ...
    def visit_comprehension(self, node: comprehension) -> Any: ...
    def visit_ExceptHandler(self, node: ExceptHandler) -> Any: ...
    def visit_arguments(self, node: arguments) -> Any: ...
    def visit_arg(self, node: arg) -> Any: ...
    def visit_keyword(self, node: keyword) -> Any: ...
    def visit_alias(self, node: alias) -> Any: ...
    def visit_withitem(self, node: withitem) -> Any: ...
    if sys.version_info >= (3, 10):
        def visit_Match(self, node: Match) -> Any: ...
        def visit_match_case(self, node: match_case) -> Any: ...
        def visit_MatchValue(self, node: MatchValue) -> Any: ...
        def visit_MatchSequence(self, node: MatchSequence) -> Any: ...
        def visit_MatchSingleton(self, node: MatchSingleton) -> Any: ...
        def visit_MatchStar(self, node: MatchStar) -> Any: ...
        def visit_MatchMapping(self, node: MatchMapping) -> Any: ...
        def visit_MatchClass(self, node: MatchClass) -> Any: ...
        def visit_MatchAs(self, node: MatchAs) -> Any: ...
        def visit_MatchOr(self, node: MatchOr) -> Any: ...

    if sys.version_info >= (3, 11):
        def visit_TryStar(self, node: TryStar) -> Any: ...

    if sys.version_info >= (3, 12):
        def visit_TypeVar(self, node: TypeVar) -> Any: ...
        def visit_ParamSpec(self, node: ParamSpec) -> Any: ...
        def visit_TypeVarTuple(self, node: TypeVarTuple) -> Any: ...
        def visit_TypeAlias(self, node: TypeAlias) -> Any: ...

    # visit methods for deprecated nodes
    def visit_ExtSlice(self, node: ExtSlice) -> Any: ...
    def visit_Index(self, node: Index) -> Any: ...
    def visit_Suite(self, node: Suite) -> Any: ...
    def visit_AugLoad(self, node: AugLoad) -> Any: ...
    def visit_AugStore(self, node: AugStore) -> Any: ...
    def visit_Param(self, node: Param) -> Any: ...

    if sys.version_info < (3, 14):
        @deprecated("Removed in Python 3.14. Use `visit_Constant` instead.")
        def visit_Num(self, node: Num) -> Any: ...  # type: ignore[deprecated]
        @deprecated("Removed in Python 3.14. Use `visit_Constant` instead.")
        def visit_Str(self, node: Str) -> Any: ...  # type: ignore[deprecated]
        @deprecated("Removed in Python 3.14. Use `visit_Constant` instead.")
        def visit_Bytes(self, node: Bytes) -> Any: ...  # type: ignore[deprecated]
        @deprecated("Removed in Python 3.14. Use `visit_Constant` instead.")
        def visit_NameConstant(self, node: NameConstant) -> Any: ...  # type: ignore[deprecated]
        @deprecated("Removed in Python 3.14. Use `visit_Constant` instead.")
        def visit_Ellipsis(self, node: Ellipsis) -> Any: ...  # type: ignore[deprecated]

class NodeTransformer(NodeVisitor):
    """
    A :class:`NodeVisitor` subclass that walks the abstract syntax tree and
    allows modification of nodes.

    The `NodeTransformer` will walk the AST and use the return value of the
    visitor methods to replace or remove the old node.  If the return value of
    the visitor method is ``None``, the node will be removed from its location,
    otherwise it is replaced with the return value.  The return value may be the
    original node in which case no replacement takes place.

    Here is an example transformer that rewrites all occurrences of name lookups
    (``foo``) to ``data['foo']``::

       class RewriteName(NodeTransformer):

           def visit_Name(self, node):
               return Subscript(
                   value=Name(id='data', ctx=Load()),
                   slice=Constant(value=node.id),
                   ctx=node.ctx
               )

    Keep in mind that if the node you're operating on has child nodes you must
    either transform the child nodes yourself or call the :meth:`generic_visit`
    method for the node first.

    For nodes that were part of a collection of statements (that applies to all
    statement nodes), the visitor may also return a list of nodes rather than
    just a single node.

    Usually you use the transformer like this::

       node = YourTransformer().visit(node)
    """

    def generic_visit(self, node: AST) -> AST: ...
    # TODO: Override the visit_* methods with better return types.
    #       The usual return type is AST | None, but Iterable[AST]
    #       is also allowed in some cases -- this needs to be mapped.

def unparse(ast_obj: AST) -> str: ...

if sys.version_info >= (3, 14):
    def main(args: Sequence[str] | None = None) -> None: ...

else:
    def main() -> None: ...
