//===-- sanitizer/ubsan_interface.h -----------------------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file is a part of UBSanitizer (UBSan).
//
// Public interface header.
//===----------------------------------------------------------------------===//
#ifndef SANITIZER_UBSAN_INTERFACE_H
#define SANITIZER_UBSAN_INTERFACE_H

#include <sanitizer/common_interface_defs.h>

#ifdef __cplusplus
extern "C" {
#endif
/// User-provided default option settings.
///
/// You can provide your own implementation of this function to return a string
/// containing UBSan runtime options (for example,
/// <c>verbosity=1:halt_on_error=0</c>).
///
/// \returns Default options string.
const char *SANITIZER_CDECL __ubsan_default_options(void);

/// Set up an interval timer and install a SIGPROF signal handler that crashes
/// the program if it is stuck in a trap loop generated by -fsanitize-trap-loop.
///
/// This function is only intended to be called by single-threaded programs.
/// Because interval timers are delivered to an arbitrary thread in the process,
/// it is possible that a thread in a trap loop will never have the signal
/// delivered to it. For multi-threaded programs, it is recommended to call
/// OS-specific APIs such as timer_create(CLOCK_THREAD_CPUTIME_ID) on Linux to
/// install a timer on each thread.
///
/// Programs with their own signal handlers can call __ubsan_is_trap_loop from
/// the signal handler to implement trap loop handling.
///
/// This function is currently only supported on Linux/x86.
void __ubsan_install_trap_loop_detection(void);

/// Returns whether uc (actually a pointer to ucontext_t) indicates that the
/// thread is stuck in a trap loop generated by -fsanitize-trap-loop.
///
/// This function is currently only supported on Linux/x86.
int __ubsan_is_trap_loop(void *uc);

#ifdef __cplusplus
} // extern "C"
#endif

#endif // SANITIZER_UBSAN_INTERFACE_H
