/* XDRBG with Ascon XOF support functions
 *
 * Copyright (C) 2024 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "alignment.h"
#include "compare.h"
#include "fips_mode.h"
#include "lc_xdrbg.h"
#include "visibility.h"
#include "xdrbg_internal.h"

/********************************** Selftest **********************************/

static void xdrbg128_drng_selftest(void)
{
	LC_FIPS_RODATA_SECTION
	static const uint8_t seed[] = {
		FIPS140_MOD(0x00),
		0x01,
		0x02,
		0x03,
		0x04,
		0x05,
		0x06,
		0x07,
		0x08,
	};
	LC_FIPS_RODATA_SECTION
	static const uint8_t exp[] = {
		0x90, 0xfc, 0x06, 0x53, 0xfb, 0x53, 0xac, 0x68, 0xe9, 0x84,
		0xc5, 0x7f, 0x99, 0xdf, 0x5b, 0x33, 0x8d, 0xba, 0xcd, 0xcb,
		0x8b, 0x87, 0x07, 0x94, 0xe5, 0x36, 0x5e, 0x1c, 0xc1, 0x12,
		0x41, 0x9b, 0xc9, 0x78, 0x0a, 0x58, 0xc9, 0xbc, 0x65, 0x9a,
		0xec, 0xd0, 0xd9, 0xad, 0x6b, 0x56, 0xb0, 0xb5, 0xc5, 0x0a,
		0xbc, 0x87, 0x87, 0x5d, 0x94, 0x2b, 0x9e, 0xe3, 0xad, 0x5f,
		0xc3, 0xa8, 0x8c, 0xc6, 0x96, 0xac, 0x38, 0x08, 0x95, 0x58,
		0x41, 0xd3, 0xd8, 0xf0, 0x31, 0xcd, 0x27, 0xe5, 0x23, 0xfd,
		0x54, 0x7d, 0x65, 0x11, 0x1b, 0xa1, 0xdb, 0x09, 0xe1, 0xe9,
		0xb6, 0x47, 0xee, 0xa3, 0x9a, 0x7e, 0x15, 0xfd, 0xcf, 0xa0,
		0x08, 0xba, 0xce, 0x57, 0xa4, 0xa7, 0x03, 0x52, 0xf0, 0x78,
		0xf7, 0x2a, 0x7b, 0xd1, 0xa1, 0xd6, 0x9f, 0xb4, 0xc2, 0x6f,
		0x7d, 0x3e, 0xfc, 0x78, 0xf5, 0x3b, 0x25, 0x51, 0x56, 0x7e,
		0xe9, 0xb3, 0x4c, 0x0d, 0x2b, 0x1f, 0xef, 0xe1, 0xda, 0x13,
		0x2d, 0xd6, 0xf0, 0x32, 0x22, 0x12, 0x8b, 0x59, 0x3b, 0x97,
		0x28, 0x27, 0x09, 0xa1, 0x9c, 0x41, 0xb3, 0x5b, 0x21, 0x53,
		0x70, 0x3d, 0x02, 0xa1, 0x13, 0x81, 0x33, 0x69, 0x71, 0x7d,
		0x3b, 0x19, 0xa2, 0x9e, 0xbf, 0x64, 0xcd, 0xc6, 0x52, 0x9b,
		0xd3, 0x78, 0x6a, 0x29, 0x1a, 0x34, 0x50, 0xc0, 0x92, 0x1b,
		0x4b, 0x4d, 0xa9, 0xc2, 0x47, 0x72, 0xc4, 0xf2, 0xef, 0x32,
		0x0b, 0x4d, 0xb7, 0x4e, 0x78, 0x58, 0x20, 0x1d, 0xbd, 0x0d,
		0x23, 0x29, 0xfe, 0x1c, 0x36, 0x67, 0xa2, 0x8f, 0x2d, 0xba,
		0x4b, 0x69, 0xfd, 0x24, 0x2d, 0x3a, 0x36, 0xc2, 0xea, 0x5e,
		0x65, 0x21, 0x44, 0x23, 0xfc, 0x25, 0x2b, 0x07, 0x1e, 0xcf,
		0x55, 0x92, 0x3e, 0x6a, 0x8a, 0x21, 0xb1
	};
	uint8_t act[sizeof(exp)] __align(sizeof(uint32_t));

	LC_SELFTEST_RUN(lc_xdrbg128_drng->algorithm_type);

	LC_XDRBG128_DRNG_CTX_ON_STACK(shake_ctx);

	if (lc_xdrbg_drng_seed_nocheck(shake_ctx->rng_state, seed, sizeof(seed),
				       NULL, 0))
		goto out;
	lc_rng_generate(shake_ctx, NULL, 0, act, sizeof(act));

out:
	lc_compare_selftest(lc_xdrbg128_drng->algorithm_type, act, exp,
			    sizeof(exp), "Ascon XOF XDRBG");
	lc_rng_zero(shake_ctx);
}

static int lc_xdrbg128_drng_seed(void *_state, const uint8_t *seed,
				 size_t seedlen, const uint8_t *alpha,
				 size_t alphalen)
{
	xdrbg128_drng_selftest();
	LC_SELFTEST_COMPLETED(lc_xdrbg128_drng->algorithm_type);

	return lc_xdrbg_drng_seed_nocheck(_state, seed, seedlen, alpha,
					  alphalen);
}

LC_INTERFACE_FUNCTION(int, lc_xdrbg128_drng_alloc, struct lc_rng_ctx **state)
{
	struct lc_rng_ctx *out_state = NULL;
	int ret;

	if (!state)
		return -EINVAL;

	ret = lc_alloc_aligned_secure((void *)&out_state,
				      LC_HASH_COMMON_ALIGNMENT,
				      LC_XDRBG128_DRNG_CTX_SIZE);
	if (ret)
		return -ret;

	LC_XDRBG128_RNG_CTX(out_state);

	*state = out_state;

	return 0;
}

static const struct lc_rng _lc_xdrbg128_drng = {
	.generate = lc_xdrbg_drng_generate,
	.seed = lc_xdrbg128_drng_seed,
	.zero = lc_xdrbg_drng_zero,
	.algorithm_type = LC_ALG_STATUS_XDRBG128,
};
LC_INTERFACE_SYMBOL(const struct lc_rng *,
		    lc_xdrbg128_drng) = &_lc_xdrbg128_drng;
