/* dmatrix.h - Handles Data Matrix ECC 200 */
/*
    libzint - the open source barcode library
    Copyright (C) 2009-2024 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

/*
    Contains Extended Rectangular Data Matrix (DMRE)
    See http://www.dmre.info for information
    Contact: harald.oehlmann@eurodatacouncil.org
 */

#ifndef Z_DMATRIX_H
#define Z_DMATRIX_H

#define DM_ASCII    1
#define DM_C40      2
#define DM_TEXT     3
#define DM_X12      4
#define DM_EDIFACT  5
#define DM_BASE256  6

/* C40 shift to use per ASCII character (Table C.1) */
static const char dm_c40_shift[128] = {
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2,
    2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 2,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
};

/* C40 ASCII values (Table C.1) */
static const char dm_c40_value[128] = {
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
     3,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
     4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 15, 16, 17, 18, 19, 20,
    21, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
    29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 22, 23, 24, 25, 26,
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31
};

/* Text shift to use per ASCII character (Table C.2) */
static const char dm_text_shift[128] = {
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2,
    2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2,
    3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 3, 3
};

/* Text ASCII values (Table C.2) */
static const char dm_text_value[128] = {
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
     3,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
     4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 15, 16, 17, 18, 19, 20,
    21,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 22, 23, 24, 25, 26,
     0, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
    29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 27, 28, 29, 30, 31
};

/* Number of DM Sizes */
#define DMSIZESCOUNT 48
/* Number of 144x144 for special interlace */
#define DMINTSYMBOL144 47

/* Total data codewords (Table 7) in ascending order */
static const unsigned short dm_matrixbytes[DMSIZESCOUNT] = {
    /* 0*/   3 /*10x10*/,      5 /*12x12*/,      5 /*8x18*/,       8 /*14x14*/,
    /* 4*/  10 /*8x32*/,      12 /*16x16*/,     16 /*12x26*/,     18 /*18x18*/,
    /* 8*/  18 /*8x48*/,      22 /*20x20*/,     22 /*12x36*/,     24 /*8x64*/,
    /*12*/  30 /*22x22*/,     32 /*16x36*/,     32 /*8x80*/,      36 /*24x24*/,
    /*16*/  38 /*8x96*/,      43 /*12x64*/,     44 /*26x26*/,     44 /*20x36*/,
    /*20*/  49 /*16x48*/,     49 /*8x120*/,     56 /*20x44*/,     62 /*32x32*/,
    /*24*/  62 /*16x64*/,     63 /*8x144*/,     64 /*12x88*/,     70 /*26x40*/,
    /*28*/  72 /*22x48*/,     80 /*24x48*/,     84 /*20x64*/,     86 /*36x36*/,
    /*32*/  90 /*26x48*/,    108 /*24x64*/,    114 /*40x40*/,    118 /*26x64*/,
    /*36*/ 144 /*44x44*/,    174 /*48x48*/,    204 /*52x52*/,    280 /*64x64*/,
    /*40*/ 368 /*72x72*/,    456 /*80x80*/,    576 /*88x88*/,    696 /*96x96*/,
    /*44*/ 816 /*104x104*/, 1050 /*120x120*/, 1304 /*132x132*/, 1558 /*144x144*/
};

/* Index into `dm_matrixbytes` array in `symbol->option_2` (CLI `--vers`) order,
   i.e. square symbols first, then standard rectangular, then DMRE.
   The bracketed comment value is the total data codewords value. */
static const unsigned char dm_intsymbol[DMSIZESCOUNT] = {
    /* Standard DM square */
    /*  1-4*/  0 /*10x10 (3)*/,      1 /*12x12 (5)*/,       3 /*14x14 (8)*/,       5 /*16x16 (12)*/,
    /*  5-8*/  7 /*18x18 (18)*/,     9 /*20x20 (22)*/,     12 /*22x22 (30)*/,     15 /*24x24 (36)*/,
    /* 9-12*/ 18 /*26x26 (44)*/,    23 /*32x32 (62)*/,     31 /*36x36 (86)*/,     34 /*40x40 (114)*/,
    /*13-16*/ 36 /*44x44 (144)*/,   37 /*48x48 (174)*/,    38 /*52x52 (204)*/,    39 /*64x64 (280)*/,
    /*17-20*/ 40 /*72x72 (368)*/,   41 /*80x80 (456)*/,    42 /*88x88 (576)*/,    43 /*96x96 (696)*/,
    /*21-24*/ 44 /*104x104 (816)*/, 45 /*120x120 (1050)*/, 46 /*132x132 (1304)*/, 47 /*144x144 (1558)*/,

    /* Standard DM rectangular */
    /*25-28*/  2 /*8x18 (5)*/,       4 /*8x32 (10)*/,       6 /*12x26 (16)*/,     10 /*12x36 (22)*/,
    /*29-30*/ 13 /*16x36 (32)*/,    20 /*16x48 (49)*/,

    /* DMRE */
    /*31-34*/  8 /*8x48 (18)*/,     11 /*8x64 (24)*/,      14 /*8x80 (32)*/,      16 /*8x96 (38)*/,
    /*35-38*/ 21 /*8x120 (49)*/,    25 /*8x144 (63)*/,     17 /*12x64 (43)*/,     26 /*12x88 (64)*/,
    /*39-42*/ 24 /*16x64 (62)*/,    19 /*20x36 (44)*/,     22 /*20x44 (56)*/,     30 /*20x64 (84)*/,
    /*43-46*/ 28 /*22x48 (72)*/,    29 /*24x48 (80)*/,     33 /*24x64 (108)*/,    27 /*26x40 (70)*/,
    /*47-48*/ 32 /*26x48 (90)*/,    35 /*26x64 (118)*/
};

/* Following arrays in total data codewords order (`dm_matrixbytes`) */

/* Whether the version is DMRE */
static const char dm_isDMRE[DMSIZESCOUNT] = {
    /* 0*/ 0 /*10x10 (3)*/,     0 /*12x12 (5)*/,      0 /*8x18 (5)*/,       0 /*14x14 (8)*/,
    /* 4*/ 0 /*8x32 (10)*/,     0 /*16x16 (12)*/,     0 /*12x26 (16)*/,     0 /*18x18 (18)*/,
    /* 8*/ 1 /*8x48 (18)*/,     0 /*20x20 (22)*/,     0 /*12x36 (22)*/,     1 /*8x64 (24)*/,
    /*12*/ 0 /*22x22 (30)*/,    0 /*16x36 (32)*/,     1 /*8x80 (32)*/,      0 /*24x24 (36)*/,
    /*16*/ 1 /*8x96 (38)*/,     1 /*12x64 (43)*/,     0 /*26x26 (44)*/,     1 /*20x36 (44)*/,
    /*20*/ 0 /*16x48 (49)*/,    1 /*8x120 (49)*/,     1 /*20x44 (56)*/,     0 /*32x32 (62)*/,
    /*24*/ 1 /*16x64 (62)*/,    1 /*8x144 (63)*/,     1 /*12x88 (64)*/,     1 /*26x40 (70)*/,
    /*28*/ 1 /*22x48 (72)*/,    1 /*24x48 (80)*/,     1 /*20x64 (84)*/,     0 /*36x36 (86)*/,
    /*32*/ 1 /*26x48 (90)*/,    1 /*24x64 (108)*/,    0 /*40x40 (114)*/,    1 /*26x64 (118)*/,
    /*36*/ 0 /*44x44 (144)*/,   0 /*48x48 (174)*/,    0 /*52x52 (204)*/,    0 /*64x64 (280)*/,
    /*40*/ 0 /*72x72 (368)*/,   0 /*80x80 (456)*/,    0 /*88x88 (576)*/,    0 /*96x96 (696)*/,
    /*44*/ 0 /*104x104 (816)*/, 0 /*120x120 (1050)*/, 0 /*132x132 (1304)*/, 0 /*144x144 (1558)*/
};

/* Horizontal matrix size */
static const unsigned char dm_matrixH[DMSIZESCOUNT] = {
    /* 0*/  10 /*10x10*/,    12 /*12x12 */,    8 /*8x18*/,     14 /*14x14*/,
    /* 4*/   8 /*8x32*/,     16 /*16x16*/,    12 /*12x26*/,    18 /*18x18*/,
    /* 8*/   8 /*8x48*/,     20 /*20x20*/,    12 /*12x36*/,     8 /*8x64*/,
    /*12*/  22 /*22x22*/,    16 /*16x36*/,     8 /*8x80*/,     24 /*24x24*/,
    /*16*/   8 /*8x96*/,     12 /*12x64*/,    26 /*26x26*/,    20 /*20x36*/,
    /*20*/  16 /*16x48*/,     8 /*8x120*/,    20 /*20x44*/,    32 /*32x32*/,
    /*24*/  16 /*16x64*/,     8 /*8x144*/,    12 /*12x88*/,    26 /*26x40*/,
    /*28*/  22 /*22x48*/,    24 /*24x48*/,    20 /*20x64*/,    36 /*36x36*/,
    /*32*/  26 /*26x48*/,    24 /*24x64*/,    40 /*40x40*/,    26 /*26x64*/,
    /*36*/  44 /*44x44*/,    48 /*48x48*/,    52 /*52x52*/,    64 /*64x64*/,
    /*40*/  72 /*72x72*/,    80 /*80x80*/,    88 /*88x88*/,    96 /*96x96*/,
    /*44*/ 104 /*104x104*/, 120 /*120x120*/, 132 /*132x132*/, 144 /*144x144*/
};

/* Vertical matrix sizes */
static const unsigned char dm_matrixW[DMSIZESCOUNT] = {
    /* 0*/  10 /*10x10*/,    12 /*12x12*/,    18 /*8x18*/,     14 /*14x14*/,
    /* 4*/  32 /*8x32*/,     16 /*16x16*/,    26 /*12x26*/,    18 /*18x18*/,
    /* 8*/  48 /*8x48*/,     20 /*20x20*/,    36 /*12x36*/,    64 /*8x64*/,
    /*12*/  22 /*22x22*/,    36 /*16x36*/,    80 /*8x80*/,     24 /*24x24*/,
    /*16*/  96 /*8x96*/,     64 /*12x64*/,    26 /*26x26*/,    36 /*20x36*/,
    /*20*/  48 /*16x48*/,   120 /*8x120*/,    44 /*20x44*/,    32 /*32x32*/,
    /*24*/  64 /*16x64*/,   144 /*8x144*/,    88 /*12x88*/,    40 /*26x40*/,
    /*28*/  48 /*22x48*/,    48 /*24x48*/,    64 /*20x64*/,    36 /*36x36*/,
    /*32*/  48 /*26x48*/,    64 /*24x64*/,    40 /*40x40*/,    64 /*26x64*/,
    /*36*/  44 /*44x44*/,    48 /*48x48*/,    52 /*52x52*/,    64 /*64x64*/,
    /*40*/  72 /*72x72*/,    80 /*80x80*/,    88 /*88x88*/,    96 /*96x96*/,
    /*44*/ 104 /*104x104*/, 120 /*120x120*/, 132 /*132x132*/, 144 /*144x144*/
};

/* Horizontal submodule size (including subfinder) - see Table 7 Data region H + 2 */
static const unsigned char dm_matrixFH[DMSIZESCOUNT] = {
    /* 0*/ 10 /*10x10*/,   12 /*12x12*/,    8 /*8x18*/,    14 /*14x14*/,
    /* 4*/  8 /*8x32*/,    16 /*16x16*/,   12 /*12x26*/,   18 /*18x18*/,
    /* 8*/  8 /*8x48*/,    20 /*20x20*/,   12 /*12x36*/,    8 /*8x64*/,
    /*12*/ 22 /*22x22*/,   16 /*16x36*/,    8 /*8x80*/,    24 /*24x24*/,
    /*16*/  8 /*8x96*/,    12 /*12x64*/,   26 /*26x26*/,   20 /*20x36*/,
    /*20*/ 16 /*16x48*/,    8 /*8x120*/,   20 /*20x44*/,   16 /*32x32*/,
    /*24*/ 16 /*16x64*/,    8 /*8x144*/,   12 /*12x88*/,   26 /*26x40*/,
    /*28*/ 22 /*22x48*/,   24 /*24x48*/,   20 /*20x64*/,   18 /*36x36*/,
    /*32*/ 26 /*26x48*/,   24 /*24x64*/,   20 /*40x40*/,   26 /*26x64*/,
    /*36*/ 22 /*44x44*/,   24 /*48x48*/,   26 /*52x52*/,   16 /*64x64*/,
    /*40*/ 18 /*72x72*/,   20 /*80x80*/,   22 /*88x88*/,   24 /*96x96*/,
    /*44*/ 26 /*104x104*/, 20 /*120x120*/, 22 /*132x132*/, 24 /*144x144*/
};

/* Vertical submodule size (including subfinder) - see Table 7 Data region W + 2 */
static const unsigned char dm_matrixFW[DMSIZESCOUNT] = {
    /* 0*/ 10 /*10x10*/,   12 /*12x12*/,   18 /*8x18*/,    14 /*14x14*/,
    /* 4*/ 16 /*8x32*/,    16 /*16x16*/,   26 /*12x26*/,   18 /*18x18*/,
    /* 8*/ 24 /*8x48*/,    20 /*20x20*/,   18 /*12x36*/,   16 /*8x64*/,
    /*12*/ 22 /*22x22*/,   18 /*16x36*/,   20 /*8x80*/,    24 /*24x24*/,
    /*16*/ 24 /*8x96*/,    16 /*12x64*/,   26 /*26x26*/,   18 /*20x36*/,
    /*20*/ 24 /*16x48*/,   20 /*8x120*/,   22 /*20x44*/,   16 /*32x32*/,
    /*24*/ 16 /*16x64*/,   24 /*8x144*/,   22 /*12x88*/,   20 /*26x40*/,
    /*28*/ 24 /*22x48*/,   24 /*24x48*/,   16 /*20x64*/,   18 /*36x36*/,
    /*32*/ 24 /*26x48*/,   16 /*24x64*/,   20 /*40x40*/,   16 /*26x64*/,
    /*36*/ 22 /*44x44*/,   24 /*48x48*/,   26 /*52x52*/,   16 /*64x64*/,
    /*40*/ 18 /*72x72*/,   20 /*80x80*/,   22 /*88x88*/,   24 /*96x96*/,
    /*44*/ 26 /*104x104*/, 20 /*120x120*/, 22 /*132x132*/, 24 /*144x144*/
};

/* Data Codewords per RS-Block */
static const unsigned char dm_matrixdatablock[DMSIZESCOUNT] = {
    /* 0*/   3 /*10x10*/,     5 /*12x12*/,     5 /*8x18*/,      8 /*14x14*/,
    /* 4*/  10 /*8x32*/,     12 /*16x16*/,    16 /*12x26*/,    18 /*18x18*/,
    /* 8*/  18 /*8x48*/,     22 /*20x20*/,    22 /*12x36*/,    24 /*8x64*/,
    /*12*/  30 /*22x22*/,    32 /*16x36*/,    32 /*8x80*/,     36 /*24x24*/,
    /*16*/  38 /*8x96*/,     43 /*12x64*/,    44 /*26x26*/,    44 /*20x36*/,
    /*20*/  49 /*16x48*/,    49 /*8x120*/,    56 /*20x44*/,    62 /*32x32*/,
    /*24*/  62 /*16x64*/,    63 /*8x144*/,    64 /*12x88*/,    70 /*26x40*/,
    /*28*/  72 /*22x48*/,    80 /*24x48*/,    84 /*20x64*/,    86 /*36x36*/,
    /*32*/  90 /*26x48*/,   108 /*24x64*/,   114 /*40x40*/,   118 /*26x64*/,
    /*36*/ 144 /*44x44*/,   174 /*48x48*/,   102 /*52x52*/,   140 /*64x64*/,
    /*40*/  92 /*72x72*/,   114 /*80x80*/,   144 /*88x88*/,   174 /*96x96*/,
    /*44*/ 136 /*104x104*/, 175 /*120x120*/, 163 /*132x132*/, 156 /*144x144*/
};

/* ECC Codewords per RS-Block */
static const unsigned char dm_matrixrsblock[DMSIZESCOUNT] = {
    /* 0*/  5 /*10x10*/,    7 /*12x12*/,    7 /*8x18*/,    10 /*14x14*/,
    /* 4*/ 11 /*8x32*/,    12 /*16x16*/,   14 /*12x26*/,   14 /*18x18*/,
    /* 8*/ 15 /*8x48*/,    18 /*20x20*/,   18 /*12x36*/,   18 /*8x64*/,
    /*12*/ 20 /*22x22*/,   24 /*16x36*/,   22 /*8x80*/,    24 /*24x24*/,
    /*16*/ 28 /*8x96*/,    27 /*12x64*/,   28 /*26x26*/,   28 /*20x36*/,
    /*20*/ 28 /*16x48*/,   32 /*8x120*/,   34 /*20x44*/,   36 /*32x32*/,
    /*24*/ 36 /*16x64*/,   36 /*8x144*/,   36 /*12x88*/,   38 /*26x40*/,
    /*28*/ 38 /*22x48*/,   41 /*24x48*/,   42 /*20x64*/,   42 /*36x36*/,
    /*32*/ 42 /*26x48*/,   46 /*24x64*/,   48 /*40x40*/,   50 /*26x64*/,
    /*36*/ 56 /*44x44*/,   68 /*48x48*/,   42 /*52x52*/,   56 /*64x64*/,
    /*40*/ 36 /*72x72*/,   48 /*80x80*/,   56 /*88x88*/,   68 /*96x96*/,
    /*44*/ 56 /*104x104*/, 68 /*120x120*/, 62 /*132x132*/, 62 /*144x144*/
};

/* vim: set ts=4 sw=4 et : */
#endif /* Z_DMATRIX_H */
