#!/bin/sh -e

# Copyright (C) Internet Systems Consortium, Inc. ("ISC")
#
# SPDX-License-Identifier: MPL-2.0
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0.  If a copy of the MPL was not distributed with this
# file, you can obtain one at https://mozilla.org/MPL/2.0/.
#
# See the COPYRIGHT file distributed with this work for additional
# information regarding copyright ownership.

# shellcheck source=conf.sh
. ../conf.sh

cd "ns3"

setup() {
  zone="$1"
  echo_i "setting up zone: $zone"
  zonefile="${zone}.db"
  infile="${zone}.db.infile"
  echo "$zone" >>zones
}

# Set in the key state files the Predecessor/Successor fields.
# Key $1 is the predecessor of key $2.
key_successor() {
  id1=$(keyfile_to_key_id "$1")
  id2=$(keyfile_to_key_id "$2")
  echo "Predecessor: ${id1}" >>"${2}.state"
  echo "Successor: ${id2}" >>"${1}.state"
}

# Make lines shorter by storing key states in environment variables.
H="HIDDEN"
R="RUMOURED"
O="OMNIPRESENT"
U="UNRETENTIVE"

#
# The zones at enable-dnssec.$tld represent the various steps of the
# initial signing of a zone.
#

for tld in autosign manual; do
  # Step 1:
  # This is an unsigned zone and named should perform the initial steps of
  # introducing the DNSSEC records in the right order.
  setup step1.enable-dnssec.$tld
  cp template.db.in $zonefile

  # Step 2:
  # The DNSKEY has been published long enough to become OMNIPRESENT.
  setup step2.enable-dnssec.$tld
  # DNSKEY TTL:             300 seconds
  # zone-propagation-delay: 5 minutes (300 seconds)
  # publish-safety:         5 minutes (300 seconds)
  # Total:                  900 seconds
  TpubN="now-900s"
  keytimes="-P ${TpubN} -A ${TpubN}"
  CSK=$($KEYGEN -k enable-dnssec-$tld -l kasp.conf $keytimes $zone 2>keygen.out.$zone.1)
  $SETTIME -s -g $O -k $R $TpubN -r $R $TpubN -d $H $TpubN -z $R $TpubN "$CSK" >settime.out.$zone.1 2>&1
  cat template.db.in "${CSK}.key" >"$infile"
  private_type_record $zone $DEFAULT_ALGORITHM_NUMBER "$CSK" >>"$infile"
  cp $infile $zonefile
  $SIGNER -S -z -x -s now-1h -e now+30d -o $zone -O raw -f "${zonefile}.signed" $infile >signer.out.$zone.1 2>&1

  # Step 3:
  # The zone signatures have been published long enough to become OMNIPRESENT.
  setup step3.enable-dnssec.$tld
  # Passed time since publication:
  # max-zone-ttl:           12 hours (43200 seconds)
  # zone-propagation-delay: 5 minutes (300 seconds)
  TpubN="now-43500s"
  # We can submit the DS now.
  keytimes="-P ${TpubN} -A ${TpubN}"
  CSK=$($KEYGEN -k enable-dnssec-$tld -l kasp.conf $keytimes $zone 2>keygen.out.$zone.1)
  $SETTIME -s -g $O -k $O $TpubN -r $O $TpubN -d $H $TpubN -z $R $TpubN "$CSK" >settime.out.$zone.1 2>&1
  cat template.db.in "${CSK}.key" >"$infile"
  private_type_record $zone $DEFAULT_ALGORITHM_NUMBER "$CSK" >>"$infile"
  cp $infile $zonefile
  $SIGNER -S -z -x -s now-1h -e now+30d -o $zone -O raw -f "${zonefile}.signed" $infile >signer.out.$zone.1 2>&1

  # Step 4:
  # The DS has been submitted long enough ago to become OMNIPRESENT.
  setup step4.enable-dnssec.$tld
  # DS TTL:                    2 hour (7200 seconds)
  # parent-propagation-delay:  1 hour (3600 seconds)
  # Total aditional time:      10800 seconds
  # 43500 + 10800 = 54300
  TpubN="now-54300s"
  TsbmN="now-10800s"
  keytimes="-P ${TpubN} -A ${TpubN} -P sync ${TsbmN}"
  CSK=$($KEYGEN -k enable-dnssec-$tld -l kasp.conf $keytimes $zone 2>keygen.out.$zone.1)
  $SETTIME -s -g $O -P ds $TsbmN -k $O $TpubN -r $O $TpubN -d $R $TpubN -z $O $TsbmN "$CSK" >settime.out.$zone.1 2>&1
  cat template.db.in "${CSK}.key" >"$infile"
  private_type_record $zone $DEFAULT_ALGORITHM_NUMBER "$CSK" >>"$infile"
  cp $infile $zonefile
  $SIGNER -S -z -x -s now-1h -e now+30d -o $zone -O raw -f "${zonefile}.signed" $infile >signer.out.$zone.1 2>&1
done
