// noinspection JSBitwiseOperatorUsage

import type * as ts from "typescript/lib/tsserverlibrary"
import type {
  AreTypesMutuallyAssignableResponse,
  GetCompletionSymbolsResponse,
  GetElementTypeResponse,
  GetResolvedSignatureResponse,
  GetSymbolTypeResponse,
  GetTypeTextResponse,
  Range,
  SymbolResponse,
  TypeRequestKind
} from "./protocol"
import {throwIdeError} from "./utils"

type TS = typeof import('./tsserverlibrary.shim')
export type ReverseMapper = (sourceFile: ts.SourceFile, generatedRange: Range) => {
  sourceRange: Range,
  fileName: string
} | undefined

let lastIdeTypeCheckerId: number = 0

export function getCompletionSymbols(
  ls: ts.LanguageService,
  ts: TS,
  program: ts.Program,
  sourceFileName: string,
  position: number,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetCompletionSymbolsResponse {
  const typeChecker = program.getTypeChecker();
  const ideProjectId = ls.ideProjectId;
  if (!typeChecker.webStormCacheInfo) {
    typeChecker.webStormCacheInfo = {
      ideTypeCheckerId: ++lastIdeTypeCheckerId,
      ideProjectId: ideProjectId,
      requestedTypeIds: new Set<number>(),
      seenTypeIds: new Map<number, ts.Type>(),
      seenSymbolIds: new Map<number, ts.Symbol>()
    };
  }
  else if (typeChecker.webStormCacheInfo.ideProjectId != ideProjectId) {
    console.error(`getCompletionSymbols - wrong ideProjectId. Cached: ${typeChecker.webStormCacheInfo.ideProjectId}, requested ${ideProjectId} `);
    return undefined;
  }
  const cacheInfo = typeChecker.webStormCacheInfo;

  const completions = ls.getCompletionsAtPosition(
    sourceFileName,
    position,
    { includeSymbol: true } as any,
    undefined
  );

  const symbols = (completions?.entries || [])
    .map(e => (e as any).symbol)
    .filter(s => !!s);

  const ctx: ConvertContext = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken);

  let prepared = symbols.map(s => convertSymbol(s, ctx) as Record<string, any>);
  return {
    responseRequired: true,
    response: {
      symbols: prepared,
      ideTypeCheckerId: cacheInfo.ideTypeCheckerId,
      ideProjectId: cacheInfo.ideProjectId
    }
  };
}

export function getElementType(
  ts: TS,
  ideProjectId: number,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  range: Range,
  typeRequestKind: TypeRequestKind,
  forceReturnType: boolean,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetElementTypeResponse {
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)
  return getElementTypeByOffsets(ts, ideProjectId, program, sourceFile, startOffset, endOffset, typeRequestKind, forceReturnType,
    cancellationToken, reverseMapper)
}

export function getResolvedSignature(
  ts: TS,
  ideProjectId: number,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  range: Range,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetResolvedSignatureResponse {
  let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character)
  let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character)

  const typeChecker = program.getTypeChecker();

  // Find the node at the given position
  let node: ts.Node = (ts as any).getTokenAtPosition(sourceFile, startOffset);
  while (node && node.getEnd() < endOffset) {
    node = node.parent;
  }

  if (!node || node === sourceFile) {
    return undefined;
  }

  // Find the call expression
  while (node && !ts.isCallLikeExpression(node)) {
    node = node.parent;
    if (!node || node === sourceFile) {
      return undefined;
    }
  }

  // Get the resolved signature
  const signature = typeChecker.getResolvedSignature(node as ts.CallLikeExpression);
  if (!signature) {
    return undefined;
  }

  // Return the signature information
  let ctx = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken)
  if (!typeChecker.webStormCacheInfo) {
    typeChecker.webStormCacheInfo = {
      ideTypeCheckerId: ++lastIdeTypeCheckerId,
      ideProjectId: ideProjectId,
      requestedTypeIds: new Set<number>(),
      seenTypeIds: new Map<number, ts.Type>(),
      seenSymbolIds: new Map<number, ts.Symbol>()
    }
  }
  else if (typeChecker.webStormCacheInfo.ideProjectId != ideProjectId) {
    console.error(`getElementTypeByOffsets - wrong ideProjectId. Cached: ${typeChecker.webStormCacheInfo.ideProjectId}, requested ${ideProjectId} `)
    return undefined
  }
  const cacheInfo = typeChecker.webStormCacheInfo;
  const prepared = convertSignature(signature, ctx) as Record<string, unknown>;
  prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId
  prepared.ideProjectId = cacheInfo.ideProjectId
  prepared.ideObjectType = "SignatureObject"
  return {
    responseRequired: true,
    response: prepared
  };
}


export function getElementTypeByOffsets(
  ts: TS,
  ideProjectId: number,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  startOffset: number,
  endOffset: number,
  typeRequestKind: TypeRequestKind,
  forceReturnType: boolean,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetElementTypeResponse {
  let node: ts.Node = (ts as any).getTokenAtPosition(sourceFile, startOffset);
  while (node && node.getEnd() < endOffset) {
    node = node.parent;
  }
  if (!node || node === sourceFile) {
    return undefined;
  }

  const contextFlags = typeRequestKindToContextFlags(typeRequestKind)
  const isContextual = contextFlags >= 0
  if (
    (isContextual
        ? !ts.isExpression(node)
        : ts.isStringLiteral(node) || ts.isNumericLiteral(node)
    ) &&
    node.pos === node.parent?.pos && node.end === node.parent?.end
  ) {
    node = node.parent;
  }

  const typeChecker = program.getTypeChecker();
  if (!typeChecker.webStormCacheInfo) {
    typeChecker.webStormCacheInfo = {
      ideTypeCheckerId: ++lastIdeTypeCheckerId,
      ideProjectId: ideProjectId,
      requestedTypeIds: new Set<number>(),
      seenTypeIds: new Map<number, ts.Type>(),
      seenSymbolIds: new Map<number, ts.Symbol>()
    }
  }
  else if (typeChecker.webStormCacheInfo.ideProjectId != ideProjectId) {
    console.error(`getElementTypeByOffsets - wrong ideProjectId. Cached: ${typeChecker.webStormCacheInfo.ideProjectId}, requested ${ideProjectId} `)
    return undefined
  }
  const cacheInfo = typeChecker.webStormCacheInfo;

  let type = contextFlags >= 0 ? (typeChecker as any).getContextualType(node as ts.Expression, contextFlags) : typeChecker.getTypeAtLocation(node);
  if (!type && isContextual && ts.isBinaryExpression(node.parent)) {
    // from getContextualType in services/completions.ts
    const {left, operatorToken, right} = node.parent as ts.BinaryExpression;
    if (ts.isEqualityOperatorKind(operatorToken.kind)) {
      type = typeChecker.getTypeAtLocation(node === right ? left : right)
    }
  }
  if (!type) return undefined;

  let prepared: Record<string, unknown>;

  if (forceReturnType || type.id == null || !cacheInfo.requestedTypeIds.has(type.id)) {
    const ctx: ConvertContext = new ConvertContext(ts, typeChecker,
      reverseMapper, cancellationToken);
    prepared = convertType(type, ctx) as Record<string, unknown>;
    prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
    prepared.ideProjectId = cacheInfo.ideProjectId;
    if (type.id != null) {
      cacheInfo.requestedTypeIds.add(type.id);
    }
  }
  else {
    prepared = {
      id: type.id,
      ideTypeCheckerId: cacheInfo.ideTypeCheckerId,
      ideProjectId: cacheInfo.ideProjectId,
      ideObjectType: "TypeObject",
    };
  }

  return {responseRequired: true, response: prepared}
}

function typeRequestKindToContextFlags(typeRequestKind: TypeRequestKind): number {
  switch (typeRequestKind) {
    case "Default":
      return -1;
    case "Contextual":
      return 0;
    case "ContextualCompletions":
      return 4;
  }
  throw Error("Unexpected typeRequestKind " + typeRequestKind);
}

class ConvertContext {

  private lastCancelCheck: number = 0
  private nextId: number = 0
  private createdObjectsIdeIds: Map<object, number> = new Map()

  constructor(
    public ts: TS,
    public checker: ts.TypeChecker,
    public reverseMapper: ReverseMapper | undefined,
    private cancellationToken: ts.HostCancellationToken | undefined,
  ) {
  }

  checkCancelled(): undefined {
    // Cancellation check might be costly, check only every 1ms
    if (this.cancellationToken && new Date().getTime() > this.lastCancelCheck) {
      this.lastCancelCheck = new Date().getTime()
      if (this.cancellationToken.isCancellationRequested()) {
        throwIdeError("OperationCancelledException")
      }
    }
    return undefined
  }

  getIdeObjectId(obj: object) {
    return this.createdObjectsIdeIds.get(obj)
  }

  registerIdeObject(obj: object) {
    this.checkCancelled()
    const id = this.nextId++
    this.createdObjectsIdeIds.set(obj, id)
    return id
  }

}

function convertType(
  type: ts.Type,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    type,
    ideObjectId => {
      const tscType: Record<string, unknown> = {
        ideObjectId,
        ideObjectType: "TypeObject",
        flags: type.flags,
      }

      // First, we map only types -- to get them on shallower levels, allowing them more room for nested items.
      // Then we map everything else -- these entries will likely get references to already mapped types.

      tscType.aliasTypeArguments = type.aliasTypeArguments
        ?.map(t => convertType(t, ctx))
        ?.filter(isNotNull);

      if (type.flags & ctx.ts.TypeFlags.Object) {
        if ((type as any).target)
          // FIXME internal API
          // FIXME Whole target is not needed, only target.objectFlags needed
          tscType.target = convertType((type as any).target, ctx);

        if ((type as ts.ObjectType).objectFlags & ctx.ts.ObjectFlags.Reference)
          tscType.resolvedTypeArguments = ctx.checker.getTypeArguments(type as ts.TypeReference)
            // TS 4 returns 'this'-type as one of class type arguments, which we don't need
            // FIXME internal API
            .filter(t => !(t as any).isThisType)
            .map((t: ts.Type) => convertType(t, ctx))
            .filter(isNotNull);
      }

      if (type.flags & (ctx.ts.TypeFlags.UnionOrIntersection | ctx.ts.TypeFlags.TemplateLiteral))
        tscType.types = (type as ts.UnionOrIntersectionType).types
          .map(t => convertType(t, ctx))
          .filter(isNotNull);

      if (type.flags & ctx.ts.TypeFlags.Literal && (type as ts.LiteralType).freshType)
        tscType.freshType = convertType((type as ts.LiteralType).freshType, ctx);

      if (type.flags & ctx.ts.TypeFlags.TypeParameter) {
        const constraint = ctx.checker.getBaseConstraintOfType(type);
        if (constraint) tscType.constraint = convertType(constraint, ctx);
      }

      if (type.flags & ctx.ts.TypeFlags.Index)
        tscType.type = convertType((type as ts.IndexType).type, ctx);

      if (type.flags & ctx.ts.TypeFlags.IndexedAccess) {
        tscType.objectType = convertType((type as ts.IndexedAccessType).objectType, ctx);
        tscType.indexType = convertType((type as ts.IndexedAccessType).indexType, ctx);
      }

      if (type.flags & ctx.ts.TypeFlags.Conditional) {
        tscType.checkType = convertType((type as ts.ConditionalType).checkType, ctx);
        tscType.extendsType = convertType((type as ts.ConditionalType).extendsType, ctx);
      }

      if (type.flags & ctx.ts.TypeFlags.Substitution) {
        tscType.baseType = convertType((type as ts.SubstitutionType).baseType, ctx)
      }

      // Now map everything else but types

      if (type.symbol) tscType.symbol = convertSymbol(type.symbol, ctx);
      if (type.aliasSymbol) tscType.aliasSymbol = convertSymbol(type.aliasSymbol, ctx);

      if (type.flags & ctx.ts.TypeFlags.Object) {
        tscType.objectFlags = (type as ts.ObjectType).objectFlags;
      }

      if (type.flags & ctx.ts.TypeFlags.Literal) {
        if (type.flags & ctx.ts.TypeFlags.BigIntLiteral)
          tscType.value = convertPseudoBigInt((type as ts.BigIntLiteralType).value, ctx);
        else
          tscType.value = (type as ts.LiteralType).value;
      }

      if (type.flags & ctx.ts.TypeFlags.EnumLiteral)
        // FIXME 'nameType' is just some random name from generated Kotlin TypeObjectProperty.
        // FIXME This field should have its own name.
        tscType.nameType = getEnumQualifiedName(type, ctx);

      if (type.flags & ctx.ts.TypeFlags.TemplateLiteral)
        tscType.texts = (type as ts.TemplateLiteralType).texts;


      // FIXME internal API
      if (type.flags & ctx.ts.TypeFlags.TypeParameter && (type as any).isThisType)
        tscType.isThisType = true;

      // FIXME internal API
      if (typeof (type as any).intrinsicName === 'string')
        tscType.intrinsicName = (type as any).intrinsicName;


      if ((type as ts.TupleType).elementFlags)
        tscType.elementFlags = (type as ts.TupleType).elementFlags;

      let typeId = type.id
      tscType.id = typeId
      if (typeId) {
        ctx.checker.webStormCacheInfo?.seenTypeIds?.set(typeId, type)
      }

      return tscType;
    },
    ctx,
  );
}

function getEnumQualifiedName(type: ts.Type, ctx: ConvertContext): string | undefined {
  let qName = ''
  // FIXME internal API. Maybe use Node.parent instead?
  let current = (type.symbol as any).parent as ts.Symbol | undefined;
  while (current && !(current.valueDeclaration && ctx.ts.isSourceFile(current.valueDeclaration))) {
    qName = current.escapedName + (qName ? '.' + qName : '');
    current = (current as any).parent;
  }
  return qName || undefined;
}

function convertSymbol(
  symbol: ts.Symbol,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    symbol,
    ideObjectId => {
      const tscSymbol: Record<string, unknown> = {
        ideObjectId,
        ideObjectType: "SymbolObject",
        flags: symbol.flags,
        escapedName: symbol.escapedName as string,
      }

      tscSymbol.declarations = symbol.declarations
        ?.map(d => convertNode(d, ctx))
        ?.filter(isNotNull)

      if (symbol.valueDeclaration)
        tscSymbol.valueDeclaration = convertNode(symbol.valueDeclaration, ctx);

      // FIXME internal API
      if ((symbol as any).links?.type)
        tscSymbol.type = convertType((symbol as any).links?.type, ctx); // TS 5
      else if ((symbol as any).type)
        tscSymbol.type = convertType((symbol as any).type, ctx); // TS 4

      const symbolId = (ctx.ts as any).getSymbolId(symbol)
      ctx.checker.webStormCacheInfo?.seenSymbolIds?.set(symbolId, symbol)
      tscSymbol.id = symbolId

      return tscSymbol;
    },
    ctx,
  );
}

function convertSignature(
  signature: ts.Signature,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    signature,
    ideObjectId => ({
      ideObjectId,
      ideObjectType: "SignatureObject",
      declaration: signature.declaration && signature.declaration?.kind != ctx.ts.SyntaxKind.JSDocSignature ? convertNode(signature.declaration, ctx) : undefined,
      parameters: signature.parameters
        .map(s => convertSymbol(s, ctx))
        .filter(isNotNull),
      typeParameters: signature.typeParameters
        ?.map(s => convertType(s, ctx)),
      resolvedReturnType: convertType(ctx.checker.getReturnTypeOfSignature(signature), ctx),
      flags: (signature as any).flags,
    }),
    ctx,
  )
}

function convertNode(
  node: ts.Node,
  ctx: ConvertContext,
  childReverseMapping: any = undefined,
): object | undefined {
  return findReferenceOrConvert(
    node,
    ideObjectId => {
      if (ctx.ts.isSourceFile(node)) {
        return {
          ideObjectId,
          ideObjectType: "SourceFileObject",
          fileName: childReverseMapping?.fileName ?? node.fileName,
        };
      }
      else {
        const sourceFileParent = getSourceFileParent(node, ctx);
        if (!sourceFileParent || node.pos == -1 || node.end == -1) {
          return {
            ideObjectId,
            ideObjectType: "NodeObject",
            parent: sourceFileParent ? convertNode(sourceFileParent, ctx) : undefined,
          }
        }

        const reverseMapping = runReverseMapper(sourceFileParent, node, ctx);
        return {
          ideObjectId,
          ideObjectType: "NodeObject",
          range: trimRange(ctx, sourceFileParent, node, reverseMapping),
          parent: convertNode(sourceFileParent, ctx, reverseMapping),
          computedProperty: (ctx.ts.isPropertyAssignment(node) || ctx.ts.isPropertySignature(node) || ctx.ts.isPropertyDeclaration(node) || ctx.ts.isMethodSignature(node) || ctx.ts.isMethodDeclaration(node))
            && ctx.ts.isComputedPropertyName(node.name)
            || undefined,
        };
      }
    },
    ctx,
  );
}

function trimRange(ctx: ConvertContext, sourceFileParent: ts.SourceFile, node: ts.Node, reverseMapping: { sourceRange: Range, fileName: string } | undefined): {start: ts.LineAndCharacter, end: ts.LineAndCharacter} {
  let reverseStartPos = reverseMapping?.sourceRange?.start
  let reverseEndPos = reverseMapping?.sourceRange?.end
  let startPos: number
  let endPos: number

  if (reverseStartPos && reverseEndPos) {
    try {
      startPos = ctx.ts.getPositionOfLineAndCharacter(sourceFileParent, reverseStartPos.line, reverseStartPos.character)
      endPos = ctx.ts.getPositionOfLineAndCharacter(sourceFileParent, reverseEndPos.line, reverseEndPos.character)
    } catch (e) {
      // If any problems with the range, don't try to trim it and return as-is
      return {
        start: reverseStartPos,
        end: reverseEndPos
      }
    }
  }
  else {
    startPos = node.pos
    endPos = node.end
  }

  // Trim spaces around the node
  let text = sourceFileParent.text
  while (startPos < endPos && isWhitespace(text.charAt(startPos))) {
    startPos++
  }
  while (endPos > startPos && isWhitespace(text.charAt(endPos - 1))) {
    endPos--
  }
  return {
    start: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, startPos),
    end: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, endPos)
  }
}

function isWhitespace(ch: string) {
  return ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r'
}

function getSourceFileParent(node: ts.Node, ctx: ConvertContext): ts.SourceFile | undefined {
  if (ctx.ts.isSourceFile(node)) return undefined;
  let current = node.parent;
  while (current) {
    if (ctx.ts.isSourceFile(current)) return current;
    current = current.parent;
  }
  return undefined
}

function runReverseMapper(sourceFileParent: ts.SourceFile, node: ts.Node, ctx: ConvertContext) {
  if (!ctx.reverseMapper) return undefined;
  let startOffset = node.pos
  let endOffset = node.end
  // Trim spaces around the node
  const text = sourceFileParent.text
  while (' \t\n\r\v'.indexOf(text.charAt(startOffset)) >= 0 && startOffset < endOffset) {
    startOffset++
  }
  while (endOffset > 0 && ' \t\n\r\v'.indexOf(text.charAt(endOffset - 1)) >= 0 && startOffset < endOffset) {
    endOffset--
  }
  return ctx.reverseMapper(
    sourceFileParent,
    {
      start: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, startOffset),
      end: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, endOffset),
    },
  )
}

function convertPseudoBigInt(
  pseudoBigInt: ts.PseudoBigInt,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    pseudoBigInt,
    ideObjectId => ({
      ideObjectId,
      ...pseudoBigInt,
    }),
    ctx,
  );
}

function convertIndexInfo(
  indexInfo: ts.IndexInfo,
  ctx: ConvertContext,
): object | undefined {
  return findReferenceOrConvert(
    indexInfo,
    ideObjectId => {
      const result: Record<string, unknown> = {
        ideObjectId,
        ideObjectType: "IndexInfo",
        keyType: convertType(indexInfo.keyType, ctx),
        type: convertType(indexInfo.type, ctx),
        isReadonly: indexInfo.isReadonly,
      }
      if (indexInfo.declaration) {
        result.declaration = convertNode(indexInfo.declaration, ctx)
      }
      return result
    },
    ctx,
  );
}

function findReferenceOrConvert(
  sourceObj: object,
  convertTarget: (ideObjectId: number) => object,
  ctx: ConvertContext,
): object | undefined {
  let ideObjectId = ctx.getIdeObjectId(sourceObj)
  if (ideObjectId) {
    return {ideObjectIdRef: ideObjectId};
  }
  ideObjectId = ctx.registerIdeObject(sourceObj)

  const newObject = convertTarget(ideObjectId);

  return newObject;
}


function isNotNull<T>(t: T | undefined): t is T {
  return t != null
}

export function getSymbolType(
  ts: TS,
  program: ts.Program,
  symbolId: number,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetSymbolTypeResponse {
  const typeChecker = program.getTypeChecker()
  const cacheInfo = typeChecker.webStormCacheInfo
  if (!cacheInfo) {
    return undefined
  }
  let symbol = cacheInfo.seenSymbolIds.get(symbolId)
  if (!symbol) {
    return undefined
  }

  const ctx: ConvertContext = new ConvertContext(ts, typeChecker,
    reverseMapper, cancellationToken);
  let prepared: Record<string, unknown> = {}
  if ((ctx.checker as any).getTypeOfSymbol) {
    prepared = convertType((ctx.checker as any).getTypeOfSymbol(symbol), ctx) as Record<string, unknown>
  }
  else if (symbol.valueDeclaration) {
    prepared = convertType(ctx.checker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration), ctx) as Record<string, unknown>
  }
  prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
  prepared.ideProjectId = cacheInfo.ideProjectId;

  return {responseRequired: true, response: prepared}
}

export function getTypeText(
  ts: TS,
  program: ts.Program,
  symbolId: number,
  flags?: number,
): GetTypeTextResponse {
  const typeChecker = program.getTypeChecker();
  const cacheInfo = typeChecker.webStormCacheInfo;

  if (!cacheInfo) return undefined;

  const symbol = cacheInfo.seenSymbolIds.get(symbolId);
  if (!symbol) return undefined;

  const type = (typeChecker as any).getTypeOfSymbol(symbol);
  if (!type) return undefined;

  const formatFlags = flags ?? (
    ts.TypeFormatFlags.AllowUniqueESSymbolType |
    ts.TypeFormatFlags.UseAliasDefinedOutsideCurrentScope
  );

  const enclosingDeclaration = symbol.declarations && symbol.declarations.length === 1 ? symbol.declarations[0] : undefined;

  return {
    response: {
      typeText: typeChecker.typeToString(type, enclosingDeclaration, formatFlags),
    },
    responseRequired: true
  };
}

export function getTypeProperties(
  ts: TS,
  program: ts.Program,
  typeId: number,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): GetElementTypeResponse {
  const typeChecker = program.getTypeChecker()
  const cacheInfo = typeChecker.webStormCacheInfo
  if (!cacheInfo) {
    return undefined
  }

  let type = cacheInfo.seenTypeIds.get(typeId)
  if (!type) {
    return undefined
  }

  const ctx: ConvertContext = new ConvertContext(ts, typeChecker,
    reverseMapper, cancellationToken);

  let prepared = convertTypeProperties(type, ctx) as Record<string, any>
  prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId
  prepared.ideProjectId = cacheInfo.ideProjectId;

  return {responseRequired: true, response: prepared}
}

/**
 * It is not exactly equivalent to getTypeProperties() and searching by name.
 * The current implementation of getTypeProperties() doesn't support non-object types like strings. This method includes properties
 * from getApparentType(), like charAt.
 * Also, it additionally includes properties from Object and Function types, which can't even be obtained from getApparentType().
 */
export function getTypeProperty(
  ts: TS,
  program: ts.Program,
  typeId: number,
  propertyName: string,
  cancellationToken: ts.HostCancellationToken | undefined,
  reverseMapper?: ReverseMapper,
): SymbolResponse {
  const typeChecker = program.getTypeChecker()
  const cacheInfo = typeChecker.webStormCacheInfo
  if (!cacheInfo) {
    return undefined
  }

  let type = cacheInfo.seenTypeIds.get(typeId)
  if (!type) {
    return undefined
  }

  const ctx: ConvertContext = new ConvertContext(ts, typeChecker,
    reverseMapper, cancellationToken);

  let property = type.getProperty(propertyName)
  let prepared = property ? convertSymbol(property, ctx) as Record<string, any> : undefined

  return {responseRequired: true, response: prepared}
}

function convertTypeProperties(type: ts.Type, ctx: ConvertContext): object | undefined {
  return findReferenceOrConvert(type, ideObjectId => {
    let prepared: Record<string, unknown> = {
      ideObjectId,
      ideObjectType: "TypeObject",
      flags: type.flags,
      objectFlags: (type as ts.ObjectType).objectFlags
    }

    if (type.flags & ctx.ts.TypeFlags.Object) {
      assignObjectTypeProperties(type as ts.ObjectType, ctx, prepared)
    }
    if (type.flags & ctx.ts.TypeFlags.UnionOrIntersection) {
      assignUnionOrIntersectionTypeProperties(type as ts.UnionOrIntersectionType, ctx, prepared)
    }
    if (type.flags & ctx.ts.TypeFlags.Conditional) {
      assignConditionalTypeProperties(type as ts.ConditionalType, ctx, prepared)
    }
    return prepared
  }, ctx)
}

function assignObjectTypeProperties(type: ts.ObjectType, ctx: ConvertContext, tscType: Record<string, unknown>) {
  tscType.constructSignatures = type.getConstructSignatures()
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
  tscType.callSignatures = type.getCallSignatures()
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
  tscType.properties = type.getProperties()
    .map((p: ts.Symbol) => convertSymbol(p, ctx))
    .filter(isNotNull);
  tscType.indexInfos = ctx.checker.getIndexInfosOfType &&
    ctx.checker.getIndexInfosOfType(type)
      .map((info: ts.IndexInfo) => convertIndexInfo(info, ctx))
      .filter(isNotNull);
}

function assignUnionOrIntersectionTypeProperties(type: ts.UnionOrIntersectionType, ctx: ConvertContext, tscType: Record<string, unknown>) {
  tscType.resolvedProperties = ctx.checker.getPropertiesOfType(type)
    .map((p: ts.Symbol) => convertSymbol(p, ctx))
    .filter(isNotNull);
  tscType.callSignatures = ctx.checker.getSignaturesOfType(type, ctx.ts.SignatureKind.Call)
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
  tscType.constructSignatures = ctx.checker.getSignaturesOfType(type, ctx.ts.SignatureKind.Construct)
    .map((s: ts.Signature) => convertSignature(s, ctx))
    .filter(isNotNull);
}

function assignConditionalTypeProperties(type: ts.ConditionalType, ctx: ConvertContext, tscType: Record<string, unknown>) {
  ctx.checker.getPropertiesOfType(type); // In TS 4 this triggers calculation of true and false types
  if (type.resolvedTrueType)
    tscType.resolvedTrueType = convertType(type.resolvedTrueType as ts.Type, ctx);
  if (type.resolvedFalseType)
    tscType.resolvedFalseType = convertType(type.resolvedFalseType as ts.Type, ctx);
}

export function areTypesMutuallyAssignable(
  ts: TS,
  program: ts.Program,
  type1Id: number,
  type2Id: number,
  cancellationToken: ts.HostCancellationToken | undefined,
): AreTypesMutuallyAssignableResponse {
  const checker = program?.getTypeChecker()
  if (!checker) return undefined

  const type1 = checker.webStormCacheInfo?.seenTypeIds?.get(type1Id)
  if (!type1) return undefined
  const type2 = checker.webStormCacheInfo?.seenTypeIds?.get(type2Id)
  if (!type2) return undefined

  const {isTypeAssignableTo} = checker as any
  if (!isTypeAssignableTo) return undefined

  const ctx = new ConvertContext(ts, checker, undefined, cancellationToken)
  const areMutuallyAssignable = isTypeAssignableTo(type1, type2) && (ctx.checkCancelled() ?? isTypeAssignableTo(type2, type1))

  return {
    response: {areMutuallyAssignable},
    responseRequired: true,
  }
}
