use itertools::{Either, Itertools};
use tombi_ast::{AstNode, DanglingCommentGroupOr, algo::ancestors_at_position};
use tombi_document_tree::IntoDocumentTreeAndErrors;
use tombi_schema_store::SchemaContext;
use tombi_text::IntoLsp;
use tower_lsp::lsp_types::{HoverParams, TextDocumentPositionParams};

use crate::{
    backend,
    config_manager::ConfigSchemaStore,
    hover::{HoverContent, get_document_comment_directive_hover_content, get_hover_content},
};

pub async fn handle_hover(
    backend: &backend::Backend,
    params: HoverParams,
) -> Result<Option<HoverContent>, tower_lsp::jsonrpc::Error> {
    log::info!("handle_hover");
    log::trace!("{:?}", params);

    let HoverParams {
        text_document_position_params:
            TextDocumentPositionParams {
                text_document,
                position,
            },
        ..
    } = params;
    let text_document_uri = text_document.uri.into();

    let ConfigSchemaStore {
        config,
        schema_store,
        ..
    } = backend
        .config_manager
        .config_schema_store_for_uri(&text_document_uri)
        .await;

    if !config
        .lsp
        .as_ref()
        .and_then(|server| server.hover.as_ref())
        .and_then(|hover| hover.enabled)
        .unwrap_or_default()
        .value()
    {
        log::debug!("`server.hover.enabled` is false");
        return Ok(None);
    }

    let document_sources = backend.document_sources.read().await;
    let Some(document_source) = document_sources.get(&text_document_uri) else {
        return Ok(None);
    };

    let root = document_source.ast();
    let toml_version = document_source.toml_version;
    let line_index = document_source.line_index();

    let position = position.into_lsp(line_index);

    let source_schema = schema_store
        .resolve_source_schema_from_ast(root, Some(Either::Left(&text_document_uri)))
        .await
        .ok()
        .flatten();

    let source_path = text_document_uri.to_file_path().ok();
    // Check if position is in a #:tombi comment directive
    if let Some(content) =
        get_document_comment_directive_hover_content(root, position, source_path.as_deref()).await
    {
        return Ok(Some(content));
    }

    let Some((keys, range)) = get_hover_keys_with_range(root, position, toml_version).await else {
        log::debug!("Failed to get hover keys with range");
        return Ok(None);
    };

    if keys.is_empty() && range.is_none() {
        log::debug!("Keys and range are empty");
        return Ok(None);
    }

    let document_tree = document_source.document_tree();

    let mut hover_content = get_hover_content(
        document_tree,
        position,
        &keys,
        &SchemaContext {
            toml_version,
            root_schema: source_schema
                .as_ref()
                .and_then(|s| s.root_schema.as_deref()),
            sub_schema_uri_map: source_schema.as_ref().map(|s| &s.sub_schema_uri_map),
            schema_visits: Default::default(),
            store: &schema_store,
            strict: None,
        },
    )
    .await;

    if let Some(HoverContent::Value(hover_value_content)) = &mut hover_content {
        hover_value_content.range = range;
    }
    Ok(hover_content)
}

pub async fn get_hover_keys_with_range(
    root: &tombi_ast::Root,
    position: tombi_text::Position,
    toml_version: tombi_config::TomlVersion,
) -> Option<(Vec<tombi_document_tree::Key>, Option<tombi_text::Range>)> {
    let mut keys_vec = vec![];
    let mut hover_range = None;

    for node in ancestors_at_position(root.syntax(), position) {
        if let Some(array) = tombi_ast::Array::cast(node.to_owned()) {
            let on_leading_comment = array
                .leading_comments()
                .any(|comment| comment.syntax().range().contains(position));
            let on_bracket_start_trailing_comment = array
                .bracket_start_trailing_comment()
                .is_some_and(|comment| comment.syntax().range().contains(position));
            let on_trailing_comment = array
                .trailing_comment()
                .is_some_and(|comment| comment.syntax().range().contains(position));

            if hover_range.is_none() && (on_leading_comment || on_bracket_start_trailing_comment) {
                hover_range = Some(array.syntax().range());
            } else if hover_range.is_none() && on_trailing_comment {
                hover_range = Some(key_value_parent_or_self_range(
                    &array,
                    array.syntax().range(),
                ));
            } else {
                for groups in array.value_with_comma_groups() {
                    match groups {
                        DanglingCommentGroupOr::DanglingCommentGroup(comment_group) => {
                            if comment_group
                                .comments()
                                .any(|comment| comment.syntax().range().contains(position))
                            {
                                hover_range = Some(comment_group.syntax().range());
                                break;
                            }
                        }
                        DanglingCommentGroupOr::ItemGroup(value_group) => {
                            for (value_or_key_value, comma) in
                                value_group.value_or_key_values_with_comma()
                            {
                                if hover_range.is_none() {
                                    let mut range = value_or_key_value.range();
                                    if let Some(comma) = comma {
                                        range += comma.range()
                                    };
                                    if range.contains(position) {
                                        hover_range = Some(range);
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } else if let Some(inline_table) = tombi_ast::InlineTable::cast(node.to_owned()) {
            let on_leading_comment = inline_table
                .leading_comments()
                .any(|comment| comment.syntax().range().contains(position));
            let on_brace_start_trailing_comment = inline_table
                .brace_start_trailing_comment()
                .is_some_and(|comment| comment.syntax().range().contains(position));
            let on_trailing_comment = inline_table
                .trailing_comment()
                .is_some_and(|comment| comment.syntax().range().contains(position));

            if hover_range.is_none() && on_leading_comment || on_brace_start_trailing_comment {
                hover_range = Some(inline_table.syntax().range());
            } else if hover_range.is_none() && on_trailing_comment {
                hover_range = Some(key_value_parent_or_self_range(
                    &inline_table,
                    inline_table.syntax().range(),
                ));
            } else {
                for groups in inline_table.key_value_with_comma_groups() {
                    match groups {
                        DanglingCommentGroupOr::DanglingCommentGroup(comment_group) => {
                            if comment_group
                                .comments()
                                .any(|comment| comment.syntax().range().contains(position))
                            {
                                hover_range = Some(comment_group.syntax().range());
                                break;
                            }
                        }
                        DanglingCommentGroupOr::ItemGroup(key_value_group) => {
                            for (key_value, comma) in key_value_group.key_values_with_comma() {
                                if hover_range.is_none() {
                                    let mut range = key_value.range();
                                    if let Some(comma) = comma {
                                        range += comma.range()
                                    };
                                    if range.contains(position) {
                                        hover_range = Some(range);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        };

        let keys = if let Some(kv) = tombi_ast::KeyValue::cast(node.to_owned()) {
            if hover_range.is_none() {
                hover_range =
                    Some(append_comma_range_if_exists(&kv, position).unwrap_or_else(|| kv.range()));
            }
            kv.keys()
        } else if let Some(table) = tombi_ast::Table::cast(node.to_owned()) {
            let header = table.header();
            if let Some(header) = &header
                && hover_range.is_none()
                && (header
                    .keys()
                    .last()
                    .is_none_or(|key| key.syntax().range().contains(position))
                    || table
                        .header_leading_comments()
                        .any(|comment| comment.syntax().range().contains(position))
                    || table
                        .header_trailing_comment()
                        .is_some_and(|comment| comment.syntax().range().contains(position))
                    || table.dangling_comment_groups().any(|comment_group| {
                        comment_group
                            .comments()
                            .any(|comment| comment.syntax().range().contains(position))
                    }))
            {
                let mut range = table.syntax().range();
                if let Some(max_end) = table
                    .sub_tables()
                    .map(|subtable| subtable.syntax().range().end)
                    .max()
                {
                    range.end = max_end;
                }
                hover_range = Some(range);
            } else {
                for group in table
                    .key_value_groups()
                    .filter_map(DanglingCommentGroupOr::into_dangling_comment_group)
                {
                    if group
                        .comments()
                        .any(|comment| comment.syntax().range().contains(position))
                    {
                        hover_range = Some(group.syntax().range());
                        break;
                    }
                }
            }

            header
        } else if let Some(array_of_table) = tombi_ast::ArrayOfTable::cast(node.to_owned()) {
            let header = array_of_table.header();
            if let Some(header) = &header
                && hover_range.is_none()
                && (header
                    .keys()
                    .last()
                    .is_none_or(|key| key.syntax().range().contains(position))
                    || array_of_table
                        .header_leading_comments()
                        .any(|comment| comment.syntax().range().contains(position))
                    || array_of_table
                        .header_trailing_comment()
                        .is_some_and(|comment| comment.syntax().range().contains(position))
                    || array_of_table
                        .dangling_comment_groups()
                        .any(|comment_group| {
                            comment_group
                                .comments()
                                .any(|comment| comment.syntax().range().contains(position))
                        }))
            {
                let mut range = array_of_table.syntax().range();
                if let Some(max_end) = array_of_table
                    .sub_tables()
                    .map(|subtable| subtable.syntax().range().end)
                    .max()
                {
                    range.end = max_end;
                }
                hover_range = Some(range);
            } else {
                for group in array_of_table
                    .key_value_groups()
                    .filter_map(DanglingCommentGroupOr::into_dangling_comment_group)
                {
                    if group
                        .comments()
                        .any(|comment| comment.syntax().range().contains(position))
                    {
                        hover_range = Some(group.syntax().range());
                        break;
                    }
                }
            }

            header
        } else if let Some(root) = tombi_ast::Root::cast(node.to_owned()) {
            if hover_range.is_none()
                && (root.dangling_comment_groups().any(|comment_group| {
                    comment_group
                        .comments()
                        .any(|comment| comment.syntax().range().contains(position))
                }))
            {
                hover_range = Some(root.syntax().range());
            } else {
                for group in root
                    .key_value_groups()
                    .filter_map(DanglingCommentGroupOr::into_dangling_comment_group)
                {
                    if group
                        .comments()
                        .any(|comment| comment.syntax().range().contains(position))
                    {
                        hover_range = Some(group.syntax().range());
                        break;
                    }
                }
            }

            continue;
        } else {
            continue;
        };

        let Some(keys) = keys else { continue };

        let keys = if keys.range().contains(position) {
            let mut new_keys = Vec::with_capacity(keys.keys().count());
            for key in keys
                .keys()
                .take_while(|key| key.token().unwrap().range().start <= position)
            {
                let document_tree_key = key.into_document_tree_and_errors(toml_version).tree;
                if let Some(document_tree_key) = document_tree_key {
                    new_keys.push(document_tree_key);
                }
            }
            new_keys
        } else {
            let mut new_keys = Vec::with_capacity(keys.keys().count());
            for key in keys.keys() {
                let document_tree_key = key.into_document_tree_and_errors(toml_version).tree;
                if let Some(document_tree_key) = document_tree_key {
                    new_keys.push(document_tree_key);
                }
            }
            new_keys
        };

        if hover_range.is_none() {
            hover_range = keys.iter().map(|key| key.range()).reduce(|k1, k2| k1 + k2);
        }

        keys_vec.push(keys);
    }

    Some((
        keys_vec.into_iter().rev().flatten().collect_vec(),
        hover_range,
    ))
}

fn key_value_parent_or_self_range<N: AstNode>(
    node: &N,
    fallback_range: tombi_text::Range,
) -> tombi_text::Range {
    node.syntax()
        .parent()
        .and_then(|parent| {
            tombi_ast::KeyValue::cast(parent.clone())
                .or_else(|| parent.parent().and_then(tombi_ast::KeyValue::cast))
        })
        .map_or(fallback_range, |key_value| key_value.range())
}

fn append_comma_range_if_exists(
    node: &impl AstNode,
    position: tombi_text::Position,
) -> Option<tombi_text::Range> {
    let node_key_value = tombi_ast::KeyValue::cast(node.syntax().clone());

    for syntax_node in node.syntax().ancestors() {
        if let Some(group) = tombi_ast::KeyValueWithCommaGroup::cast(syntax_node.clone()) {
            if let Some(target_key_value) = node_key_value.as_ref() {
                for (item, comma) in group.key_values_with_comma() {
                    if item.syntax() == target_key_value.syntax() {
                        return Some(
                            comma.map_or(item.range(), |comma| item.range() + comma.range()),
                        );
                    }
                }
            }

            if let Some(range) = with_comma_item_range_contains_position(
                group
                    .key_values_with_comma()
                    .map(|(item, comma)| (item.range(), comma.map(|comma| comma.range()))),
                position,
            ) {
                return Some(range);
            }
        } else if let Some(group) = tombi_ast::ValueWithCommaGroup::cast(syntax_node) {
            if let Some(range) = with_comma_item_range_contains_position(
                group
                    .value_or_key_values_with_comma()
                    .map(|(item, comma)| (item.range(), comma.map(|comma| comma.range()))),
                position,
            ) {
                return Some(range);
            }
        }
    }

    None
}

fn with_comma_item_range_contains_position<I>(
    items_with_comma: I,
    position: tombi_text::Position,
) -> Option<tombi_text::Range>
where
    I: IntoIterator<Item = (tombi_text::Range, Option<tombi_text::Range>)>,
{
    for (item_range, comma_range) in items_with_comma {
        let range = comma_range.map_or(item_range, |comma_range| item_range + comma_range);
        if range.contains(position) {
            return Some(range);
        }
    }

    None
}

#[cfg(test)]
mod tests {
    use super::*;
    use tombi_config::TomlVersion;
    use tombi_parser::parse;
    use tombi_text::{Position, RelativePosition};

    fn parse_root_and_position_with_marker(
        source_with_marker: &str,
    ) -> (tombi_ast::Root, Position) {
        let marker = '█';
        let marker_index = source_with_marker.find(marker).unwrap();

        let mut source = source_with_marker.to_string();
        source.remove(marker_index);

        let root = tombi_ast::Root::cast(parse(&source).into_syntax_node()).unwrap();
        let position = Position::default() + RelativePosition::of(&source[..marker_index]);

        (root, position)
    }

    #[tokio::test]
    async fn array_trailing_comment_hover_range_includes_key() {
        let (root, position) = parse_root_and_position_with_marker(
            r#"authors = ["ya7010 <ya7010@outlook.com>"]  # a█aa"#,
        );

        let (_, hover_range) = get_hover_keys_with_range(&root, position, TomlVersion::V1_0_0)
            .await
            .unwrap();

        let key_value = root.key_values().next().unwrap();
        pretty_assertions::assert_eq!(hover_range, Some(key_value.range()));
    }

    #[tokio::test]
    async fn inline_table_trailing_comment_hover_range_includes_key() {
        let (root, position) =
            parse_root_and_position_with_marker(r#"dependency = { version = "1.0" }  # a█aa"#);

        let (_, hover_range) = get_hover_keys_with_range(&root, position, TomlVersion::V1_0_0)
            .await
            .unwrap();

        let key_value = root.key_values().next().unwrap();
        pretty_assertions::assert_eq!(hover_range, Some(key_value.range()));
    }

    #[tokio::test]
    async fn inline_table_key_hover_range_includes_comma_and_trailing_comment() {
        let (root, position) = parse_root_and_position_with_marker(
            r#"
            array5 = [
              {
                # key1 leading comment1
                # key1 leading comment2
                key█1 = 1
                # key1 comma leading comment
                ,  # key1 comma trailing comment
              },
            ]
            "#,
        );

        let (_, hover_range) = get_hover_keys_with_range(&root, position, TomlVersion::V1_0_0)
            .await
            .unwrap();

        let array = match root.key_values().next().unwrap().value().unwrap() {
            tombi_ast::Value::Array(array) => array,
            _ => panic!("expected array"),
        };

        let inline_table = match array.values().next().unwrap() {
            tombi_ast::Value::InlineTable(inline_table) => inline_table,
            _ => panic!("expected inline table"),
        };
        let (key_value, comma) = inline_table.key_values_with_comma().next().unwrap();
        let expected_range =
            comma.map_or(key_value.range(), |comma| key_value.range() + comma.range());

        pretty_assertions::assert_eq!(hover_range, Some(expected_range));
    }

    #[tokio::test]
    async fn inline_table_key_in_array_hover_range_includes_array_item_comma_and_trailing_comment()
    {
        let (root, position) = parse_root_and_position_with_marker(
            r#"
            array5 = [
              {
                key█1 = 1,
              }, # array item trailing comment
            ]
            "#,
        );

        let (_, hover_range) = get_hover_keys_with_range(&root, position, TomlVersion::V1_0_0)
            .await
            .unwrap();

        let array = match root.key_values().next().unwrap().value().unwrap() {
            tombi_ast::Value::Array(array) => array,
            _ => panic!("expected array"),
        };

        let inline_table = match array.values().next().unwrap() {
            tombi_ast::Value::InlineTable(inline_table) => inline_table,
            _ => panic!("expected inline table"),
        };
        let (key_value, comma) = inline_table.key_values_with_comma().next().unwrap();
        let expected_range =
            comma.map_or(key_value.range(), |comma| key_value.range() + comma.range());

        pretty_assertions::assert_eq!(hover_range, Some(expected_range));
    }

    #[tokio::test]
    async fn nested_array_hover_range_uses_innermost_value_with_comma_group() {
        let (root, position) = parse_root_and_position_with_marker(
            r#"
            array5 = [
              [
                { key█1 = 1 }, # inner array item trailing comment
              ], # outer array item trailing comment
            ]
            "#,
        );

        let (_, hover_range) = get_hover_keys_with_range(&root, position, TomlVersion::V1_0_0)
            .await
            .unwrap();

        let outer_array = match root.key_values().next().unwrap().value().unwrap() {
            tombi_ast::Value::Array(array) => array,
            _ => panic!("expected outer array"),
        };
        let inner_array = match outer_array.values().next().unwrap() {
            tombi_ast::Value::Array(array) => array,
            _ => panic!("expected inner array"),
        };
        let inline_table = match inner_array.values().next().unwrap() {
            tombi_ast::Value::InlineTable(inline_table) => inline_table,
            _ => panic!("expected inline table"),
        };
        let (key_value, comma) = inline_table.key_values_with_comma().next().unwrap();
        let expected_range =
            comma.map_or(key_value.range(), |comma| key_value.range() + comma.range());

        pretty_assertions::assert_eq!(hover_range, Some(expected_range));
    }
}
