use std::sync::Arc;

use crate::{
    compiler,
    node::SchemaNode,
    paths::{LazyEvaluationPath, Location},
    regex::{analyze_pattern, is_ecma_whitespace, LiteralMatchError, PatternOptimization},
    validator::Validate as _,
    ValidationContext,
};
use ahash::AHashMap;
use serde_json::{Map, Value};

use crate::ValidationError;

/// A compiled pattern that can be a literal optimized match or a full regex.
#[derive(Debug, Clone)]
pub(crate) enum CompiledPattern<R> {
    /// Simple prefix match using `starts_with()`.
    Prefix(Arc<str>),
    /// Exact match using `==` - for `^...$` patterns.
    Exact(Arc<str>),
    /// `^(a|b|c)$` — linear scan over a small sorted array of alternatives.
    Alternation(Arc<[String]>, Arc<str>),
    /// `^\S*$` — no ECMA-262 whitespace characters.
    NoWhitespace(Arc<str>),
    /// Full regex pattern.
    Regex(R),
}

impl<R: crate::regex::RegexEngine> crate::regex::RegexEngine for CompiledPattern<R> {
    type Error = LiteralMatchError;

    #[inline]
    fn is_match(&self, text: &str) -> Result<bool, Self::Error> {
        match self {
            CompiledPattern::Prefix(prefix) => Ok(text.starts_with(prefix.as_ref())),
            CompiledPattern::Exact(exact) => Ok(text == exact.as_ref()),
            CompiledPattern::Alternation(alts, _) => Ok(alts.iter().any(|a| a.as_str() == text)),
            CompiledPattern::NoWhitespace(_) => Ok(!text.chars().any(is_ecma_whitespace)),
            // Treat regex errors as non-match for compatibility
            CompiledPattern::Regex(re) => Ok(re.is_match(text).unwrap_or(false)),
        }
    }

    fn pattern(&self) -> &str {
        match self {
            CompiledPattern::Prefix(prefix) => prefix.as_ref(),
            CompiledPattern::Exact(exact) => exact.as_ref(),
            CompiledPattern::Alternation(_, original) | CompiledPattern::NoWhitespace(original) => {
                original.as_ref()
            }
            CompiledPattern::Regex(re) => re.pattern(),
        }
    }
}

pub(crate) type FancyRegexValidators = Vec<(CompiledPattern<fancy_regex::Regex>, SchemaNode)>;
pub(crate) type RegexValidators = Vec<(CompiledPattern<regex::Regex>, SchemaNode)>;

/// A value that can look up property validators by name.
pub(crate) trait PropertiesValidatorsMap: Send + Sync {
    fn get_validator(&self, property: &str) -> Option<&SchemaNode>;
    fn get_key_validator(&self, property: &str) -> Option<(&String, &SchemaNode)>;
}

/// Threshold for switching from linear scan to `HashMap`.
pub(crate) const HASHMAP_THRESHOLD: usize = 15;

pub(crate) type SmallValidatorsMap = Vec<(String, SchemaNode)>;
pub(crate) type BigValidatorsMap = AHashMap<String, SchemaNode>;

impl PropertiesValidatorsMap for SmallValidatorsMap {
    #[inline]
    fn get_validator(&self, property: &str) -> Option<&SchemaNode> {
        for (prop, node) in self {
            if prop == property {
                return Some(node);
            }
        }
        None
    }
    #[inline]
    fn get_key_validator(&self, property: &str) -> Option<(&String, &SchemaNode)> {
        for (prop, node) in self {
            if prop == property {
                return Some((prop, node));
            }
        }
        None
    }
}

impl PropertiesValidatorsMap for BigValidatorsMap {
    #[inline]
    fn get_validator(&self, property: &str) -> Option<&SchemaNode> {
        self.get(property)
    }

    #[inline]
    fn get_key_validator(&self, property: &str) -> Option<(&String, &SchemaNode)> {
        self.get_key_value(property)
    }
}

pub(crate) fn compile_small_map<'a>(
    ctx: &compiler::Context,
    map: &'a Map<String, Value>,
) -> Result<SmallValidatorsMap, ValidationError<'a>> {
    let mut properties = Vec::with_capacity(map.len());
    let kctx = ctx.new_at_location("properties");
    for (key, subschema) in map {
        let pctx = kctx.new_at_location(key.as_str());
        properties.push((
            key.clone(),
            compiler::compile(&pctx, pctx.as_resource_ref(subschema))?,
        ));
    }
    Ok(properties)
}

pub(crate) fn compile_big_map<'a>(
    ctx: &compiler::Context,
    map: &'a Map<String, Value>,
) -> Result<BigValidatorsMap, ValidationError<'a>> {
    let mut properties = AHashMap::with_capacity(map.len());
    let kctx = ctx.new_at_location("properties");
    for (key, subschema) in map {
        let pctx = kctx.new_at_location(key.as_str());
        properties.insert(
            key.clone(),
            compiler::compile(&pctx, pctx.as_resource_ref(subschema))?,
        );
    }
    Ok(properties)
}

pub(crate) fn are_properties_valid<M, F>(
    prop_map: &M,
    props: &Map<String, Value>,
    ctx: &mut ValidationContext,
    check: F,
) -> bool
where
    M: PropertiesValidatorsMap,
    F: Fn(&Value, &mut ValidationContext) -> bool,
{
    for (property, instance) in props {
        if let Some(validator) = prop_map.get_validator(property) {
            if !validator.is_valid(instance, ctx) {
                return false;
            }
        } else if !check(instance, ctx) {
            return false;
        }
    }
    true
}

/// Create a vector of pattern-validators pairs.
/// Uses prefix optimization when patterns are simple `^prefix` patterns.
#[inline]
pub(crate) fn compile_fancy_regex_patterns<'a>(
    ctx: &compiler::Context,
    obj: &'a Map<String, Value>,
) -> Result<FancyRegexValidators, ValidationError<'a>> {
    let kctx = ctx.new_at_location("patternProperties");
    let mut compiled_patterns = Vec::with_capacity(obj.len());
    for (pattern, subschema) in obj {
        let pctx = kctx.new_at_location(pattern.as_str());
        let compiled_pattern = match analyze_pattern(pattern) {
            Some(PatternOptimization::Prefix(prefix)) => CompiledPattern::Prefix(Arc::from(prefix)),
            Some(PatternOptimization::Exact(exact)) => CompiledPattern::Exact(Arc::from(exact)),
            Some(PatternOptimization::Alternation(alts)) => CompiledPattern::Alternation(
                Arc::from(alts.into_boxed_slice()),
                Arc::from(pattern.as_str()),
            ),
            Some(PatternOptimization::NoWhitespace) => {
                CompiledPattern::NoWhitespace(Arc::from(pattern.as_str()))
            }
            None => {
                let regex = ctx.get_or_compile_regex(pattern).map_err(|()| {
                    ValidationError::format(
                        kctx.location().clone(),
                        LazyEvaluationPath::SameAsSchemaPath,
                        Location::new(),
                        subschema,
                        "regex",
                    )
                })?;
                CompiledPattern::Regex((*regex).clone())
            }
        };
        let node = compiler::compile(&pctx, pctx.as_resource_ref(subschema))?;
        compiled_patterns.push((compiled_pattern, node));
    }
    Ok(compiled_patterns)
}

/// Create a vector of pattern-validators pairs using standard regex.
/// Uses literal optimizations when patterns are simple prefix or exact-match patterns.
#[inline]
pub(crate) fn compile_regex_patterns<'a>(
    ctx: &compiler::Context,
    obj: &'a Map<String, Value>,
) -> Result<RegexValidators, ValidationError<'a>> {
    let kctx = ctx.new_at_location("patternProperties");
    let mut compiled_patterns = Vec::with_capacity(obj.len());
    for (pattern, subschema) in obj {
        let pctx = kctx.new_at_location(pattern.as_str());
        let compiled_pattern = match analyze_pattern(pattern) {
            Some(PatternOptimization::Prefix(prefix)) => CompiledPattern::Prefix(Arc::from(prefix)),
            Some(PatternOptimization::Exact(exact)) => CompiledPattern::Exact(Arc::from(exact)),
            Some(PatternOptimization::Alternation(alts)) => CompiledPattern::Alternation(
                Arc::from(alts.into_boxed_slice()),
                Arc::from(pattern.as_str()),
            ),
            Some(PatternOptimization::NoWhitespace) => {
                CompiledPattern::NoWhitespace(Arc::from(pattern.as_str()))
            }
            None => {
                let regex = ctx.get_or_compile_standard_regex(pattern).map_err(|()| {
                    ValidationError::format(
                        kctx.location().clone(),
                        LazyEvaluationPath::SameAsSchemaPath,
                        Location::new(),
                        subschema,
                        "regex",
                    )
                })?;
                CompiledPattern::Regex((*regex).clone())
            }
        };
        let node = compiler::compile(&pctx, pctx.as_resource_ref(subschema))?;
        compiled_patterns.push((compiled_pattern, node));
    }
    Ok(compiled_patterns)
}

macro_rules! compile_dynamic_prop_map_validator {
    ($validator:tt, $properties:ident, $ctx:expr, $( $arg:expr ),* $(,)*) => {{
        if let Value::Object(map) = $properties {
            if map.len() < HASHMAP_THRESHOLD {
                Some($validator::<SmallValidatorsMap>::compile(
                    map, $ctx, $($arg, )*
                ))
            } else {
                Some($validator::<BigValidatorsMap>::compile(
                    map, $ctx, $($arg, )*
                ))
            }
        } else {
            let location = $ctx.location().clone();
            Some(Err(ValidationError::compile_error(
                location.clone(),
                location,
                Location::new(),
                $properties,
                "Unexpected type",
            )))
        }
    }};
}

pub(crate) use compile_dynamic_prop_map_validator;
