/*
  This file is part of TALER
  Copyright (C) 2014-2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler_mhd_lib.h
 * @brief API for generating MHD replies
 * @author Florian Dold
 * @author Benedikt Mueller
 * @author Christian Grothoff
 */
#ifndef TALER_MHD_LIB_H
#define TALER_MHD_LIB_H

#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_mhd_compat.h>
#include <taler/taler_error_codes.h>
#include <taler/taler_util.h>


/**
 * Maximum POST request size.
 */
#define TALER_MHD_REQUEST_BUFFER_MAX (1024 * 1024 * 16)


/**
 * Global options for response generation.
 */
enum TALER_MHD_GlobalOptions
{

  /**
   * Use defaults.
   */
  TALER_MHD_GO_NONE = 0,

  /**
   * Add "Connection: Close" header.
   */
  TALER_MHD_GO_FORCE_CONNECTION_CLOSE = 1,

  /**
   * Disable use of compression, even if the client
   * supports it.
   */
  TALER_MHD_GO_DISABLE_COMPRESSION = 2

};


#if MHD_VERSION < 0x00097701
#define MHD_create_response_from_buffer_static(s, b)            \
        MHD_create_response_from_buffer (s,                     \
                                         (const char *) b,      \
                                         MHD_RESPMEM_PERSISTENT)
#endif


/**
 * Find out if an MHD connection is using HTTPS (either
 * directly or via proxy).
 *
 * @param connection MHD connection
 * @returns #GNUNET_YES if the MHD connection is using https,
 *          #GNUNET_NO if the MHD connection is using http,
 *          #GNUNET_SYSERR if the connection type couldn't be determined
 */
enum GNUNET_GenericReturnValue
TALER_mhd_is_https (struct MHD_Connection *connection);


/**
 * Convert query argument to @a yna value.
 *
 * @param connection connection to take query argument from
 * @param arg argument to try for
 * @param default_val value to assign if the argument is not present
 * @param[out] yna value to set
 * @return true on success, false if the parameter was malformed
 */
bool
TALER_MHD_arg_to_yna (struct MHD_Connection *connection,
                      const char *arg,
                      enum TALER_EXCHANGE_YesNoAll default_val,
                      enum TALER_EXCHANGE_YesNoAll *yna);


/**
 * Set global options for response generation within libtalermhd.
 *
 * @param go global options to use
 */
void
TALER_MHD_setup (enum TALER_MHD_GlobalOptions go);


/**
 * Add headers we want to return in every response.  Useful for testing, like
 * if we want to always close connections.
 *
 * @param response response to modify
 * @param allow_store set to true to NOT add a "Cache-Control"
 *        directive that prevents caches and browsers from storing the data;
 *        if false, we set "Cache-Control: no-store" (privacy by default);
 *        set to true if the response contains no personal data
 */
void
TALER_MHD_add_global_headers (struct MHD_Response *response,
                              bool allow_store);


/**
 * Try to compress a response body.  Updates @a buf and @a buf_size.
 *
 * @param[in,out] buf pointer to body to compress
 * @param[in,out] buf_size pointer to initial size of @a buf
 * @return #MHD_YES if @a buf was compressed
 */
MHD_RESULT
TALER_MHD_body_compress (void **buf,
                         size_t *buf_size);


/**
 * List of compression types we check for. Larger numeric values
 * indicate a preferred algorithm.
 */
enum TALER_MHD_CompressionType
{
  /**
   * Compression is not supported.
   */
  TALER_MHD_CT_NONE = 0,

  /**
   * Deflate compression supported.
   */
  TALER_MHD_CT_DEFLATE,

  /**
   * gzip compression supported.
   */
  TALER_MHD_CT_GZIP,

  /**
   * zstd compression supported.
   */
  TALER_MHD_CT_ZSTD,

  /**
   * End of list marker.
   */
  TALER_MHD_CT_MAX
};


/**
 * What type of HTTP compression is supported by the client?
 *
 * @param connection connection to check
 * @param max maximum compression level to check for
 * @return #MHD_YES if 'deflate' compression is allowed
 */
enum TALER_MHD_CompressionType
TALER_MHD_can_compress (struct MHD_Connection *connection,
                        enum TALER_MHD_CompressionType max);


/**
 * Send JSON object as response.
 *
 * @param connection the MHD connection
 * @param json the json object
 * @param response_code the http response code
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_json (struct MHD_Connection *connection,
                      const json_t *json,
                      unsigned int response_code);


/**
 * Send JSON object as response, and free the @a json
 * object.
 *
 * @param connection the MHD connection
 * @param json the json object (freed!)
 * @param response_code the http response code
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_json_steal (struct MHD_Connection *connection,
                            json_t *json,
                            unsigned int response_code);


/**
 * Function to call to handle the request by building a JSON
 * reply from varargs.
 *
 * @param connection the MHD connection to handle
 * @param response_code HTTP response code to use
 * @param ... varargs of JSON pack specification
 * @return MHD result code
 */
#define TALER_MHD_REPLY_JSON_PACK(connection,response_code,...) \
        TALER_MHD_reply_json_steal (connection, GNUNET_JSON_PACK (__VA_ARGS__), \
                                    response_code)


/**
 * Send a response indicating an error.
 *
 * @param connection the MHD connection to use
 * @param ec error code uniquely identifying the error
 * @param http_status HTTP status code to use
 * @param detail additional optional detail about the error
 * @return a MHD result code
 */
MHD_RESULT
TALER_MHD_reply_with_error (struct MHD_Connection *connection,
                            unsigned int http_status,
                            enum TALER_ErrorCode ec,
                            const char *detail);


/**
 * Send a response indicating an error. The HTTP status code is
 * to be derived from the @a ec.
 *
 * @param connection the MHD connection to use
 * @param ec error code uniquely identifying the error
 * @param detail additional optional detail about the error
 * @return a MHD result code
 */
MHD_RESULT
TALER_MHD_reply_with_ec (struct MHD_Connection *connection,
                         enum TALER_ErrorCode ec,
                         const char *detail);


/**
 * Produce HTTP "Date:" header.
 *
 * @param at time to write to @a date
 * @param[out] date where to write the header, with
 *        at least 128 bytes available space.
 */
void
TALER_MHD_get_date_string (struct GNUNET_TIME_Absolute at,
                           char date[128]);


/**
 * Make JSON response object.
 *
 * @param json the json object
 * @return MHD response object
 */
struct MHD_Response *
TALER_MHD_make_json (const json_t *json);


/**
 * Make JSON response object and free @a json.
 *
 * @param json the json object, freed.
 * @return MHD response object
 */
struct MHD_Response *
TALER_MHD_make_json_steal (json_t *json);


/**
 * Make JSON response object.
 *
 * @param ... varargs
 * @return MHD response object
 */
#define TALER_MHD_MAKE_JSON_PACK(...) \
        TALER_MHD_make_json_steal (GNUNET_JSON_PACK (__VA_ARGS__))


/**
 * Pack Taler error code @a ec and associated hint into a
 * JSON object.
 *
 * @param ec error code to pack
 * @return packer array entries (two!)
 */
#define TALER_MHD_PACK_EC(ec) \
        GNUNET_JSON_pack_uint64 ("code", ec), \
        GNUNET_JSON_pack_string ("hint", TALER_ErrorCode_get_hint (ec))

/**
 * Create a response indicating an internal error.
 *
 * @param ec error code to return
 * @param detail additional optional detail about the error, can be NULL
 * @return a MHD response object
 */
struct MHD_Response *
TALER_MHD_make_error (enum TALER_ErrorCode ec,
                      const char *detail);


/**
 * Send a response indicating that the request was too big.
 *
 * @param connection the MHD connection to use
 * @return a MHD result code
 */
MHD_RESULT
TALER_MHD_reply_request_too_large (struct MHD_Connection *connection);


/**
 * Function to call to handle the request by sending
 * back a redirect to the AGPL source code.
 *
 * @param connection the MHD connection to handle
 * @param url where to redirect for the sources
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_agpl (struct MHD_Connection *connection,
                      const char *url);


/**
 * Function to call to handle the request by sending
 * back static data.
 *
 * @param connection the MHD connection to handle
 * @param http_status status code to return
 * @param mime_type content-type to use
 * @param body response payload
 * @param body_size number of bytes in @a body
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_static (struct MHD_Connection *connection,
                        unsigned int http_status,
                        const char *mime_type,
                        const char *body,
                        size_t body_size);


/**
 * Process a POST request containing a JSON object.  This
 * function realizes an MHD POST processor that will
 * (incrementally) process JSON data uploaded to the HTTP
 * server.  It will store the required state in the
 * "connection_cls", which must be cleaned up using
 * #TALER_MHD_parse_post_cleanup_callback().
 *
 * @param connection the MHD connection
 * @param con_cls the closure (points to a `struct Buffer *`)
 * @param upload_data the POST data
 * @param upload_data_size number of bytes in @a upload_data
 * @param json the JSON object for a completed request
 * @return
 *    #GNUNET_YES if json object was parsed or at least
 *               may be parsed in the future (call again);
 *               `*json` will be NULL if we need to be called again,
 *                and non-NULL if we are done.
 *    #GNUNET_NO is request incomplete or invalid
 *               (error message was generated)
 *    #GNUNET_SYSERR on internal error
 *               (we could not even queue an error message,
 *                close HTTP session with MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_post_json (struct MHD_Connection *connection,
                           void **con_cls,
                           const char *upload_data,
                           size_t *upload_data_size,
                           json_t **json);


/**
 * Function called whenever we are done with a request
 * to clean up our state.
 *
 * @param con_cls value as it was left by
 *        #TALER_MHD_parse_post_json(), to be cleaned up
 */
void
TALER_MHD_parse_post_cleanup_callback (void *con_cls);


/**
 * Parse JSON object into components based on the given field
 * specification.  If parsing fails, we return an HTTP
 * status code of 400 (#MHD_HTTP_BAD_REQUEST).
 *
 * @param connection the connection to send an error response to
 * @param root the JSON node to start the navigation at.
 * @param spec field specification for the parser
 * @return
 *    #GNUNET_YES if navigation was successful (caller is responsible
 *                for freeing allocated variable-size data using
 *                GNUNET_JSON_parse_free() when done)
 *    #GNUNET_NO if json is malformed, error response was generated
 *    #GNUNET_SYSERR on internal error
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_json_data (struct MHD_Connection *connection,
                           const json_t *root,
                           struct GNUNET_JSON_Specification *spec);


/**
 * Parse JSON object that we (the server!) generated into components based on
 * the given field specification.  The difference to
 * #TALER_MHD_parse_json_data() is that this function will fail
 * with an HTTP failure of 500 (internal server error) in case
 * parsing fails, instead of blaming it on the client with a
 * 400 (#MHD_HTTP_BAD_REQUEST).
 *
 * @param connection the connection to send an error response to
 * @param root the JSON node to start the navigation at.
 * @param spec field specification for the parser
 * @return
 *    #GNUNET_YES if navigation was successful (caller is responsible
 *                for freeing allocated variable-size data using
 *                GNUNET_JSON_parse_free() when done)
 *    #GNUNET_NO if json is malformed, error response was generated
 *    #GNUNET_SYSERR on internal error
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_internal_json_data (struct MHD_Connection *connection,
                                    const json_t *root,
                                    struct GNUNET_JSON_Specification *spec);


/**
 * Parse JSON array into components based on the given field
 * specification.  Generates error response on parse errors.
 *
 * @param connection the connection to send an error response to
 * @param root the JSON node to start the navigation at.
 * @param[in,out] spec field specification for the parser
 * @param ... -1-terminated list of array offsets of type 'int'
 * @return
 *    #GNUNET_YES if navigation was successful (caller is responsible
 *                for freeing allocated variable-size data using
 *                GNUNET_JSON_parse_free() when done)
 *    #GNUNET_NO if json is malformed, error response was generated
 *    #GNUNET_SYSERR on internal error
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_json_array (struct MHD_Connection *connection,
                            const json_t *root,
                            struct GNUNET_JSON_Specification *spec,
                            ...);


/**
 * Extract optional relative time argument from request.
 *
 * @param connection the MHD connection
 * @param label name of the argument to parse
 * @param[out] duration set to #GNUNET_TIME_UNIT_ZERO if there was no duration argument given
 * @return #GNUNET_OK on success, #GNUNET_NO if an
 *     error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_rel_time (struct MHD_Connection *connection,
                                      const char *label,
                                      struct GNUNET_TIME_Relative *duration);


/**
 * Extract optional relative time argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the @a label
 * argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param label label to check for
 * @param[out] duration set to #GNUNET_TIME_UNIT_ZERO if there was no duration given
 */
#define TALER_MHD_parse_request_rel_time(connection,label,duration)   \
        do {                                                          \
          switch (TALER_MHD_parse_request_arg_rel_time (connection,   \
                                                        label,        \
                                                        duration))  \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract optional "timeout_ms" argument from request.
 *
 * @param connection the MHD connection
 * @param[out] expiration set to #GNUNET_TIME_UNIT_ZERO_ABS if there was no timeout,
 *         the current time plus the value given under "timeout_ms" otherwise
 * @return #GNUNET_OK on success, #GNUNET_NO if an
 *     error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_timeout (struct MHD_Connection *connection,
                                     struct GNUNET_TIME_Absolute *expiration);


/**
 * Extract optional "timeout_ms" argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the "timeout_ms"
 * argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param[out] expiration set to #GNUNET_TIME_UNIT_ZERO_ABS if there was no timeout,
 *         the current time plus the value given under "timeout_ms" otherwise
 */
#define TALER_MHD_parse_request_timeout(connection,expiration) \
        do {                                                         \
          switch (TALER_MHD_parse_request_arg_timeout (connection,   \
                                                       expiration))  \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract optional timestamp argument from request.
 *
 * @param connection the MHD connection
 * @param fname name of the argument to parse
 * @param[out] ts set to #GNUNET_TIME_UNIT_ZERO_TS if there was no timestamp
 * @return #GNUNET_OK on success, #GNUNET_NO if an
 *     error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_timestamp (struct MHD_Connection *connection,
                                       const char *fname,
                                       struct GNUNET_TIME_Timestamp *ts);


/**
 * Extract optional timestamp argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the timestamp
 * argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param fname name of the argument
 * @param[out] ts set to #GNUNET_TIME_UNIT_ZERO_TS if there was no timestamp
 */
#define TALER_MHD_parse_request_timestamp(connection,fname,ts)  \
        do {                                                         \
          switch (TALER_MHD_parse_request_arg_timestamp (connection,   \
                                                         fname, \
                                                         ts))   \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract optional "yes/no/all" argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the
 * argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param name name of the query parameter to parse
 * @param def default value to set if absent
 * @param[out] ret set to the yes/no/all value
 */
#define TALER_MHD_parse_request_yna(connection,name,def,ret) \
        do {                                        \
          if (! (TALER_MHD_arg_to_yna (connection,      \
                                       name,            \
                                       def,             \
                                       ret)) )          \
          {                                         \
            GNUNET_break_op (0);                    \
            return TALER_MHD_reply_with_error (     \
              connection,                           \
              MHD_HTTP_BAD_REQUEST,                 \
              TALER_EC_GENERIC_PARAMETER_MALFORMED, \
              name);                                \
          }                                         \
        } while (0)

/**
 * Extract optional numeric limit argument from request.
 *
 * @param connection the MHD connection
 * @param name name of the query parameter
 * @param[out] off set to the offset, unchanged if the
 *             option was not given
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO if an error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_number (struct MHD_Connection *connection,
                                    const char *name,
                                    uint64_t *off);


/**
 * Extract optional numeric argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the
 * requested argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param name name of the argument to parse
 * @param[out] off set to the given numeric value,
 *    unchanged if value was not specified
 */
#define TALER_MHD_parse_request_number(connection,name,off)  \
        do {                                                         \
          switch (TALER_MHD_parse_request_arg_number (connection,   \
                                                      name, \
                                                      off))  \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract optional signed numeric limit argument from request.
 *
 * @param connection the MHD connection
 * @param name name of the query parameter
 * @param[out] val set to the signed value, unchanged if the
 *             option was not given
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO if an error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_snumber (struct MHD_Connection *connection,
                                     const char *name,
                                     int64_t *val);


/**
 * Extract optional numeric argument from request.
 * Macro that *returns* #MHD_YES/#MHD_NO if the
 * requested argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param name name of the argument to parse
 * @param[out] val set to the given numeric value,
 *    unchanged if value was not specified
 */
#define TALER_MHD_parse_request_snumber(connection,name,val)  \
        do {                                                         \
          switch (TALER_MHD_parse_request_arg_snumber (connection,   \
                                                       name, \
                                                       val))  \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract optional amount argument from request.
 *
 * @param connection the MHD connection
 * @param name name of the query parameter
 * @param[out] val set to the amount, unchanged if the
 *             option was not given
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO if an error was returned on @a connection (caller should return #MHD_YES) and
 *     #GNUNET_SYSERR if we failed to return an error (caller should return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_amount (struct MHD_Connection *connection,
                                    const char *name,
                                    struct TALER_Amount *val);


/**
 * Extract optional amount argument from request.  Macro that *returns*
 * #MHD_YES/#MHD_NO if the requested argument existed but failed to parse.
 *
 * @param connection the MHD connection
 * @param name name of the argument to parse
 * @param[out] val set to the given amount,
 *    unchanged if value was not specified
 */
#define TALER_MHD_parse_request_amount(connection,name,val)  \
        do {                                                         \
          switch (TALER_MHD_parse_request_arg_amount (connection,   \
                                                      name, \
                                                      val))  \
          {                      \
          case GNUNET_SYSERR:    \
            GNUNET_break (0);    \
            return MHD_NO;       \
          case GNUNET_NO:        \
            GNUNET_break_op (0); \
            return MHD_YES;      \
          case GNUNET_OK:        \
            break;               \
          }                      \
        } while (0)


/**
 * Extract fixed-size base32crockford encoded data from request argument.
 *
 * Queues an error response to the connection if the parameter is missing or
 * invalid.
 *
 * @param connection the MHD connection
 * @param param_name the name of the parameter with the key
 * @param[out] out_data pointer to store the result
 * @param out_size expected size of @a out_data
 * @param[out] present set to true if argument was found
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_arg_data (struct MHD_Connection *connection,
                                  const char *param_name,
                                  void *out_data,
                                  size_t out_size,
                                  bool *present);


/**
 * Extract fixed-size base32crockford encoded data from request header.
 *
 * Queues an error response to the connection if the parameter is missing or
 * invalid.
 *
 * @param connection the MHD connection
 * @param header_name the name of the HTTP header with the value
 * @param[out] out_data pointer to store the result
 * @param out_size expected size of @a out_data
 * @param[out] present set to true if argument was found
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_parse_request_header_data (struct MHD_Connection *connection,
                                     const char *header_name,
                                     void *out_data,
                                     size_t out_size,
                                     bool *present);

/**
 * Extract fixed-size base32crockford encoded data from request.
 *
 * @param connection the MHD connection
 * @param name the name of the parameter with the key
 * @param[out] val pointer to store the result, type must determine size
 * @param[in,out] required pass true to require presence of this argument; if 'false'
 *                         set to true if the argument was found
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is absent or malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
#define TALER_MHD_parse_request_arg_auto(connection,name,val,required) \
        do {                                                                 \
          bool p;                                                            \
          switch (TALER_MHD_parse_request_arg_data (connection, name,        \
                                                    val, sizeof (*val), &p)) \
          {                        \
          case GNUNET_SYSERR:      \
            GNUNET_break (0);      \
            return MHD_NO;         \
          case GNUNET_NO:          \
            GNUNET_break_op (0);   \
            return MHD_YES;        \
          case GNUNET_OK:          \
            if (required & (! p))  \
            {                      \
              GNUNET_break_op (0); \
              return TALER_MHD_reply_with_error (   \
                connection,                         \
                MHD_HTTP_BAD_REQUEST,               \
                TALER_EC_GENERIC_PARAMETER_MISSING, \
                name);                              \
            }                                       \
            required = p;                           \
            break;               \
          }                      \
        } while (0)


/**
 * Extract required fixed-size base32crockford encoded data from request.
 *
 * @param connection the MHD connection
 * @param name the name of the parameter with the key
 * @param[out] val pointer to store the result, type must determine size
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is absent or malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
#define TALER_MHD_parse_request_arg_auto_t(connection,name,val) \
        do {                                                          \
          bool b = true;                                              \
          TALER_MHD_parse_request_arg_auto (connection,name,val,b);   \
        } while (0)

/**
 * Extract fixed-size base32crockford encoded data from request.
 *
 * @param connection the MHD connection
 * @param name the name of the header with the key
 * @param[out] val pointer to store the result, type must determine size
 * @param[in,out] required pass true to require presence of this argument; if 'false'
 *                         set to true if the argument was found
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is absent or malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
#define TALER_MHD_parse_request_header_auto(connection,name,val,required)  \
        do {                                                               \
          bool p;                                                          \
          switch (TALER_MHD_parse_request_header_data (connection, name,   \
                                                       val, sizeof (*val), \
                                                       &p))                \
          {                       \
          case GNUNET_SYSERR:     \
            GNUNET_break (0);     \
            return MHD_NO;        \
          case GNUNET_NO:         \
            GNUNET_break_op (0);  \
            return MHD_YES;       \
          case GNUNET_OK:         \
            if (required & (! p)) \
            return TALER_MHD_reply_with_error (   \
              connection,                         \
              MHD_HTTP_BAD_REQUEST,               \
              TALER_EC_GENERIC_PARAMETER_MISSING, \
              name);                              \
            required = p;                         \
            break;               \
          }                      \
        } while (0)


/**
 * Extract required fixed-size base32crockford encoded data from request.
 *
 * @param connection the MHD connection
 * @param name the name of the header with the key
 * @param[out] val pointer to store the result, type must determine size
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is absent or malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
#define TALER_MHD_parse_request_header_auto_t(connection,name,val) \
        do {                                                             \
          bool b = true;                                                 \
          TALER_MHD_parse_request_header_auto (connection,name,val,b);   \
        } while (0)


/**
 * Check that the 'Content-Length' header is giving
 * a length below @a max_len. If not, return an
 * appropriate error response and return the
 * correct #MHD_YES/#MHD_NO value from this function.
 *
 * @param connection the MHD connection
 * @param max_len maximum allowed content length
 * @return
 *   #GNUNET_YES if the the argument is present
 *   #GNUNET_NO if the argument is absent or malformed
 *   #GNUNET_SYSERR on internal error (error response could not be sent)
 */
enum GNUNET_GenericReturnValue
TALER_MHD_check_content_length_ (struct MHD_Connection *connection,
                                 unsigned long long max_len);


/**
 * Check that the 'Content-Length' header is giving
 * a length below @a max_len. If not, return an
 * appropriate error response and return the
 * correct #MHD_YES/#MHD_NO value from this function.
 *
 * @param connection the MHD connection
 * @param max_len maximum allowed content length
 */
#define TALER_MHD_check_content_length(connection,max_len)         \
        do {                                                             \
          switch (TALER_MHD_check_content_length_ (connection, max_len)) \
          {                       \
          case GNUNET_SYSERR:     \
            GNUNET_break (0);     \
            return MHD_NO;        \
          case GNUNET_NO:         \
            GNUNET_break_op (0);  \
            return MHD_YES;       \
          case GNUNET_OK:         \
            break;                \
          }                       \
        } while (0)


/**
 * Function called for logging by MHD.
 *
 * @param cls closure, NULL
 * @param fm format string (`printf()`-style)
 * @param ap arguments to @a fm
 */
void
TALER_MHD_handle_logs (void *cls,
                       const char *fm,
                       va_list ap);


/**
 * Function called on each successfully bound listen
 * socket by #TALER_MHD_listen_bind().
 *
 * @param cls closure
 * @param fd bound listen socket (must be used and eventually
 *      closed by the callee). Never -1.
 */
typedef void
(*TALER_MHD_ListenSocketCallback)(void *cls,
                                  int fd);


/**
 * Bind a listen socket to the UNIX domain path,
 * or the TCP port(s) and IP address(es) configured,
 * or return to @a cb the inherited sockets from systemd,
 * all depending on what was specified in @a cfg in
 * the section named @a section.
 *
 * @param cfg configuration to parse
 * @param section configuration section to use
 * @param cb function to call with each bound socket
 * @param cb_cls closure for @a cb
 * @return #GNUNET_OK on success (all configured sockets were bound)
 *         #GNUNET_NO if some configured binding failed but the config is OK,
 *           note that some listen sockets may have been created
 *         #GNUNET_SYSERR if the configuration is invalid
 */
enum GNUNET_GenericReturnValue
TALER_MHD_listen_bind (const struct GNUNET_CONFIGURATION_Handle *cfg,
                       const char *section,
                       TALER_MHD_ListenSocketCallback cb,
                       void *cb_cls);


/**
 * Start to run an event loop for @a daemon.
 * Only one daemon can be running per process
 * using this API.
 *
 * @param daemon the MHD service to run
 */
void
TALER_MHD_daemon_start (struct MHD_Daemon *daemon);


/**
 * Stop running the event loops of all MHD daemons.
 */
void
TALER_MHD_daemons_halt (void);


/**
 * Stop accepting new connections on all MHD daemons
 * (and close the listen sockets).
 */
void
TALER_MHD_daemons_quiesce (void);


/**
 * Destroy all state associated with all MHD daemons.
 */
void
TALER_MHD_daemons_destroy (void);

/**
 * Trigger all MHD daemons that were running. Needed when
 * a connection was resumed.
 */
void
TALER_MHD_daemon_trigger (void);


/**
 * Prepared responses for legal documents
 * (terms of service, privacy policy).
 */
struct TALER_MHD_Legal;


/**
 * Load set of legal documents as specified in @a cfg in section @a section
 * where the Etag is given under the @a tagoption and the directory under
 * the @a diroption.
 *
 * @param cfg configuration to use
 * @param section section to load values from
 * @param diroption name of the option with the
 *        path to the legal documents
 * @param tagoption name of the files to use
 *        for the legal documents and the Etag
 * @return NULL on error
 */
struct TALER_MHD_Legal *
TALER_MHD_legal_load (const struct GNUNET_CONFIGURATION_Handle *cfg,
                      const char *section,
                      const char *diroption,
                      const char *tagoption);


/**
 * Free set of legal documents
 *
 * @param legal legal documents to free
 */
void
TALER_MHD_legal_free (struct TALER_MHD_Legal *legal);


/**
 * Generate a response with a legal document in
 * the format and language of the user's choosing.
 *
 * @param conn HTTP connection to handle
 * @param legal legal document to serve
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_legal (struct MHD_Connection *conn,
                       struct TALER_MHD_Legal *legal);


/**
 * Send back a "204 No Content" response with headers
 * for the CORS pre-flight request.
 *
 * @param connection the MHD connection
 * @return MHD result code
 */
MHD_RESULT
TALER_MHD_reply_cors_preflight (struct MHD_Connection *connection);


/**
 * Load SPA files from @a dir, relative to the project's
 * data directory.
 *
 *
 * @param pd project data to use to determine the parent directory
 * @param dir directory suffix to append to our data directory with the location of the files of the SPA
 * @return handle to serve static files from @a dir
 */
struct TALER_MHD_Spa *
TALER_MHD_spa_load (const struct GNUNET_OS_ProjectData *pd,
                    const char *dir);

/**
 * Load SPA files from absolute path to directory @a dn.
 *
 *
 * @param dn directory with the location of the files of the SPA,
 *        should be an absolute path.
 * @return handle to serve static files from @a dir
 */
struct TALER_MHD_Spa *
TALER_MHD_spa_load_dir (const char *dn);


/**
 * Release resources used by SPA handler.
 *
 * @param[in] spa data structure to release
 */
void
TALER_MHD_spa_free (struct TALER_MHD_Spa *spa);


/**
 * Handle HTTP request for files in a @a spa. Generates
 * a 404 if no file at @a path does exists.
 *
 * @param spa the SPA to serve files from
 * @param connection HTTP connection to return data on
 * @param path request path to match against the @a spa
 * @return MHD status code to give to MHD
 */
MHD_RESULT
TALER_MHD_spa_handler (const struct TALER_MHD_Spa *spa,
                       struct MHD_Connection *connection,
                       const char *path);


/**
 * Information about a document #TALER_MHD_typst() should output.
 */
struct TALER_MHD_TypstDocument
{
  /**
   * Form name, used to determine the Typst template to use.
   * NULL if @e data is a JSON string with a PDF to inline.
   */
  const char *form_name;

  /**
   * Form data.
   */
  const json_t *data;
};


/**
 * Context for generating PDF responses.
 */
struct TALER_MHD_TypstContext;


/**
 * Result from a #TALER_MHD_typst() operation.
 */
struct TALER_MHD_TypstResponse
{

  /**
   * Error status of the operation.
   */
  enum TALER_ErrorCode ec;

  /**
   * Details depending on @e ec.
   */
  union
  {
    /**
     * Hint if @e ec is not #TALER_EC_NONE.
     */
    const char *hint;

    /**
     * Filename with the result if @e ec is #TALER_EC_NONE.
     */
    const char *filename;
  } details;

};

/**
 * Function called with the result of a #TALER_MHD_typst() operation.
 *
 * @param cls closure
 * @param tr result of the operation
 */
typedef void
(*TALER_MHD_TypstResultCallback) (void *cls,
                                  const struct TALER_MHD_TypstResponse *tr);


/**
 * Generate PDFs using Typst and combine them using pdftk.  The
 * file will be returned to @a cb and after @a cb returns all data
 * will be deleted from the local disk.
 *
 * @param cfg configuration to use (where to find Typst templates)
 * @param remove_on_exit should the directory be removed when done?
 * @param cfg_section_name name of the configuration section to use
 * @param num_documents length of the @a docs array
 * @param docs list of documents to combine into one large PDF
 * @param cb function to call with the resulting file(name)
 * @param cb_cls closure for @a cb
 * @return NULL on error
 */
struct TALER_MHD_TypstContext *
TALER_MHD_typst (
  const struct GNUNET_CONFIGURATION_Handle *cfg,
  bool remove_on_exit,
  const char *cfg_section_name,
  unsigned int num_documents,
  const struct TALER_MHD_TypstDocument docs[static num_documents],
  TALER_MHD_TypstResultCallback cb,
  void *cb_cls);


/**
 * Abort all typst response generation processes.
 * To be used when the system is shutting down.
 */
void
TALER_MHD_typst_cancel (struct TALER_MHD_TypstContext *tc);


/**
 * Create HTTP response from the PDF file at @a filename
 *
 * @param filename file to return as PDF
 * @return NULL on error
 */
struct MHD_Response *
TALER_MHD_response_from_pdf_file (const char *filename);


#endif
