import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { renderWithProviders } from '../test-utils/render.js';
import { Text, useIsScreenReaderEnabled } from 'ink';
import { App } from './App.js';
import {} from './contexts/UIStateContext.js';
import { StreamingState, ToolCallStatus } from './types.js';
import { makeFakeConfig } from '@google/gemini-cli-core';
vi.mock('ink', async (importOriginal) => {
    const original = await importOriginal();
    return {
        ...original,
        useIsScreenReaderEnabled: vi.fn(),
    };
});
vi.mock('./components/DialogManager.js', () => ({
    DialogManager: () => _jsx(Text, { children: "DialogManager" }),
}));
vi.mock('./components/Composer.js', () => ({
    Composer: () => _jsx(Text, { children: "Composer" }),
}));
vi.mock('./components/Notifications.js', async () => {
    const { Text, Box } = await import('ink');
    return {
        Notifications: () => (_jsx(Box, { children: _jsx(Text, { children: "Notifications" }) })),
    };
});
vi.mock('./components/QuittingDisplay.js', () => ({
    QuittingDisplay: () => _jsx(Text, { children: "Quitting..." }),
}));
vi.mock('./components/HistoryItemDisplay.js', () => ({
    HistoryItemDisplay: () => _jsx(Text, { children: "HistoryItemDisplay" }),
}));
vi.mock('./components/Footer.js', async () => {
    const { Text, Box } = await import('ink');
    return {
        Footer: () => (_jsx(Box, { children: _jsx(Text, { children: "Footer" }) })),
    };
});
describe('App', () => {
    beforeEach(() => {
        useIsScreenReaderEnabled.mockReturnValue(false);
    });
    const mockUIState = {
        streamingState: StreamingState.Idle,
        quittingMessages: null,
        dialogsVisible: false,
        mainControlsRef: {
            current: null,
        },
        rootUiRef: {
            current: null,
        },
        historyManager: {
            addItem: vi.fn(),
            history: [],
            updateItem: vi.fn(),
            clearItems: vi.fn(),
            loadHistory: vi.fn(),
        },
        history: [],
        pendingHistoryItems: [],
        pendingGeminiHistoryItems: [],
        bannerData: {
            defaultText: 'Mock Banner Text',
            warningText: '',
        },
        backgroundShells: new Map(),
    };
    it('should render main content and composer when not quitting', () => {
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
            useAlternateBuffer: false,
        });
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Composer');
    });
    it('should render quitting display when quittingMessages is set', () => {
        const quittingUIState = {
            ...mockUIState,
            quittingMessages: [{ id: 1, type: 'user', text: 'test' }],
        };
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: quittingUIState,
            useAlternateBuffer: false,
        });
        expect(lastFrame()).toContain('Quitting...');
    });
    it('should render full history in alternate buffer mode when quittingMessages is set', () => {
        const quittingUIState = {
            ...mockUIState,
            quittingMessages: [{ id: 1, type: 'user', text: 'test' }],
            history: [{ id: 1, type: 'user', text: 'history item' }],
            pendingHistoryItems: [{ type: 'user', text: 'pending item' }],
        };
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: quittingUIState,
            useAlternateBuffer: true,
        });
        expect(lastFrame()).toContain('HistoryItemDisplay');
        expect(lastFrame()).toContain('Quitting...');
    });
    it('should render dialog manager when dialogs are visible', () => {
        const dialogUIState = {
            ...mockUIState,
            dialogsVisible: true,
        };
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: dialogUIState,
        });
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('DialogManager');
    });
    it.each([
        { key: 'C', stateKey: 'ctrlCPressedOnce' },
        { key: 'D', stateKey: 'ctrlDPressedOnce' },
    ])('should show Ctrl+$key exit prompt when dialogs are visible and $stateKey is true', ({ key, stateKey }) => {
        const uiState = {
            ...mockUIState,
            dialogsVisible: true,
            [stateKey]: true,
        };
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState,
        });
        expect(lastFrame()).toContain(`Press Ctrl+${key} again to exit.`);
    });
    it('should render ScreenReaderAppLayout when screen reader is enabled', () => {
        useIsScreenReaderEnabled.mockReturnValue(true);
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
        });
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Footer');
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Composer');
    });
    it('should render DefaultAppLayout when screen reader is not enabled', () => {
        useIsScreenReaderEnabled.mockReturnValue(false);
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: mockUIState,
        });
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Composer');
    });
    it('should render ToolConfirmationQueue along with Composer when tool is confirming and experiment is on', () => {
        useIsScreenReaderEnabled.mockReturnValue(false);
        const toolCalls = [
            {
                callId: 'call-1',
                name: 'ls',
                description: 'list directory',
                status: ToolCallStatus.Confirming,
                resultDisplay: '',
                confirmationDetails: {
                    type: 'exec',
                    title: 'Confirm execution',
                    command: 'ls',
                    rootCommand: 'ls',
                    rootCommands: ['ls'],
                },
            },
        ];
        const stateWithConfirmingTool = {
            ...mockUIState,
            pendingHistoryItems: [{ type: 'tool_group', tools: toolCalls }],
            pendingGeminiHistoryItems: [{ type: 'tool_group', tools: toolCalls }],
        };
        const configWithExperiment = makeFakeConfig();
        vi.spyOn(configWithExperiment, 'isEventDrivenSchedulerEnabled').mockReturnValue(true);
        vi.spyOn(configWithExperiment, 'isTrustedFolder').mockReturnValue(true);
        vi.spyOn(configWithExperiment, 'getIdeMode').mockReturnValue(false);
        const { lastFrame } = renderWithProviders(_jsx(App, {}), {
            uiState: stateWithConfirmingTool,
            config: configWithExperiment,
        });
        expect(lastFrame()).toContain('Tips for getting started');
        expect(lastFrame()).toContain('Notifications');
        expect(lastFrame()).toContain('Action Required'); // From ToolConfirmationQueue
        expect(lastFrame()).toContain('Composer');
        expect(lastFrame()).toMatchSnapshot();
    });
    describe('Snapshots', () => {
        it('renders default layout correctly', () => {
            useIsScreenReaderEnabled.mockReturnValue(false);
            const { lastFrame } = renderWithProviders(_jsx(App, {}), {
                uiState: mockUIState,
            });
            expect(lastFrame()).toMatchSnapshot();
        });
        it('renders screen reader layout correctly', () => {
            useIsScreenReaderEnabled.mockReturnValue(true);
            const { lastFrame } = renderWithProviders(_jsx(App, {}), {
                uiState: mockUIState,
            });
            expect(lastFrame()).toMatchSnapshot();
        });
        it('renders with dialogs visible', () => {
            const dialogUIState = {
                ...mockUIState,
                dialogsVisible: true,
            };
            const { lastFrame } = renderWithProviders(_jsx(App, {}), {
                uiState: dialogUIState,
            });
            expect(lastFrame()).toMatchSnapshot();
        });
    });
});
//# sourceMappingURL=App.test.js.map