/*
 * Decompiled with CFR 0.152.
 */
package ghidra.formats.gfilesystem;

import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.FileByteProvider;
import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FSUtilities;
import ghidra.formats.gfilesystem.FileSystemRefManager;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileHashProvider;
import ghidra.formats.gfilesystem.GFileImpl;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.formats.gfilesystem.factory.GFileSystemFactoryIgnore;
import ghidra.formats.gfilesystem.fileinfo.FileAttribute;
import ghidra.formats.gfilesystem.fileinfo.FileAttributeType;
import ghidra.formats.gfilesystem.fileinfo.FileAttributes;
import ghidra.formats.gfilesystem.fileinfo.FileType;
import ghidra.framework.OperatingSystem;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import org.apache.commons.collections4.map.ReferenceMap;
import org.apache.commons.io.FilenameUtils;

@FileSystemInfo(type="file", description="Local filesystem", factory=GFileSystemFactoryIgnore.class)
public class LocalFileSystem
implements GFileSystem,
GFileHashProvider {
    public static final String FSTYPE = "file";
    private final FSRLRoot fsFSRL;
    private final GFile rootDir;
    private final FileSystemRefManager refManager = new FileSystemRefManager(this);
    private final ReferenceMap<FileFingerprintRec, String> fileFingerprintToMD5Map = new ReferenceMap();
    private final boolean isWindows = OperatingSystem.CURRENT_OPERATING_SYSTEM == OperatingSystem.WINDOWS;

    public static LocalFileSystem makeGlobalRootFS() {
        return new LocalFileSystem(FSRLRoot.makeRoot(FSTYPE));
    }

    private LocalFileSystem(FSRLRoot fsrl) {
        this.fsFSRL = fsrl;
        this.rootDir = GFileImpl.fromFSRL(this, null, this.fsFSRL.withPath("/"), true, -1L);
    }

    boolean isSameFS(FSRL fsrl) {
        return this.fsFSRL.equals(fsrl.getFS());
    }

    public boolean isLocalSubdir(FSRL fsrl) {
        if (!this.isSameFS(fsrl)) {
            return false;
        }
        File localFile = new File(fsrl.getPath());
        return localFile.isDirectory();
    }

    public File getLocalFile(FSRL fsrl) throws IOException {
        if (!this.isSameFS(fsrl)) {
            throw new IOException("FSRL does not specify local file: " + String.valueOf(fsrl));
        }
        File localFile = new File(fsrl.getPath());
        return localFile;
    }

    public FSRL getLocalFSRL(File f) {
        String absPath = f.getAbsolutePath();
        if (this.isWindows) {
            absPath = FilenameUtils.separatorsToUnix((String)absPath);
        }
        String fsrlPath = FSUtilities.appendPath("/", absPath);
        return this.fsFSRL.withPath(fsrlPath);
    }

    public GFile getGFile(File f) {
        List<File> parts = LocalFileSystem.getFilePathParts(f);
        GFile current = this.rootDir;
        for (int i = parts.size() - (this.isWindows ? 1 : 2); i >= 0; --i) {
            File part = parts.get(i);
            FSRL childFSRL = this.getLocalFSRL(part);
            current = GFileImpl.fromFSRL(this, current, childFSRL, part.isDirectory(), part.length());
        }
        return current;
    }

    @Override
    public String getName() {
        return "Root Filesystem";
    }

    @Override
    public void close() {
    }

    @Override
    public boolean isStatic() {
        return false;
    }

    @Override
    public List<GFile> getListing(GFile directory) {
        ArrayList<GFile> results = new ArrayList<GFile>();
        if ((directory = Objects.requireNonNullElse(directory, this.rootDir)).equals(this.rootDir) && this.isWindows) {
            for (File f : File.listRoots()) {
                results.add(GFileImpl.fromFSRL(this, null, this.getLocalFSRL(f), f.isDirectory(), -1L));
            }
        } else {
            File localDir = new File(directory.getPath());
            if (!localDir.isDirectory() || FSUtilities.isSymlink(localDir)) {
                return List.of();
            }
            File[] files = localDir.listFiles();
            if (files == null) {
                return List.of();
            }
            for (File f : files) {
                if (!f.isFile() && !f.isDirectory() && !FSUtilities.isSymlink(f)) continue;
                FSRL newFileFSRL = directory.getFSRL().appendPath(f.getName());
                results.add(GFileImpl.fromFSRL(this, directory, newFileFSRL, f.isDirectory(), f.length()));
            }
        }
        return results;
    }

    @Override
    public FileAttributes getFileAttributes(GFile file, TaskMonitor monitor) {
        File f = new File(file.getPath());
        return this.getFileAttributes(f);
    }

    public FileAttributes getFileAttributes(File f) {
        FileType fileType = FSUtilities.getFileType(f);
        String symLinkDest = fileType == FileType.SYMBOLIC_LINK ? FSUtilities.readSymlink(f) : null;
        return FileAttributes.of(FileAttribute.create(FileAttributeType.NAME_ATTR, f.getName()), FileAttribute.create(FileAttributeType.FILE_TYPE_ATTR, fileType), FileAttribute.create(FileAttributeType.SIZE_ATTR, Long.valueOf(f.length())), FileAttribute.create(FileAttributeType.MODIFIED_DATE_ATTR, new Date(f.lastModified())), symLinkDest != null ? FileAttribute.create(FileAttributeType.SYMLINK_DEST_ATTR, symLinkDest) : null);
    }

    @Override
    public FSRLRoot getFSRL() {
        return this.fsFSRL;
    }

    @Override
    public GFile getRootDir() {
        return this.rootDir;
    }

    @Override
    public GFile lookup(String path) throws IOException {
        return this.lookup(path, null);
    }

    @Override
    public GFile lookup(String path, Comparator<String> nameComp) throws IOException {
        if (path == null || path.equals(this.rootDir.getPath())) {
            return this.rootDir;
        }
        File f = LocalFileSystem.lookupFile(null, path, nameComp);
        return f != null ? this.getGFile(f) : null;
    }

    @Override
    public boolean isClosed() {
        return false;
    }

    @Override
    public FileSystemRefManager getRefManager() {
        return this.refManager;
    }

    @Override
    public InputStream getInputStream(GFile file, TaskMonitor monitor) throws IOException {
        return this.getInputStream(file.getFSRL(), monitor);
    }

    InputStream getInputStream(FSRL fsrl, TaskMonitor monitor) throws IOException {
        return new FileInputStream(this.getLocalFile(fsrl));
    }

    @Override
    public ByteProvider getByteProvider(GFile file, TaskMonitor monitor) throws IOException {
        return this.getByteProvider(file.getFSRL(), monitor);
    }

    ByteProvider getByteProvider(FSRL fsrl, TaskMonitor monitor) throws IOException {
        File f = this.getLocalFile(fsrl);
        return new FileByteProvider(f, fsrl, AccessMode.READ);
    }

    @Override
    public GFile resolveSymlinks(GFile file) throws IOException {
        File canonicalFile;
        File f = this.getLocalFile(file.getFSRL());
        if (f.equals(canonicalFile = f.getCanonicalFile())) {
            return file;
        }
        return this.getGFile(canonicalFile);
    }

    public String toString() {
        return "Local file system " + String.valueOf(this.fsFSRL);
    }

    @Override
    public String getMD5Hash(GFile file, boolean required, TaskMonitor monitor) throws CancelledException, IOException {
        return this.getMD5Hash(file.getFSRL(), required, monitor);
    }

    synchronized String getMD5Hash(FSRL fsrl, boolean required, TaskMonitor monitor) throws CancelledException, IOException {
        File f = this.getLocalFile(fsrl);
        if (!f.isFile()) {
            return null;
        }
        FileFingerprintRec fileFingerprintRec = new FileFingerprintRec(f.getPath(), f.lastModified(), f.length());
        String md5 = (String)this.fileFingerprintToMD5Map.get((Object)fileFingerprintRec);
        if (md5 == null && required) {
            md5 = FSUtilities.getFileMD5(f, monitor);
            this.fileFingerprintToMD5Map.put((Object)fileFingerprintRec, (Object)md5);
        }
        return md5;
    }

    public static File lookupFile(File baseDir, String path, Comparator<String> nameComp) {
        File f = new File(baseDir, path = Objects.requireNonNullElse(path, "/"));
        if (!f.isAbsolute()) {
            Msg.debug(LocalFileSystem.class, (Object)("Non-absolute path encountered in LocalFileSystem lookup: " + path));
            f = f.getAbsoluteFile();
        }
        try {
            if (nameComp == null || f.getParentFile() == null) {
                return FSUtilities.isSymlink(f = LocalFileSystem.updateCaseInsensitiveFilePath(f)) || f.exists() ? f : null;
            }
            List<File> pathParts = LocalFileSystem.getFilePathParts(f);
            for (int i = pathParts.size() - 2; i >= 0; --i) {
                File part;
                File parentDir = pathParts.get(i + 1);
                File foundFile = LocalFileSystem.findInDir(parentDir, (part = pathParts.get(i)).getName(), nameComp);
                if (foundFile == null) {
                    return null;
                }
                pathParts.set(i, foundFile);
            }
            return pathParts.get(0);
        }
        catch (IOException e) {
            Msg.warn(LocalFileSystem.class, (Object)("Error resolving path: " + path), (Throwable)e);
            return null;
        }
    }

    static File updateCaseInsensitiveFilePath(File f) throws IOException {
        return OperatingSystem.CURRENT_OPERATING_SYSTEM == OperatingSystem.WINDOWS ? f.getCanonicalFile() : f;
    }

    static File findInDir(File dir, String name, Comparator<String> nameComp) throws IOException {
        File exact = new File(dir, name);
        if (exact.exists() && nameComp.compare((exact = LocalFileSystem.updateCaseInsensitiveFilePath(exact)).getName(), name) == 0) {
            return exact;
        }
        File[] files = dir.listFiles();
        ArrayList<File> candidateMatches = new ArrayList<File>();
        if (files != null) {
            for (File f : files) {
                String foundFilename = f.getName();
                if (nameComp.compare(name, foundFilename) != 0) continue;
                if (name.equals(foundFilename)) {
                    return f;
                }
                candidateMatches.add(f);
            }
        }
        Collections.sort(candidateMatches);
        return !candidateMatches.isEmpty() ? (File)candidateMatches.get(0) : null;
    }

    static List<File> getFilePathParts(File f) {
        ArrayList<File> results = new ArrayList<File>();
        while (f != null) {
            results.add(f);
            f = f.getParentFile();
        }
        return results;
    }

    private record FileFingerprintRec(String path, long timestamp, long length) {
    }
}

