/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.cart;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.ByteProviderWrapper;
import ghidra.file.formats.cart.CartCancelDialogs;
import ghidra.file.formats.cart.CartConfigurationException;
import ghidra.file.formats.cart.CartFileSystemFactory;
import ghidra.file.formats.cart.CartInvalidARC4KeyException;
import ghidra.file.formats.cart.CartInvalidCartException;
import ghidra.file.formats.cart.CartV1Constants;
import ghidra.file.formats.cart.CartV1File;
import ghidra.file.formats.cart.CartV1PayloadExtractor;
import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FileSystemRefManager;
import ghidra.formats.gfilesystem.FileSystemService;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.formats.gfilesystem.SingleFileSystemIndexHelper;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.formats.gfilesystem.crypto.CryptoSession;
import ghidra.formats.gfilesystem.fileinfo.FileAttributeType;
import ghidra.formats.gfilesystem.fileinfo.FileAttributes;
import ghidra.framework.generic.auth.Password;
import ghidra.util.HashUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.text.StringEscapeUtils;

@FileSystemInfo(type="cart", description="Compressed and ARC4 Transport (CaRT) neutering format.", factory=CartFileSystemFactory.class)
public class CartFileSystem
implements GFileSystem {
    private final FSRLRoot fsFSRL;
    private final FileSystemRefManager refManager = new FileSystemRefManager((GFileSystem)this);
    private final FileSystemService fsService;
    private ByteProvider byteProvider;
    private ByteProvider payloadProvider;
    private SingleFileSystemIndexHelper fsIndexHelper;
    private CartV1File cartFile;

    public CartFileSystem(FSRLRoot fsFSRL, FileSystemService fsService) {
        this.fsFSRL = fsFSRL;
        this.fsService = fsService;
    }

    public void mount(ByteProvider bProvider, TaskMonitor monitor) throws CancelledException, IOException {
        this.byteProvider = bProvider;
        try {
            try {
                this.cartFile = new CartV1File(this.byteProvider);
            }
            catch (CartInvalidARC4KeyException e) {
                try (CryptoSession cryptoSession = this.fsService.newCryptoSession();){
                    String prompt = this.fsFSRL.getContainer().getName() + " (plaintext or base64)";
                    Iterator pwIt = cryptoSession.getPasswordsFor(this.fsFSRL.getContainer(), prompt);
                    while (pwIt.hasNext()) {
                        try (Password passwordValue = (Password)pwIt.next();){
                            monitor.setMessage("Testing key...");
                            String password = String.valueOf(passwordValue.getPasswordChars());
                            this.cartFile = new CartV1File(this.byteProvider, password);
                        }
                        catch (CartInvalidARC4KeyException arc4E) {
                            if (CartCancelDialogs.promptErrorContinue("Bad Key", "Error when testing key for " + this.fsFSRL.getContainer().getName() + ":\n" + (arc4E.getMessage() != null ? arc4E.getMessage() : "Unknown") + "\n Try another key?")) continue;
                        }
                        break;
                    }
                }
            }
        }
        catch (CartInvalidARC4KeyException e) {
            throw new IOException("Invalid CaRT ARC4 Key: " + e.getMessage());
        }
        catch (CartInvalidCartException e) {
            throw new IOException("Invalid CaRT file: " + e.getMessage());
        }
        catch (CartConfigurationException e) {
            throw new IOException("Invalid CaRT configuration file: " + e.getMessage());
        }
        if (this.cartFile == null) {
            throw new IOException("ARC4 key not found or user cancelled.");
        }
        this.payloadProvider = this.getPayload(null, monitor);
        this.fsIndexHelper = new SingleFileSystemIndexHelper((GFileSystem)this, this.fsFSRL, this.cartFile.getPath(), this.cartFile.getDataSize(), null);
    }

    public void close() throws IOException {
        this.refManager.onClose();
        if (this.fsIndexHelper != null) {
            this.fsIndexHelper.clear();
        }
        if (this.byteProvider != null) {
            this.byteProvider.close();
            this.byteProvider = null;
        }
        if (this.payloadProvider != null) {
            this.payloadProvider.close();
            this.payloadProvider = null;
        }
    }

    public boolean isClosed() {
        return this.fsIndexHelper == null || this.fsIndexHelper.isClosed();
    }

    public String getName() {
        return this.fsFSRL.getContainer().getName();
    }

    public FSRLRoot getFSRL() {
        return this.fsFSRL;
    }

    public FileSystemRefManager getRefManager() {
        return this.refManager;
    }

    public GFile lookup(String path) throws IOException {
        return this.fsIndexHelper.lookup(path);
    }

    public GFile lookup(String path, Comparator<String> nameComp) throws IOException {
        return this.fsIndexHelper.lookup(null, path, nameComp);
    }

    private ByteProvider getPayload(FSRL payloadFSRL, TaskMonitor monitor) throws CancelledException, IOException {
        return this.fsService.getDerivedByteProviderPush(this.byteProvider.getFSRL(), payloadFSRL, "cart", -1L, os -> {
            CartV1PayloadExtractor extractor = new CartV1PayloadExtractor(this.byteProvider, os, this.cartFile);
            extractor.extract(monitor);
        }, monitor);
    }

    public ByteProvider getByteProvider(GFile file, TaskMonitor monitor) throws IOException, CancelledException {
        if (this.fsIndexHelper.isPayloadFile(file)) {
            return new ByteProviderWrapper(this.payloadProvider, file.getFSRL());
        }
        return null;
    }

    public List<GFile> getListing(GFile directory) throws IOException {
        return this.fsIndexHelper.getListing(directory);
    }

    public FileAttributes getFileAttributes(GFile file, TaskMonitor monitor) {
        FileAttributes result = new FileAttributes();
        if (!this.fsIndexHelper.isPayloadFile(file) || this.cartFile == null) {
            return result;
        }
        HashSet<String> skipKeys = new HashSet<String>(Set.of("name"));
        HashSet<String> addedAttributes = new HashSet<String>();
        if (this.cartFile.getDataSize() >= 0L) {
            result.add(FileAttributeType.SIZE_ATTR, (Object)this.cartFile.getDataSize());
        }
        skipKeys.add("length");
        result.add(FileAttributeType.COMPRESSED_SIZE_ATTR, (Object)this.cartFile.getPackedSize());
        result.add(FileAttributeType.IS_ENCRYPTED_ATTR, (Object)true);
        result.add(FileAttributeType.HAS_GOOD_PASSWORD_ATTR, (Object)true);
        result.add("WARNING", (Object)"CaRT format is often used to neuter and share malicious files.\nPlease use caution if exporting original binary.");
        result.add("ARC4 Key", (Object)new String(HashUtilities.hexDump((byte[])this.cartFile.getDecryptor().getARC4Key())));
        for (String string : CartV1Constants.EXPECTED_HASHES.keySet()) {
            byte[] footerHashValue = this.cartFile.getFooterHash(string);
            skipKeys.add(string.toLowerCase());
            if (footerHashValue != null) {
                result.add("Reported " + string, (Object)new String(HashUtilities.hexDump((byte[])footerHashValue)));
                continue;
            }
            result.add("Reported " + string, (Object)"Missing");
        }
        HashSet<String> protectedKeys = new HashSet<String>();
        for (Object attribute : result.getAttributes()) {
            protectedKeys.add(attribute.getAttributeDisplayName().toLowerCase());
            addedAttributes.add(attribute.getAttributeDisplayName());
        }
        HashSet<String> hashSet = new HashSet<String>();
        for (Map.Entry entry : this.cartFile.getHeader().optionalHeaderData().entrySet()) {
            if (!CartV1Constants.FOOTER_ONLY_KEYS.contains(((String)entry.getKey()).toLowerCase())) continue;
            hashSet.add((String)entry.getKey());
        }
        if (!hashSet.isEmpty()) {
            result.add("SECURITY WARNING", (Object)("CaRT file metadata may be attempting to overwrite protected file data: " + StringEscapeUtils.escapeHtml4((String)String.join((CharSequence)", ", hashSet)) + "."));
        }
        Gson gson = new GsonBuilder().serializeNulls().setPrettyPrinting().create();
        hashSet.clear();
        for (Map.Entry entry : this.cartFile.getMetadata().entrySet()) {
            if (skipKeys.contains(((String)entry.getKey()).toLowerCase())) continue;
            if (protectedKeys.contains(((String)entry.getKey()).toLowerCase())) {
                hashSet.add((String)entry.getKey());
                continue;
            }
            String value = "<Unknown>";
            try {
                value = gson.toJson((JsonElement)entry.getValue());
            }
            catch (IllegalStateException e) {
                value = "Invalid JSON String";
            }
            Object key = (String)entry.getKey();
            int suffix_counter = 0;
            while (addedAttributes.contains(key)) {
                if (++suffix_counter > 100) {
                    suffix_counter = -1;
                    break;
                }
                key = (String)entry.getKey() + "_" + suffix_counter;
            }
            if (suffix_counter == -1) continue;
            result.add((String)key, (Object)value);
            addedAttributes.add((String)key);
        }
        if (!hashSet.isEmpty()) {
            result.add("SECURITY WARNING", (Object)("CaRT file metadata may be attempting to overwrite protected file data: " + StringEscapeUtils.escapeHtml4((String)String.join((CharSequence)", ", hashSet)) + ". Skipped those keys."));
        }
        return result;
    }
}

