# -*- coding: utf-8 -*-
#
# Author: Taylor Smith <taylor.smith@alkaline-ml.com>
#
# This is the taylor dataset found in R.

import numpy as np
import pandas as pd

from ..compat import DTYPE


def load_taylor(as_series=False, dtype=DTYPE):
    """Half-hourly electricity demand

    Half-hourly electricity demand in England and Wales from Monday, 5 June,
    2000 to Sunday, 27 August, 2000. Discussed in Taylor (2003), and kindly
    provided by James W Taylor. Units: Megawatts

    Parameters
    ----------
    as_series : bool, optional (default=False)
        Whether to return a Pandas series. If True, the index will be set to
        the observed years. If False, will return a 1d numpy array.

    dtype : type, optional (default=np.float64)
        The type to return for the array. Default is np.float64, which is used
        throughout the package as the default type.

    Examples
    --------
    >>> from pmdarima.datasets import load_taylor
    >>> load_taylor()[:10]
    array([22262., 21756., 22247., 22759., 22549., 22313., 22128., 21860.,
           21751., 21336.])

    >>> load_taylor(True).head()
    0    22262.0
    1    21756.0
    2    22247.0
    3    22759.0
    4    22549.0
    dtype: float64

    Notes
    -----
    This is annual data and not seasonal in nature (i.e., :math:`m=1`)

    References
    ----------
    .. [1] Taylor, J.W. (2003) Short-term electricity demand forecasting using
           double seasonal exponential smoothing. Journal of the Operational
           Research Society, 54, 799-805.

    .. [2] https://www.rdocumentation.org/packages/forecast/versions/8.9/topics/taylor

    Returns
    -------
    taylor : array-like, shape=(n_samples,)
        The Taylor dataset. There are 4032 observations.
    """  # noqa: E501
    rslt = np.array([
        22262, 21756, 22247, 22759, 22549, 22313,
        22128, 21860, 21751, 21336, 21363, 22176,
        24649, 27466, 31020, 33529, 35428, 36034,
        36834, 37296, 37338, 37608, 37692, 37944,
        37880, 37493, 37065, 36855, 36871, 36837,
        36603, 36982, 37369, 37650, 37413, 36475,
        35486, 34452, 33704, 32883, 31940, 32215,
        32572, 32850, 32251, 30895, 28460, 26572,
        25093, 24320, 24684, 25338, 24943, 24637,
        24424, 24163, 23943, 23324, 23412, 24065,
        26035, 28632, 32023, 34315, 35927, 36367,
        37087, 37184, 37009, 37212, 37531, 37890,
        37982, 37540, 37160, 36920, 37060, 36930,
        36640, 36910, 37220, 37530, 37320, 36331,
        35206, 34203, 33273, 32376, 31838, 31211,
        31505, 31838, 32185, 30881, 28800, 26833,
        25095, 24437, 24697, 25259, 25029, 24806,
        24437, 24279, 23902, 23418, 23424, 24094,
        26039, 28636, 32034, 34235, 35694, 35978,
        36475, 36675, 36720, 36842, 36890, 36917,
        36854, 36525, 36431, 36212, 36061, 35931,
        35835, 36148, 36523, 36859, 36731, 35804,
        34721, 34035, 33234, 32532, 31474, 31957,
        32500, 32586, 31917, 30636, 28705, 26707,
        25092, 24412, 24714, 25079, 24904, 24380,
        24009, 23913, 23657, 23262, 23283, 23870,
        26001, 28448, 31876, 34363, 35819, 36295,
        36949, 37095, 37086, 37520, 37540, 37671,
        37695, 37314, 37113, 36796, 36705, 36634,
        36357, 36664, 37020, 37152, 36836, 36008,
        34981, 34265, 33385, 32477, 31966, 31402,
        31703, 32192, 32322, 30972, 28910, 26992,
        25324, 24684, 24893, 25092, 24751, 24238,
        23906, 23755, 23625, 23027, 23097, 23692,
        25936, 28501, 31838, 34308, 36305, 36743,
        37705, 37980, 38084, 38379, 38526, 38429,
        38344, 37785, 37495, 37190, 37010, 36641,
        36296, 36309, 36532, 36662, 36229, 35321,
        34409, 33425, 32708, 31460, 30955, 30274,
        30302, 30427, 30154, 29311, 27699, 26213,
        24756, 23847, 23917, 23989, 23555, 23026,
        22668, 22451, 21969, 21212, 20978, 21155,
        22370, 23537, 25210, 26961, 28770, 29915,
        30895, 31273, 31390, 31287, 31163, 30811,
        30310, 29767, 29016, 28374, 28064, 27599,
        27338, 27156, 27355, 27860, 28315, 28396,
        27932, 27776, 27520, 27026, 26649, 26343,
        26487, 27219, 27384, 26680, 25658, 24258,
        22864, 21998, 22165, 22308, 21863, 21439,
        21067, 20781, 20389, 19663, 19305, 19410,
        20079, 20701, 21639, 22722, 24046, 25150,
        26285, 27420, 28438, 28770, 29309, 29586,
        29721, 29598, 28776, 28008, 27487, 27272,
        27091, 27156, 27190, 27461, 27722, 27583,
        27575, 27512, 27603, 27155, 27250, 27200,
        27798, 28487, 28691, 27260, 25371, 23689,
        22454, 21818, 22002, 22516, 22292, 21935,
        21683, 21656, 21481, 21140, 21308, 22138,
        24699, 27366, 30973, 33497, 35192, 35624,
        36444, 36903, 36944, 37223, 37345, 37710,
        37641, 37391, 37116, 37017, 37024, 36882,
        36605, 36861, 37241, 37550, 37215, 36203,
        35035, 34112, 33482, 32349, 31186, 30920,
        30270, 31869, 31651, 30821, 28528, 26675,
        25129, 24425, 24538, 24577, 24215, 24012,
        23723, 23546, 23379, 22857, 22939, 23464,
        25734, 28240, 31600, 34011, 35567, 35946,
        36733, 36963, 37009, 37100, 37265, 37478,
        37462, 37040, 36778, 36679, 36570, 36509,
        36433, 36730, 37125, 37359, 36995, 36149,
        34845, 33986, 33353, 32595, 31833, 31382,
        31264, 31805, 32067, 30723, 28736, 26800,
        25296, 24612, 24614, 24699, 24358, 23930,
        23690, 23720, 23472, 23233, 23376, 23913,
        26027, 28641, 32093, 34598, 36118, 36505,
        37295, 37562, 37546, 37734, 38071, 38204,
        38233, 37710, 37459, 37224, 37208, 36962,
        36837, 37151, 37602, 38055, 37780, 36832,
        35861, 35202, 34347, 33562, 32860, 32606,
        32578, 32552, 32256, 30893, 28956, 26926,
        25339, 24677, 24774, 24932, 24615, 24244,
        23962, 23970, 23624, 23113, 23221, 23940,
        26205, 28582, 32017, 34569, 36166, 36510,
        37250, 37372, 37318, 37333, 37611, 37736,
        37756, 37275, 37010, 36926, 36837, 36726,
        36449, 36734, 37067, 37174, 36824, 35980,
        34953, 34235, 33350, 32664, 31923, 31603,
        31736, 32134, 31993, 30956, 28861, 27012,
        25267, 24560, 24415, 24490, 24057, 23680,
        23440, 23432, 23184, 22774, 22930, 23579,
        25821, 28193, 31647, 34117, 35590, 36167,
        36741, 36768, 36783, 36991, 37115, 37224,
        37439, 36822, 36375, 36126, 35805, 35478,
        35161, 35123, 35226, 35435, 35062, 34357,
        33239, 32583, 31586, 30604, 30091, 29610,
        29462, 29893, 30100, 29071, 27660, 26242,
        24618, 23740, 23600, 23350, 22870, 22390,
        22082, 21823, 21548, 20867, 20671, 20948,
        22106, 23295, 25222, 27075, 28876, 30152,
        31325, 31720, 31770, 31694, 31507, 31267,
        30679, 30227, 29601, 29002, 28670, 28308,
        28101, 28074, 28364, 28772, 29191, 29118,
        28661, 28005, 27536, 26823, 25958, 25897,
        25390, 26690, 27855, 27276, 26136, 24839,
        23579, 22570, 22269, 21878, 21405, 21024,
        20766, 20541, 20232, 19485, 19279, 19342,
        19895, 20744, 21849, 23073, 24670, 25975,
        27186, 27996, 28736, 28957, 29271, 29490,
        29371, 29068, 28394, 27685, 27330, 27130,
        26901, 26873, 26930, 27084, 27216, 27139,
        27040, 26990, 27063, 27059, 26716, 26960,
        27133, 27954, 28947, 28247, 26477, 24635,
        23168, 22498, 22272, 22291, 22054, 21737,
        21774, 21654, 21462, 21173, 21521, 22334,
        24897, 27609, 31136, 33625, 35517, 36628,
        37545, 37995, 38086, 38389, 38715, 38777,
        38699, 38307, 37894, 37832, 37717, 37563,
        37294, 37604, 37640, 37584, 37196, 36090,
        35053, 34283, 33378, 32754, 31904, 31817,
        31795, 32467, 32401, 31574, 29600, 27540,
        25752, 25196, 24938, 24740, 24465, 24187,
        23923, 23837, 23611, 23198, 23350, 23994,
        26223, 28568, 31749, 34325, 35968, 36745,
        37427, 37718, 37880, 38206, 38488, 38657,
        38762, 38230, 37990, 37673, 37673, 37530,
        37455, 37622, 37993, 38264, 37712, 36707,
        35667, 34737, 33728, 32967, 31725, 31408,
        30991, 31696, 32234, 30990, 28949, 26904,
        25308, 24764, 24417, 24377, 23924, 23583,
        23373, 23398, 23322, 23038, 22992, 23673,
        25741, 28276, 31602, 34031, 35538, 36017,
        36993, 37420, 37409, 37412, 37547, 37664,
        37564, 37271, 36915, 36791, 36625, 36357,
        36265, 36582, 37072, 37238, 36884, 35890,
        34723, 33724, 32888, 32110, 31263, 30907,
        30953, 31413, 31264, 30239, 28562, 26606,
        24808, 24126, 23858, 23809, 23535, 23123,
        23025, 22977, 22759, 22478, 22584, 23160,
        25369, 27933, 31222, 33533, 35135, 35717,
        36555, 36843, 36885, 37138, 37311, 37550,
        37619, 37199, 36951, 36844, 36785, 36624,
        36350, 36614, 37167, 37388, 36880, 36185,
        35197, 34300, 33586, 32645, 32269, 31671,
        31998, 31890, 31643, 30227, 28309, 26457,
        24818, 23982, 23887, 23898, 23618, 23263,
        22987, 22990, 22764, 22399, 22589, 23219,
        25275, 27918, 31219, 33642, 35220, 35757,
        36569, 36876, 36928, 37087, 37191, 37439,
        37385, 36914, 36253, 35983, 35707, 35269,
        34879, 34885, 35075, 35346, 35131, 34313,
        33544, 32824, 32102, 31035, 30421, 30052,
        30107, 30249, 30060, 28893, 27258, 25692,
        24039, 23381, 23128, 23016, 22516, 22196,
        21834, 21720, 21384, 20703, 20462, 20707,
        21880, 23026, 24656, 26407, 28200, 29479,
        30652, 31175, 31370, 31497, 31436, 31377,
        30833, 30254, 29571, 28827, 28635, 28406,
        28136, 28065, 28262, 28945, 29254, 29355,
        29091, 28699, 28218, 27876, 27371, 26955,
        26786, 27440, 27474, 26648, 25523, 24304,
        22914, 22095, 21902, 21947, 21516, 21077,
        20723, 20501, 20234, 19590, 19194, 19374,
        19962, 20551, 21558, 22652, 23820, 25012,
        26294, 27372, 28278, 28957, 29437, 29743,
        30032, 29798, 28856, 28128, 27602, 27416,
        27185, 26997, 27165, 27555, 27727, 27887,
        27803, 27657, 27749, 27593, 27626, 27495,
        27509, 27996, 28413, 27306, 25514, 23667,
        22428, 21687, 21865, 21984, 21701, 21514,
        21395, 21224, 21188, 20759, 20948, 21826,
        24283, 26965, 30507, 33060, 34783, 35322,
        36130, 36494, 36593, 36826, 36985, 37098,
        37200, 36654, 36464, 36257, 36157, 36098,
        35959, 36207, 36559, 36790, 36577, 35727,
        34602, 33578, 32714, 31926, 31142, 30833,
        31109, 31310, 31236, 30184, 28147, 26299,
        24702, 23875, 23785, 23876, 23658, 23346,
        23059, 23039, 22859, 22414, 22518, 23241,
        25541, 28017, 31278, 33530, 35240, 35783,
        36372, 36610, 36651, 36835, 37248, 37128,
        37230, 36841, 36529, 36500, 36445, 36348,
        36152, 36282, 36853, 37137, 36844, 35875,
        34589, 33987, 32959, 32015, 31847, 31404,
        31501, 31824, 31759, 30395, 28511, 26709,
        24940, 24165, 24078, 24101, 23755, 23378,
        23122, 23155, 22902, 22598, 22661, 23335,
        25540, 28023, 31383, 33907, 35592, 36022,
        36643, 36976, 36985, 37123, 37144, 37385,
        37334, 36914, 36630, 36433, 36461, 36246,
        36131, 36534, 37099, 37346, 37098, 36145,
        34997, 34342, 33431, 33088, 32201, 32185,
        31966, 32214, 31736, 30325, 28319, 26480,
        24966, 24194, 24143, 24020, 23750, 23398,
        23179, 23197, 23163, 22808, 22887, 23495,
        25600, 28064, 31494, 33985, 35754, 35972,
        36855, 37135, 37159, 37282, 37347, 37408,
        37444, 37066, 36874, 36696, 36707, 36456,
        36362, 36506, 36831, 37268, 36851, 36027,
        34950, 34149, 33515, 32938, 32777, 32443,
        32620, 32453, 31486, 30006, 28139, 26396,
        24714, 24047, 24050, 24083, 23671, 23354,
        22935, 23009, 23004, 22721, 22911, 23404,
        25429, 27965, 31283, 33772, 35443, 35929,
        36898, 37272, 37270, 37363, 37445, 37550,
        37543, 36995, 36620, 36032, 35843, 35606,
        35131, 35079, 35341, 35546, 35305, 34459,
        33526, 32835, 32000, 31136, 30696, 30054,
        29993, 30178, 29786, 28757, 27433, 25781,
        24272, 23520, 23361, 23268, 22677, 22324,
        21853, 21854, 21574, 21078, 20827, 21003,
        22034, 23208, 24792, 26592, 28313, 29687,
        30737, 31531, 31785, 31805, 31710, 31593,
        31228, 30907, 30011, 29356, 28974, 28618,
        28268, 28292, 28518, 29006, 29463, 29508,
        29035, 28476, 28005, 27658, 26955, 26327,
        26569, 27221, 27427, 26845, 25780, 24384,
        23084, 22221, 22108, 21854, 21401, 21023,
        20711, 20569, 20217, 19687, 19391, 19433,
        20108, 20674, 21676, 22689, 24104, 25283,
        26606, 27556, 28707, 29415, 29983, 30335,
        30650, 30474, 29654, 28741, 28299, 27957,
        28013, 27779, 27966, 28383, 28508, 28442,
        28272, 28345, 28120, 27929, 27848, 27868,
        28378, 29042, 28948, 27699, 25889, 24053,
        22627, 21979, 22025, 22158, 21906, 21615,
        21366, 21327, 21257, 21032, 21261, 22085,
        24437, 27187, 30833, 33448, 35245, 35900,
        36798, 37251, 37335, 37677, 37887, 38066,
        38091, 37784, 37622, 37408, 37354, 37172,
        36995, 37311, 37693, 37887, 37446, 36451,
        35267, 34434, 33733, 32812, 32061, 32089,
        32345, 32564, 32096, 30702, 28412, 26445,
        24933, 24205, 24262, 24154, 23786, 23431,
        23268, 23290, 23166, 22943, 23042, 23492,
        25723, 28273, 31617, 34140, 35901, 36473,
        37122, 37559, 37703, 37933, 38059, 38274,
        38303, 37876, 37688, 37414, 37312, 37234,
        37086, 37421, 37923, 38363, 38127, 37433,
        36363, 35617, 34829, 34036, 33520, 33163,
        33322, 32823, 32221, 30611, 28724, 26678,
        25050, 24246, 24305, 24317, 24019, 23684,
        23449, 23430, 23282, 23111, 23102, 23612,
        25812, 28246, 31571, 33923, 35446, 35973,
        36833, 37164, 37342, 37642, 37713, 37898,
        37963, 37603, 37290, 36955, 36956, 36815,
        36637, 37003, 37526, 37775, 37630, 36722,
        35559, 34742, 34002, 33230, 32809, 32446,
        32614, 32715, 32150, 30568, 28613, 26727,
        25157, 24403, 24354, 24156, 23846, 23539,
        23256, 23233, 23253, 22932, 22969, 23574,
        25737, 28306, 31552, 34093, 35828, 36337,
        37142, 37407, 37471, 37632, 37920, 38095,
        38118, 37735, 37411, 37205, 37064, 36870,
        36749, 37029, 37542, 37684, 37470, 36532,
        35290, 34597, 33819, 33104, 32718, 32512,
        32693, 32784, 32151, 30654, 28721, 26754,
        25228, 24439, 24440, 24284, 23918, 23582,
        23295, 23219, 23124, 22838, 22834, 23420,
        25594, 27965, 31460, 33905, 35501, 36128,
        36898, 37056, 37165, 37294, 37516, 37537,
        37478, 36905, 36316, 36081, 35478, 35101,
        34944, 34929, 35327, 35498, 35065, 34315,
        33258, 32375, 31716, 30629, 29779, 29340,
        29324, 29776, 29963, 29024, 27300, 25726,
        24204, 23325, 23000, 23044, 22515, 22123,
        21767, 21659, 21418, 21029, 20726, 20909,
        21885, 23052, 24710, 26360, 28141, 29612,
        30662, 31096, 31292, 31464, 31393, 31269,
        30742, 30282, 29564, 29052, 28800, 28266,
        28042, 27744, 28022, 28777, 29265, 29112,
        29024, 28670, 28278, 27954, 27769, 27266,
        27354, 27792, 27328, 26650, 25350, 24166,
        22823, 21928, 21880, 21835, 21337, 20910,
        20662, 20406, 20262, 19830, 19438, 19419,
        20138, 20625, 21517, 22611, 24061, 25219,
        26549, 27609, 28951, 29592, 30153, 30634,
        31033, 30704, 29870, 29244, 28944, 28578,
        28295, 28170, 28448, 28772, 28856, 28824,
        28702, 28426, 28269, 28094, 28196, 27913,
        28361, 29118, 29093, 27596, 25651, 23892,
        22387, 21817, 22012, 22081, 21839, 21599,
        21460, 21442, 21353, 21143, 21307, 22058,
        24474, 27218, 30787, 33457, 35273, 36011,
        36866, 37346, 37510, 37825, 38248, 38518,
        38621, 38270, 37907, 37766, 37646, 37477,
        37261, 37766, 38302, 38496, 38279, 37451,
        36435, 35601, 34543, 33925, 33161, 32709,
        32881, 32719, 32030, 30569, 28389, 26384,
        24822, 24126, 24296, 24461, 24172, 23917,
        23664, 23528, 23498, 23108, 23044, 23585,
        25750, 28353, 31634, 34080, 35646, 36079,
        36758, 36958, 37169, 37410, 37601, 37773,
        37672, 37245, 36995, 36519, 36533, 36464,
        36274, 36577, 37042, 37242, 36996, 36020,
        34759, 33932, 33001, 32254, 31581, 31270,
        31776, 32287, 32117, 30604, 28597, 26528,
        24897, 24117, 24330, 24524, 24179, 23915,
        23877, 23725, 23499, 23077, 22919, 23596,
        25786, 28277, 31689, 33984, 35449, 35832,
        36526, 36882, 36959, 37105, 37355, 37510,
        37519, 37159, 36810, 36474, 36565, 36459,
        36267, 36585, 37062, 37301, 37110, 36312,
        35274, 34483, 33804, 33187, 32918, 32800,
        32820, 32686, 31877, 30754, 28729, 26781,
        25257, 24292, 24519, 24610, 24207, 23882,
        23748, 23692, 23602, 23299, 23493, 23934,
        26095, 28615, 32005, 34393, 35901, 36370,
        37303, 37585, 37565, 37943, 37990, 38062,
        38124, 37692, 37429, 37067, 37005, 36949,
        36646, 37019, 37402, 37476, 37062, 36292,
        35247, 34449, 33822, 32999, 32569, 32485,
        32879, 32866, 32106, 30590, 28500, 26640,
        25050, 24352, 24382, 24432, 24041, 23871,
        23754, 23547, 23380, 23043, 23089, 23586,
        25660, 28125, 31541, 33932, 35554, 36040,
        36630, 36857, 36971, 37020, 37181, 37304,
        37487, 36880, 36159, 35654, 35564, 35302,
        34901, 34910, 35229, 35529, 35253, 34636,
        33774, 33031, 32090, 31084, 30377, 29800,
        29891, 30297, 30087, 29109, 27248, 25675,
        24173, 23461, 23473, 23536, 22979, 22600,
        22224, 21999, 21842, 21230, 20984, 21009,
        22206, 23293, 25031, 26663, 28460, 29684,
        30672, 31072, 31512, 31397, 31348, 31058,
        30639, 30135, 29432, 28907, 28471, 28126,
        27925, 27809, 27954, 28318, 28834, 28724,
        28518, 28181, 27605, 27165, 26661, 26316,
        26514, 27119, 27223, 26513, 25354, 23947,
        22782, 22034, 22004, 21934, 21541, 21189,
        20806, 20556, 20390, 19881, 19496, 19418,
        19949, 20714, 21604, 22589, 23902, 25052,
        26384, 27504, 28492, 29177, 29732, 29982,
        30197, 29994, 29104, 28063, 27510, 27184,
        27021, 26775, 26879, 27097, 27143, 27119,
        27142, 27110, 27107, 26775, 26886, 26834,
        27667, 28552, 28495, 27452, 25565, 23764,
        22421, 21724, 21844, 22095, 21862, 21554,
        21466, 21428, 21326, 21131, 21356, 22078,
        24352, 27030, 30528, 33059, 34851, 35312,
        36154, 36505, 36568, 36774, 37118, 37178,
        37214, 36869, 36460, 36140, 36118, 35976,
        35708, 36061, 36546, 36640, 36326, 35617,
        34407, 33339, 32612, 31904, 30899, 31018,
        31444, 32118, 31907, 30526, 28336, 26355,
        24734, 24085, 24069, 24101, 23761, 23408,
        23348, 23215, 23144, 22998, 22906, 23350,
        25446, 27975, 31231, 33490, 35026, 35632,
        36220, 36620, 36723, 36903, 36974, 37261,
        37289, 36930, 36633, 36432, 36440, 36248,
        36290, 36496, 36996, 37118, 36876, 35845,
        34725, 33861, 32843, 32012, 31430, 31386,
        31681, 32285, 32111, 30656, 28577, 26503,
        25097, 24328, 24180, 24265, 23924, 23566,
        23332, 23302, 23252, 23072, 22909, 23491,
        25587, 27993, 31313, 33752, 35371, 35944,
        36773, 36978, 36973, 37072, 37051, 37318,
        37513, 37156, 36774, 36567, 36449, 36343,
        36090, 36426, 36812, 36980, 36662, 35654,
        34351, 33697, 32613, 31711, 31450, 31103,
        31332, 31917, 31781, 30581, 28726, 27009,
        25316, 24548, 24480, 24308, 24028, 23725,
        23537, 23472, 23452, 23209, 23207, 23616,
        25862, 28298, 31361, 33843, 35585, 36274,
        36993, 37242, 37304, 37390, 37603, 37711,
        37606, 37214, 36938, 36825, 36834, 36712,
        36546, 36786, 37042, 37216, 36708, 35737,
        34684, 33970, 32902, 31870, 31381, 31149,
        31595, 32431, 32026, 30680, 28690, 26598,
        25163, 24438, 24145, 23965, 23674, 23347,
        23177, 23117, 23008, 22778, 22633, 23081,
        25184, 27593, 30702, 33104, 34820, 35496,
        36326, 36505, 36678, 36890, 36892, 37020,
        36845, 36321, 35868, 35555, 35385, 35109,
        34850, 34972, 35122, 35141, 34713, 33882,
        32938, 32034, 31020, 30197, 29527, 29055,
        29407, 30121, 29997, 28810, 27115, 25592,
        24187, 23142, 22938, 22665, 22161, 21630,
        21378, 21229, 21101, 20737, 20523, 20506,
        21498, 22577, 24152, 25606, 27360, 28560,
        29677, 30179, 30298, 30321, 30336, 30105,
        29755, 29411, 28656, 27992, 27600, 27304,
        27049, 26967, 27127, 27538, 27917, 27958,
        27588, 27425, 27008, 26673, 26142, 26085,
        26687, 27249, 26797, 25910, 24683, 23459,
        22227, 21495, 21310, 21086, 20684, 20264,
        19973, 19806, 19581, 19383, 19109, 18869,
        19428, 20015, 20768, 21720, 22900, 24092,
        25373, 26353, 27383, 27979, 28699, 28908,
        29089, 28964, 28209, 27484, 27017, 26663,
        26474, 26387, 26546, 26915, 27031, 27065,
        27014, 26793, 26985, 26765, 26903, 27023,
        28019, 28423, 27579, 26359, 24589, 22936,
        21453, 20977, 20803, 20899, 20668, 20451,
        20324, 20391, 20383, 20388, 20564, 21114,
        23298, 25505, 28507, 30854, 32956, 34017,
        35016, 35396, 35587, 35741, 35997, 36161,
        36165, 35827, 35512, 35094, 35088, 35045,
        34863, 34957, 35262, 35423, 35352, 34880,
        33870, 33122, 32325, 31660, 31083, 31367,
        31863, 31412, 30246, 28889, 26762, 25002,
        23456, 22720, 22733, 22864, 22467, 22145,
        21983, 22045, 21829, 21813, 21997, 22232,
        24139, 26224, 29145, 31450, 33325, 34444,
        35086, 35356, 35352, 35544, 35693, 35762,
        35756, 35411, 35128, 34994, 34857, 34731,
        34449, 34624, 35043, 35045, 34912, 34208,
        32994, 32467, 31603, 30855, 30608, 30868,
        31549, 31661, 30667, 29098, 27114, 25315,
        23564, 22865, 22674, 22738, 22335, 21911,
        21876, 21874, 21714, 21683, 21769, 22076,
        23999, 26330, 29003, 31196, 33068, 34227,
        35033, 35415, 35587, 35608, 35796, 35902,
        36013, 35708, 35309, 35053, 35040, 34900,
        34708, 34856, 35024, 35316, 35193, 34396,
        33400, 32566, 31541, 30842, 30472, 30457,
        31072, 31416, 30812, 29215, 27341, 25409,
        23835, 23059, 22905, 22777, 22402, 22131,
        21969, 21839, 21819, 21806, 21762, 22145,
        24117, 26273, 28922, 31153, 33064, 34178,
        35173, 35586, 35711, 35874, 36057, 36140,
        36196, 35823, 35563, 35315, 35320, 35199,
        35107, 35167, 35417, 35681, 35535, 34700,
        33641, 32701, 31842, 31062, 30697, 30772,
        31411, 31522, 30618, 29156, 27117, 25359,
        23694, 22923, 22673, 22633, 22240, 21877,
        21728, 21622, 21580, 21434, 21446, 21812,
        23691, 25686, 28460, 30737, 32619, 33860,
        34728, 35036, 35166, 35232, 35389, 35456,
        35482, 35052, 34760, 34362, 34158, 33919,
        33761, 33541, 33602, 33778, 33793, 33031,
        32212, 31415, 30522, 29606, 29001, 28797,
        29410, 29711, 29031, 27747, 26212, 24681,
        23203, 22495, 22290, 22059, 21573, 21223,
        20999, 20803, 20561, 20291, 19957, 19866,
        20765, 21923, 23524, 25069, 26802, 28036,
        29275, 29801, 30107, 30144, 30136, 30065,
        29713, 29265, 28627, 28054, 27727, 27423,
        27253, 27101, 27349, 27795, 28115, 28123,
        27747, 27411, 27018, 26642, 26078, 25917,
        26574, 27137, 26858, 25981, 24671, 23515,
        22208, 21516, 21219, 20973, 20447, 19959,
        19705, 19567, 19335, 19179, 18809, 18640,
        19074, 19661, 20561, 21627, 22992, 24278,
        25553, 26442, 27265, 27822, 28352, 28572,
        28733, 28560, 27746, 27015, 26790, 26377,
        26169, 26086, 26208, 26401, 26500, 26582,
        26514, 26344, 26245, 25940, 25933, 26309,
        27776, 28249, 27695, 26385, 24740, 23204,
        21771, 21118, 20901, 20947, 20625, 20313,
        20251, 20237, 20244, 20276, 20446, 20851,
        22746, 24860, 27773, 30156, 32181, 33448,
        34577, 35023, 35220, 35332, 35651, 35650,
        35651, 35429, 35163, 34859, 34766, 34683,
        34511, 34644, 34766, 35028, 34891, 34159,
        33201, 32423, 31652, 30989, 30326, 30789,
        31846, 31630, 30437, 28691, 26658, 24829,
        23241, 22491, 22408, 22303, 21912, 21481,
        21301, 21389, 21307, 21321, 21413, 21747,
        23426, 25564, 28304, 30417, 32491, 33601,
        34525, 34774, 34813, 34950, 35047, 35123,
        35204, 34910, 34653, 34393, 34423, 34327,
        34248, 34290, 34537, 34629, 34318, 33512,
        32740, 31936, 30867, 30089, 29659, 29631,
        30792, 30850, 30253, 28545, 26754, 24824,
        23351, 22627, 22427, 22170, 21785, 21330,
        21271, 21285, 21247, 21219, 21338, 21601,
        23346, 25466, 28096, 30271, 32148, 33282,
        34234, 34525, 34767, 34880, 35144, 35258,
        35388, 35037, 34643, 34285, 34351, 34162,
        34099, 34093, 34437, 34601, 34608, 33741,
        32956, 32133, 31292, 30807, 30401, 30501,
        31087, 30899, 30088, 28469, 26738, 24810,
        23334, 22560, 22329, 22352, 21988, 21686,
        21501, 21543, 21437, 21455, 21576, 21866,
        23514, 25591, 28145, 30315, 32075, 33250,
        34180, 34470, 34601, 34839, 35092, 35194,
        35193, 34977, 34686, 34448, 34526, 34389,
        34349, 34414, 34569, 34817, 34707, 34030,
        33248, 32595, 31573, 30654, 30274, 30515,
        31608, 31203, 30051, 28526, 26719, 24987,
        23373, 22570, 22439, 22316, 21855, 21577,
        21419, 21390, 21277, 21287, 21475, 21736,
        23496, 25485, 28125, 30308, 32242, 33468,
        34354, 34522, 34629, 34809, 34854, 34901,
        34787, 34489, 34139, 33820, 33623, 33287,
        32920, 32840, 32891, 33047, 33128, 32460,
        31744, 31093, 30435, 29584, 29321, 29533,
        30431, 29989, 29018, 27702, 26074, 24618,
        23212, 22310, 22095, 21909, 21420, 21024,
        20737, 20607, 20464, 20262, 20223, 20080,
        20952, 21950, 23480, 25019, 26825, 27917,
        29044, 29518, 29800, 29820, 29796, 29682,
        29322, 28918, 28236, 27650, 27240, 26879,
        26726, 26682, 26894, 27461, 27884, 27942,
        27622, 27305, 27023, 26688, 26255, 26495,
        27767, 27678, 27058, 26109, 24922, 23646,
        22434, 21595, 21299, 20946, 20487, 20030,
        19748, 19637, 19495, 19294, 19193, 18939,
        19339, 19900, 20848, 21865, 23206, 24338,
        25482, 26422, 27315, 28044, 28546, 28906,
        29096, 28876, 28227, 27516, 27087, 26648,
        26416, 26392, 26592, 26814, 26908, 26786,
        26816, 26634, 26514, 26299, 26454, 26760,
        28578, 28609, 27913, 26523, 24989, 23375,
        22078, 21410, 21377, 21269, 20951, 20655,
        20495, 20585, 20586, 20641, 21037, 21377,
        23389, 25622, 28487, 30870, 32993, 34211,
        35256, 35617, 35821, 36032, 36282, 36340,
        36537, 36296, 36017, 35994, 35948, 35859,
        35765, 35753, 36058, 36315, 36050, 35248,
        34222, 33545, 32731, 31885, 31397, 32224,
        33166, 32505, 31230, 29430, 27526, 25691,
        24203, 23484, 23436, 23207, 22817, 22562,
        22427, 22423, 22342, 22402, 22785, 22949,
        24695, 26750, 29468, 31734, 33757, 34643,
        35517, 35895, 35908, 36210, 36438, 36612,
        36620, 36319, 36123, 35978, 35777, 35661,
        35575, 35845, 35950, 36177, 36075, 35137,
        34164, 33281, 32329, 31468, 31167, 31809,
        33196, 32601, 31436, 29679, 27635, 25833,
        24315, 23690, 23441, 23198, 22818, 22551,
        22354, 22359, 22348, 22282, 22676, 22926,
        24621, 26681, 29533, 31842, 33888, 34826,
        35747, 36136, 36303, 36479, 36565, 36689,
        36740, 36378, 36043, 35930, 35760, 35676,
        35644, 35661, 35851, 36151, 36010, 35225,
        34283, 33618, 32731, 32070, 32037, 32639,
        33449, 32635, 31361, 29759, 27891, 25833,
        24246, 23580, 23512, 23310, 22893, 22577,
        22392, 22392, 22384, 22428, 22749, 23062,
        24803, 26788, 29596, 31831, 33770, 35053,
        35888, 36010, 36251, 36312, 36525, 36806,
        36647, 36385, 36059, 35900, 35703, 35622,
        35529, 35492, 35553, 35692, 35387, 34589,
        33958, 33170, 32316, 31738, 31830, 32595,
        33382, 32527, 31194, 29578, 27652, 25848,
        24311, 23460, 23333, 23232, 22764, 22486,
        22372, 22377, 22270, 22231, 22613, 22756,
        24529, 26522, 29281, 31635, 33533, 34586,
        35502, 35664, 35733, 35823, 35880, 35880,
        35985, 35608, 35166, 34949, 34612, 34452,
        34221, 34122, 34192, 34293, 34079, 33352,
        32510, 31804, 31013, 30103, 29685, 30215,
        31615, 31118, 29956, 28531, 26846, 25326,
        23854, 23167, 22917, 22525, 22035, 21592,
        21291, 21177, 21053, 20840, 20917, 20728,
        21492, 22631, 24221, 25848, 27553, 28954,
        30034, 30639, 30852, 30835, 30760, 30617,
        30216, 29643, 28976, 28507, 28110, 27700,
        27529, 27588, 27868, 28237, 28729, 28777,
        28389, 28315, 27919, 27627, 27641, 28345,
        28867, 28212, 27567, 26869, 25442, 24145,
        22947, 22213, 21917, 21597, 21050, 20515,
        20215, 20142, 20010, 19829, 19837, 19704,
        19881, 20463, 21414, 22572, 23859, 25007,
        26286, 27291, 28202, 28946, 29344, 29639,
        29897, 29777, 29078, 28579, 28012, 27615,
        27549, 27595, 27641, 27979, 28210, 28217,
        28362, 28337, 28312, 28292, 28823, 29935,
        30106, 29615, 28625, 27150, 25427, 23841,
        22489, 21898, 21932, 21943, 21651, 21412,
        21136, 21227, 21341, 21382, 21880, 22598,
        24631, 26989, 29734, 32143, 34391, 35745,
        36919, 37318, 37299, 37521, 37631, 37755,
        37849, 37594, 37152, 37126, 37039, 36981,
        36846, 36841, 37047, 37160, 36856, 35761,
        34906, 34139, 33489, 32784, 32243, 33516,
        34568, 33545, 32173, 30495, 28426, 26405,
        24923, 24327, 24097, 24009, 23636, 23326,
        23164, 22936, 23025, 23031, 23496, 23734,
        25322, 27439, 30185, 32466, 34420, 35521,
        36394, 36642, 36761, 36809, 36901, 37020,
        37068, 36698, 36472, 36403, 36268, 36239,
        36078, 36264, 36588, 36902, 36690, 35842,
        35119, 34309, 33636, 33128, 33231, 34024,
        34658, 33370, 32176, 30417, 28428, 26673,
        25124, 24383, 24114, 24167, 23746, 23444,
        23259, 23217, 23197, 23185, 23561, 23818,
        25395, 27464, 30307, 32365, 34165, 35240,
        36050, 36221, 36283, 36439, 36504, 36710,
        36921, 36427, 36032, 35964, 35817, 35794,
        35626, 35744, 35977, 36337, 36128, 35343,
        34456, 33700, 33127, 32626, 32706, 33837,
        34144, 33043, 32013, 30575, 28624, 26782,
        25168, 24479, 24239, 24063, 23701, 23279,
        23166, 23130, 23161, 23148, 23509, 23799,
        25471, 27502, 30255, 32610, 34374, 35506,
        36167, 36345, 36483, 36680, 36819, 36964,
        37029, 36613, 36380, 36234, 36092, 35935,
        35867, 35976, 36174, 36354, 36078, 35454,
        34455, 33604, 33082, 32537, 32560, 33891,
        34075, 32982, 31860, 30198, 28433, 26770,
        25179, 24314, 24157, 23944, 23387, 23140,
        23067, 22994, 22940, 22904, 23375, 23742,
        25401, 27491, 30268, 32611, 34417, 35604,
        36531, 36818, 36875, 36998, 37110, 37296,
        37392, 36897, 36333, 35999, 35656, 35441,
        35062, 34861, 34834, 35039, 34876, 34243,
        33596, 32946, 32395, 31698, 31837, 32422,
        32452, 31564, 30367, 29055, 27393, 25924,
        24446, 23832, 23517, 23189, 22544, 22096,
        21913, 21776, 21602, 21460, 21622, 21817,
        22398, 23322, 25034, 26611, 28339, 29635,
        30840, 31339, 31521, 31709, 31548, 31366,
        30966, 30464, 29621, 29087, 28491, 28044,
        27791, 27842, 28011, 28398, 28802, 28718,
        28372, 27955, 27537, 27606, 27690, 28652,
        28897, 28307, 27657, 26860, 25377, 24107,
        22869, 22202, 21806, 21483, 20986, 20599,
        20296, 20238, 20082, 19802, 19747, 19718,
        19953, 20489, 21431, 22514, 23961, 25355,
        26602, 27510, 28306, 28788, 29166, 29536,
        29557, 29406, 28543, 28013, 27430, 27187,
        26980, 27033, 27173, 27392, 27578, 27561,
        27771, 27772, 27625, 27497, 28461, 29738,
        30108, 29309, 28351, 26974, 25265, 23835,
        22651, 21874, 21763, 21798, 21625, 21321,
        21071, 21093, 21177, 21223, 21854, 22429,
        24271, 26533, 29373, 31848, 33979, 35182,
        36245, 36535, 36653, 36699, 36928, 37015,
        37202, 36850, 36522, 36332, 36201, 36249,
        36159, 36327, 36462, 36629, 36405, 35531,
        34835, 34261, 33676, 33203, 33164, 34409,
        34469, 33085, 31815, 30031, 27989, 26190,
        24702, 24068, 23964, 23932, 23508, 23271,
        23113, 23018, 22997, 23046, 23450, 23841,
        25520, 27645, 30235, 32532, 34497, 35575,
        36403, 36578, 36659, 36738, 36832, 37016,
        37050, 36685, 36273, 36180, 36019, 35842,
        35768, 35846, 36149, 36376, 36134, 35383,
        34660, 33885, 33017, 32553, 33152, 34742,
        34849, 33554, 32208, 30423, 28566, 26714,
        24969, 24269, 24219, 24115, 23772, 23482,
        23383, 23256, 23257, 23332, 23680, 24290,
        25831, 27888, 30556, 32830, 34816, 35867,
        36625, 36829, 36969, 36981, 37068, 37261,
        37272, 36826, 36465, 36220, 36148, 36163,
        35938, 36028, 36233, 36503, 36350, 35533,
        34639, 33834, 33133, 32679, 33512, 34804,
        34652, 33661, 32448, 30767, 28762, 27016,
        25338, 24514, 24375, 24224, 23739, 23447,
        23222, 23171, 23187, 23232, 23674, 24368,
        25814, 27894, 30584, 32926, 34831, 35890,
        36720, 36903, 36869, 37091, 37307, 37480,
        37440, 37095, 36805, 36642, 36461, 36358,
        36234, 36431, 36573, 36790, 36530, 35562,
        34767, 34197, 33452, 33026, 33940, 35455,
        35489, 34041, 32394, 30961, 28958, 27151,
        25394, 24722, 24370, 24319, 23895, 23687,
        23451, 23408, 23307, 23304, 23755, 24278,
        25747, 27698, 30325, 32614, 34547, 35652,
        36463, 36864, 36803, 36907, 36997, 37106,
        37070, 36651, 36098, 35667, 35487, 35301,
        35067, 34928, 34738, 35004, 34748, 34090,
        33476, 32893, 32250, 31997, 33010, 33477,
        33064, 31953, 30548, 29236, 27623, 26063,
        24653, 23879, 23508, 23275, 22890, 22462,
        22124, 22085, 21978, 21862, 21954, 22144,
        22972, 23788, 25099, 26677, 28494, 29881,
        31048, 31513, 31841, 32092, 31975, 31740,
        31341, 30738, 30108, 29767, 29213, 28727,
        28557, 28425, 28664, 29198, 29768, 29894,
        29628, 29120, 28838, 28936, 29366, 29754,
        29309, 28603, 27695, 26749, 25382, 24128,
        22914, 22150, 21808, 21484, 21041, 20569,
        20190, 20191, 19974, 19741, 19896, 19949,
        20062, 20497, 21501, 22349, 23741, 24870,
        26044, 26849, 27618, 28421, 28852, 29139,
        29385, 29103, 28291, 27778, 27178, 26837,
        26609, 26487, 26507, 26700, 26824, 26903,
        26834, 26646, 26476, 26615, 27940, 28693,
        28677, 27946, 27133, 25996, 24610, 23132]).astype(dtype)

    # Set the index if necessary
    if as_series:
        return pd.Series(rslt)
    return rslt
