/*
Gwenview - A simple image viewer for TDE
Copyright 2000-2004 Aur�lien G�teau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/
// Our header
#include "imageviewtools.h"

// KDE 
#include <tdeaction.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <kstandarddirs.h>

// Local
#include "imageviewconfig.h"

namespace Gwenview {


// Helper function
static TQCursor loadCursor(const TQString& name) {
	TQString path;
	path=locate("data", TQString("gwenview/cursors/%1.png").arg(name));
	return TQCursor(TQPixmap(path));
}


//----------------------------------------------------------------------------
//
// ImageView::ToolBase
//
//----------------------------------------------------------------------------
ImageView::ToolBase::ToolBase(ImageView* view)
: mView(view) {}


ImageView::ToolBase::~ToolBase() {}

void ImageView::ToolBase::mouseMoveEvent(TQMouseEvent*) {}
void ImageView::ToolBase::leftButtonPressEvent(TQMouseEvent*) {}
void ImageView::ToolBase::leftButtonReleaseEvent(TQMouseEvent*) {}

void ImageView::ToolBase::midButtonReleaseEvent(TQMouseEvent*) {
	mView->zoomToFit()->activate();
}

void ImageView::ToolBase::rightButtonPressEvent(TQMouseEvent* event) {
	emit mView->requestContextMenu(event->globalPos());
}

void ImageView::ToolBase::rightButtonReleaseEvent(TQMouseEvent*) {
}

void ImageView::ToolBase::wheelEvent(TQWheelEvent* event) {
	event->accept();
}

void ImageView::ToolBase::updateCursor() {
	mView->viewport()->setCursor(ArrowCursor);
}


//----------------------------------------------------------------------------
//
// ImageView::ZoomTool
//
//----------------------------------------------------------------------------
ImageView::ZoomTool::ZoomTool(ImageView* view)
: ImageView::ToolBase(view) {
	mZoomCursor=loadCursor("zoom");
}


void ImageView::ZoomTool::zoomTo(const TQPoint& pos, bool in) {
	if (!mView->canZoom(in)) return;

	TQPoint centerPos=TQPoint(mView->visibleWidth(), mView->visibleHeight())/2;
	// Compute image position
	TQPoint imgPos=mView->viewportToContents(pos) - mView->offset();
	double newZoom=mView->computeZoom(in);

	imgPos*=newZoom/mView->zoom();
	imgPos=imgPos-pos+centerPos;
	mView->setZoom(newZoom, imgPos.x(), imgPos.y());
}


void ImageView::ZoomTool::leftButtonReleaseEvent(TQMouseEvent* event) {
	zoomTo(event->pos(), true);
}


void ImageView::ZoomTool::wheelEvent(TQWheelEvent* event) {
	zoomTo(event->pos(), event->delta()>0);
	event->accept();
}


void ImageView::ZoomTool::rightButtonPressEvent(TQMouseEvent*) {
}


void ImageView::ZoomTool::rightButtonReleaseEvent(TQMouseEvent* event) {
	zoomTo(event->pos(), false);
}


void ImageView::ZoomTool::updateCursor() {
	mView->viewport()->setCursor(mZoomCursor);
}


TQString ImageView::ZoomTool::hint() const {
    return i18n("Left click to zoom in, right click to zoom out. You can also use the mouse wheel.");
}


//----------------------------------------------------------------------------
//
// ImageView::ScrollTool
//
//----------------------------------------------------------------------------
ImageView::ScrollTool::ScrollTool(ImageView* view)
: ImageView::ToolBase(view)
, mScrollStartX(0), mScrollStartY(0)
, mDragStarted(false) {
}


void ImageView::ScrollTool::leftButtonPressEvent(TQMouseEvent* event) {
	mScrollStartX=event->x();
	mScrollStartY=event->y();
	mView->viewport()->setCursor(SizeAllCursor);
	mDragStarted=true;
}


void ImageView::ScrollTool::mouseMoveEvent(TQMouseEvent* event) {
	if (!mDragStarted) return;

	int deltaX,deltaY;

	deltaX=mScrollStartX - event->x();
	deltaY=mScrollStartY - event->y();

	mScrollStartX=event->x();
	mScrollStartY=event->y();
	mView->scrollBy(deltaX,deltaY);
}


void ImageView::ScrollTool::leftButtonReleaseEvent(TQMouseEvent*) {
	if (!mDragStarted) return;

	mDragStarted=false;
	mView->viewport()->setCursor(ArrowCursor);
}


void ImageView::ScrollTool::wheelEvent(TQWheelEvent* event) {
	if (ImageViewConfig::mouseWheelScroll()) {
		int deltaX, deltaY;

		if (event->state() & AltButton || event->orientation()==Horizontal) {
			deltaX = event->delta();
			deltaY = 0;
		} else {
			deltaX = 0;
			deltaY = event->delta();
		}
		mView->scrollBy(-deltaX, -deltaY);
	} else {
		if (event->delta()<0) {
			mView->emitSelectNext();
		} else {
			mView->emitSelectPrevious();
		}
	}
	event->accept();
}


void ImageView::ScrollTool::updateCursor() {
	if (mDragStarted) {
		mView->viewport()->setCursor(SizeAllCursor);
	} else {
		mView->viewport()->setCursor(ArrowCursor);
	}
}


TQString ImageView::ScrollTool::hint() const {
    return i18n("Drag to move the image, middle-click to toggle auto-zoom. Hold the Control key to switch to the zoom tool.");
}


} // namespace
