/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Dario Freddi <drf@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef POLKIT_TQT_GUI_ACTIONBUTTONS_H
#define POLKIT_TQT_GUI_ACTIONBUTTONS_H

#include "polkit-tqt-gui-actionbutton.h"


namespace PolkitTQt
{

namespace Gui
{

class ActionButtonsPrivate;

/**
 * \class ActionButtons polkit-tqt-gui-actionbuttons.h ActionButtons
 * \author Dario Freddi <drf@kde.org>
 *
 * \brief Class used to hold and update a list of TQButtons
 *
 * This class is a convenience wrapper around ActionButton that lets
 * you associate an undefined number of TQButtons with a single action.
 * Every button will be updated accordingly upon action's properties changes.
 *
 * \see ActionButton
 */
class ActionButtons : public ActionButton
{
  TQ_OBJECT

  public:
    /**
     * Constructs a new ActionButtons. You need to pass this
     * constructor an existing list of TQButtons, whose properties
     * will be modified according to the underlying Action
     * object. As ActionButtons inherits from Action, you can
     * define your buttons' behavior right through this wrapper.
     *
     * \see Action
     *
     * \param buttons the TQButton to associate to this ActionButtons
     * \param actionId the action Id to create the underlying Action
     * \param parent the parent object
     */
    explicit ActionButtons(TQValueList<TQButton*> &buttons,
            const TQString &actionId = TQString::null, TQObject *parent = nullptr);
    virtual ~ActionButtons();

    /**
     * Sets a list of buttons associated to the underlying action.
     *
     * \note If you are calling this function, you're probably
     *       changing the buttons list the action is referring to. If this
     *       is the case, please note that Polkit-TQt does not handle
     *       the previous buttons' memory, so you should take care of
     *       deleting them yourself (if needed). You can retrieve it by
     *       using buttons()
     *
     * \see buttons
     *
     * \param buttons the new buttons associated with the underlying action
     */
    void setButtons(TQValueList<TQButton*> &buttons);

    /**
     * Returns the current buttons list
     *
     * \return the buttons currently associated with the underlying action
     */
    TQValueList<TQButton*> buttons() const;

    /**
     * Adds a button to the current button list. The button's properties
     * will be updated according to the action upon adding.
     *
     * \param button the button to add
     */
    void addButton(TQButton *button);

    /**
     * Removes a button from the current list. Please note that Polkit-TQt
     * does not handle the removed button's memory, so you should take care of
     * deleting it yourself (if needed).
     *
     * \param button the button to remove
     */
    void removeButton(TQButton *button);

  private:
    // Disable copy
    ActionButtons(const ActionButtons&);
    ActionButtons& operator=(const ActionButtons&);
};

}

}

#endif
