/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Jaroslav Reznik <jreznik@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <polkit/polkit.h>

#include "polkit-tqt-subject.h"

#include <tqshared.h>
#include <tqstring.h>


namespace PolkitTQt
{

//--------------------------------------
// Subject::Data
//--------------------------------------

class Subject::Data : public TQShared
{
  public:
    Data() : subject(nullptr)
    {
    }

    Data(const Data &other) : subject(other.subject)
    {
      if (subject)
      {
        g_object_ref(subject);
      }
    }

    Data(PolkitSubject *_subject) : subject(_subject)
    {
      if (subject)
      {
        g_object_ref(subject);
      }
    }

    ~Data()
    {
      if (subject)
      {
        g_object_unref(subject);
      }
    }

    PolkitSubject *subject;
};

//--------------------------------------
// Subject
//--------------------------------------

Subject::Subject() : d(new Data)
{
}

Subject::Subject(PolkitSubject *pkSubject) : d(new Data(pkSubject))
{
}

Subject::Subject(const Subject &other) : d(other.d)
{
  d->ref();
}

Subject& Subject::operator=(const Subject &other)
{
  if (d != other.d)
  {
    if (d->deref())
    {
      delete d;
    }
    d = other.d;
    d->ref();
  }
  return *this;
}

Subject::~Subject()
{
  if (d->deref())
  {
    delete d;
  }
}

bool Subject::isValid() const
{
  return (d->subject != nullptr);
}

PolkitSubject* Subject::subject() const
{
  return d->subject;
}

void Subject::setSubject(PolkitSubject *subject)
{
  if (d->subject != subject)
  {
    if (d->subject)
    {
      g_object_unref(d->subject);
    }
    d->subject = subject;
    if (d->subject)
    {
      g_object_ref(d->subject);
    }
  }
}

TQString Subject::toString() const
{
  if (!d->subject)
  {
    return TQString::null;
  }
  return TQString::fromUtf8(polkit_subject_to_string(d->subject));
}

Subject Subject::fromString(const TQString &string)
{
  if (string.isEmpty())
  {
    tqWarning(TQString("Cannot create valid Subject from empty string."));
    return Subject();
  }

  GError *error = nullptr;
  PolkitSubject *polsub = polkit_subject_from_string(string.utf8().data(), &error);
  if (error != nullptr)
  {
    tqWarning(TQString("Cannot create valid Subject from string: %1").arg(error->message));
    return Subject();
  }

  return Subject(polsub);
}

//--------------------------------------
// UnixProcessSubject
//--------------------------------------

UnixProcessSubject::UnixProcessSubject(TQ_LONG pid) : Subject()
{
  setSubject(polkit_unix_process_new_for_owner(pid, 0 , -1));
}

UnixProcessSubject::UnixProcessSubject(TQ_LONG pid, TQ_ULONG startTime) : Subject()
{
  setSubject(polkit_unix_process_new_for_owner(pid, startTime, -1));
}

UnixProcessSubject::UnixProcessSubject(PolkitUnixProcess *pkUnixProcess)
        : Subject((PolkitSubject*)pkUnixProcess)
{
}

TQ_LONG UnixProcessSubject::pid() const
{
  return polkit_unix_process_get_pid((PolkitUnixProcess*)subject());
}

TQ_LONG UnixProcessSubject::startTime() const
{
  return polkit_unix_process_get_start_time((PolkitUnixProcess*)subject());
}

void UnixProcessSubject::setPid(TQ_LONG pid)
{
  polkit_unix_process_set_pid((PolkitUnixProcess*)subject(), pid);
}

//--------------------------------------
// SystemBusNameSubject
//--------------------------------------

SystemBusNameSubject::SystemBusNameSubject(const TQString &name) : Subject()
{
  setSubject(polkit_system_bus_name_new(name.utf8().data()));
}

SystemBusNameSubject::SystemBusNameSubject(PolkitSystemBusName *pkSystemBusName)
        : Subject((PolkitSubject*)pkSystemBusName)
{
}

TQString SystemBusNameSubject::name() const
{
  return TQString::fromUtf8(polkit_system_bus_name_get_name((PolkitSystemBusName*)subject()));
}

void SystemBusNameSubject::setName(const TQString &name)
{
  polkit_system_bus_name_set_name((PolkitSystemBusName*)subject(), name.utf8().data());
}

//--------------------------------------
// UnixSessionSubject
//--------------------------------------

UnixSessionSubject::UnixSessionSubject(const TQString &sessionId) : Subject()
{
  setSubject(polkit_unix_session_new(sessionId.utf8().data()));
}

UnixSessionSubject::UnixSessionSubject(TQ_LONG pid) : Subject()
{
  GError *error = nullptr;
  setSubject(polkit_unix_session_new_for_process_sync(pid, nullptr, &error));
  if (error != nullptr)
  {
    tqWarning(TQString("Cannot create unix session subject from pid: %1").arg(error->message));
    setSubject(nullptr);
  }
}

UnixSessionSubject::UnixSessionSubject(PolkitSystemBusName *pkUnixSession)
        : Subject((PolkitSubject*)pkUnixSession)
{
}

TQString UnixSessionSubject::sessionId() const
{
  return TQString::fromUtf8(polkit_unix_session_get_session_id((PolkitUnixSession*)subject()));
}

void UnixSessionSubject::setSessionId(const TQString &sessionId)
{
  polkit_unix_session_set_session_id((PolkitUnixSession*)subject(), sessionId.utf8().data());
}

}
