/*
    This file is part of KOrganizer.
    Copyright (c) 2002 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#include "datenavigator.h"

#include "koglobals.h"

#include <kcalendarsystem.h>

#include <kdebug.h>
#include <tdeglobal.h>
#include <tdelocale.h>

using namespace KCal;

DateNavigator::DateNavigator( TQObject *parent, const char *name )
  : TQObject( parent, name )
{
  mSelectedDates.append( TQDate::currentDate() );
}

DateNavigator::~DateNavigator()
{
}

DateList DateNavigator::selectedDates()
{
  return mSelectedDates;
}

int DateNavigator::datesCount() const
{
  return mSelectedDates.count();
}

void DateNavigator::selectDates( const DateList &dateList )
{
  if ( dateList.count() > 0 ) {
    mSelectedDates = dateList;

    emitSelected();
  }
}

void DateNavigator::selectDate( const TQDate &date )
{
  TQDate d = date;

  if ( !d.isValid() ) {
    kdDebug(5850) << "DateNavigator::selectDates(TQDate): an invalid date was passed as a parameter!" << endl;
    d = TQDate::currentDate();
  }

  mSelectedDates.clear();
  mSelectedDates.append( d );

  emitSelected();
}

void DateNavigator::selectDates( int count )
{
  selectDates( mSelectedDates.first(), count );
}

void DateNavigator::selectDates( const TQDate &d, int count, const TQDate &preferredMonth )
{
  if ( count > MAX_SELECTABLE_DAYS ) {
    count = MAX_SELECTABLE_DAYS;
  }

  DateList dates;

  int i;
  for( i = 0; i < count; ++i ) {
    dates.append( d.addDays( i ) );
  }

  mSelectedDates = dates;

  emitSelected( preferredMonth );
}

void DateNavigator::selectWeekByDay( int weekDay, const TQDate &d, const TQDate &preferredMonth )
{
  int dateCount = mSelectedDates.count();
  bool weekStart = ( weekDay == TDEGlobal::locale()->weekStartDay() );
  if ( weekStart && dateCount == 7 ) {
    selectWeek( d, preferredMonth );
  } else {
    selectDates( d, dateCount, preferredMonth );
  }
}

void DateNavigator::selectWeek()
{
  selectWeek( mSelectedDates.first() );
}

void DateNavigator::selectWeek( const TQDate &d, const TQDate &preferredMonth )
{
  int dayOfWeek = KOGlobals::self()->calendarSystem()->dayOfWeek( d );

  int weekStart = TDEGlobal::locale()->weekStartDay();

  TQDate firstDate = d.addDays( weekStart - dayOfWeek );

  if ( weekStart != 1 && dayOfWeek < weekStart ) {
    firstDate = firstDate.addDays( -7 );
  }

  selectDates( firstDate, 7, preferredMonth );
}

void DateNavigator::selectWorkWeek()
{
  selectWorkWeek( mSelectedDates.first() );
}

void DateNavigator::selectWorkWeek( const TQDate &d )
{
  int weekStart = TDEGlobal::locale()->weekStartDay();

  int dayOfWeek = KOGlobals::self()->calendarSystem()->dayOfWeek( d );

  TQDate currentDate = d.addDays( weekStart - dayOfWeek );

  if ( weekStart != 1 && dayOfWeek < weekStart ) {
    currentDate = currentDate.addDays( -7 );
  }

  mSelectedDates.clear();
  int mask = KOGlobals::self()->getWorkWeekMask();

  for ( int i = 0; i < 7; ++i ) {
    if( (1<< ((i + weekStart + 6) % 7)) & (mask) ) {
	mSelectedDates.append( currentDate.addDays(i) );
    }
  }

  emitSelected();
}

void DateNavigator::selectToday()
{
  TQDate d = TQDate::currentDate();

  int dateCount = mSelectedDates.count();

  if ( dateCount == 7 ) {
    selectWeek( d );
  } else if ( dateCount == 5 ) {
    selectWorkWeek( d );
  } else {
    selectDates( d, dateCount );
  }
}

void DateNavigator::selectPreviousYear()
{
  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();
  firstSelected = KOGlobals::self()->calendarSystem()->addYears( firstSelected, -1 );

  selectWeekByDay( weekDay, firstSelected );
}

void DateNavigator::selectPreviousMonth( const TQDate &currentMonth,
                                         const TQDate &selectionLowerLimit,
                                         const TQDate &selectionUpperLimit )
{
  shiftMonth( currentMonth,
              selectionLowerLimit,
              selectionUpperLimit,
              -1 );
}

void DateNavigator::selectPreviousWeek()
{
  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();
  firstSelected = KOGlobals::self()->calendarSystem()->addDays( firstSelected, -7 );

  selectWeekByDay( weekDay, firstSelected );
}

void DateNavigator::selectNextWeek()
{
  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();

  firstSelected = KOGlobals::self()->calendarSystem()->addDays( firstSelected, 7 );

  selectWeekByDay( weekDay, firstSelected );
}

void DateNavigator::shiftMonth( const TQDate &currentMonth,
                                const TQDate &selectionLowerLimit,
                                const TQDate &selectionUpperLimit,
                                int offset )
{
  const KCalendarSystem *calSys = KOGlobals::self()->calendarSystem();

  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();
  firstSelected = calSys->addMonths( firstSelected, offset );

  /* Don't trust firstSelected to calculate the nextMonth. firstSelected
     can belong to a month other than currentMonth because KDateNavigator
     displays 7*6 days. firstSelected should only be used for selection
     purposes */
  const TQDate nextMonth = currentMonth.isValid() ?
                          calSys->addMonths( currentMonth, offset ) : firstSelected;

  /* When firstSelected doesn't belong to currentMonth it can happen
     that the new selection won't be visible on our KDateNavigators
     so we must adjust it */
  if ( selectionLowerLimit.isValid() &&
       firstSelected < selectionLowerLimit ) {
    firstSelected = selectionLowerLimit;
  } else if ( selectionUpperLimit.isValid() &&
              firstSelected > selectionUpperLimit ) {
    firstSelected = selectionUpperLimit.addDays( -6 );
  }

  selectWeekByDay( weekDay, firstSelected, nextMonth );
}

void DateNavigator::selectNextMonth( const TQDate &currentMonth,
                                     const TQDate &selectionLowerLimit,
                                     const TQDate &selectionUpperLimit )
{
  shiftMonth( currentMonth,
              selectionLowerLimit,
              selectionUpperLimit,
              1 );
}

void DateNavigator::selectNextYear()
{
  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();
  firstSelected = KOGlobals::self()->calendarSystem()->addYears( firstSelected, 1 );

  selectWeekByDay( weekDay, firstSelected );
}

void DateNavigator::selectPrevious()
{
  int offset = -7;
  if ( datesCount() == 1 ) {
    offset = -1;
  }

  selectDates( mSelectedDates.first().addDays( offset ), datesCount() );
}

void DateNavigator::selectNext()
{
  int offset = 7;
  if ( datesCount() == 1 ) {
    offset = 1;
  }

  selectDates( mSelectedDates.first().addDays( offset ), datesCount() );
}

void DateNavigator::selectMonth( int month )
{
  const KCalendarSystem *calSys = KOGlobals::self()->calendarSystem();

  TQDate firstSelected = mSelectedDates.first();
  int weekDay = firstSelected.dayOfWeek();

  int day = calSys->day( firstSelected );
  calSys->setYMD( firstSelected, calSys->year( firstSelected ), month, 1 );
  int days = calSys->daysInMonth( firstSelected );
  // As day we use either the selected date, or if the month has less days
  // than that, we use the max day of that month
  if ( day > days ) {
    day = days;
  }
  TQDate requestedMonth;
  calSys->setYMD( firstSelected, calSys->year( firstSelected ), month, day );
  calSys->setYMD( requestedMonth, calSys->year( firstSelected ), month, 1 );

  selectWeekByDay( weekDay, firstSelected, requestedMonth );
}

void DateNavigator::selectYear( int year )
{
  TQDate firstSelected = mSelectedDates.first();
  int deltaYear = year - KOGlobals::self()->calendarSystem()->year( firstSelected );
  firstSelected = KOGlobals::self()->calendarSystem()->addYears( firstSelected, deltaYear );

  int weekDay = firstSelected.dayOfWeek();
  selectWeekByDay( weekDay, firstSelected );
}

void DateNavigator::emitSelected( const TQDate &preferredMonth )
{
  emit datesSelected( mSelectedDates, preferredMonth );
}

#include "datenavigator.moc"
