/*
    This file is part of the kolab resource - the implementation of the
    Kolab storage format. See www.kolab.org for documentation on this.

    Copyright (c) 2004 Bo Thorsen <bo@sonofthor.dk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include "resourcekolabbase.h"
#include "kmailconnection.h"

#include <folderselectdialog.h>

#include <tdelocale.h>
#include <kstandarddirs.h>
#include <kinputdialog.h>
#include <kurl.h>
#include <tdetempfile.h>
#include <tdemessagebox.h>
#include <tqtextstream.h>
#include <kdebug.h>

using namespace Kolab;

static unsigned int uniquifier = 0;

ResourceKolabBase::ResourceKolabBase( const TQCString& objId )
  : mSilent( false )
{
  TDEGlobal::locale()->insertCatalogue( "kres_kolab" );
  TDEGlobal::locale()->insertCatalogue( "libkcal" );
  TQString uniqueObjId = TQString( objId ) + TQString::number( uniquifier++ );
  mConnection = new KMailConnection( this, uniqueObjId.utf8() );
}

ResourceKolabBase::~ResourceKolabBase()
{
  delete mConnection;
}


bool ResourceKolabBase::kmailSubresources( TQValueList<KMailICalIface::SubResource>& lst,
                                           const TQString& contentsType ) const
{
  return mConnection->kmailSubresources( lst, contentsType );
}

bool ResourceKolabBase::kmailTriggerSync( const TQString& contentsType ) const
{
  return mConnection->kmailTriggerSync( contentsType );
}


bool ResourceKolabBase::kmailIncidencesCount( int &count,
                                              const TQString& mimetype,
                                              const TQString& resource ) const
{
  return mConnection->kmailIncidencesCount( count, mimetype, resource );
}

bool ResourceKolabBase::kmailIncidences( TQMap<TQ_UINT32, TQString>& lst,
                                         const TQString& mimetype,
                                         const TQString& resource,
                                         int startIndex,
                                         int nbMessages ) const
{
  return mConnection->kmailIncidences( lst, mimetype, resource, startIndex, nbMessages );
}

bool ResourceKolabBase::kmailGetAttachment( KURL& url, const TQString& resource,
                                            TQ_UINT32 sernum,
                                            const TQString& filename ) const
{
  return mConnection->kmailGetAttachment( url, resource, sernum, filename );
}

bool ResourceKolabBase::kmailAttachmentMimetype( TQString & mimeType, TQString & resource,
                                                 TQ_UINT32 sernum, const TQString & filename ) const
{
  return mConnection->kmailAttachmentMimetype( mimeType, resource, sernum, filename );
}

bool ResourceKolabBase::kmailListAttachments( TQStringList &list,
                                              const TQString & resource,
                                              TQ_UINT32 sernum ) const
{
  return mConnection->kmailListAttachments( list, resource, sernum );
}

bool ResourceKolabBase::kmailDeleteIncidence( const TQString& resource,
                                              TQ_UINT32 sernum )
{
  return mSilent || mConnection->kmailDeleteIncidence( resource, sernum );
}

static TQString plainTextBody()
{
  const char * firstPartTextToTranslate = I18N_NOOP(
    "This is a Kolab Groupware object.\nTo view this object you"
    " will need an email client that can understand the Kolab"
    " Groupware format.\nFor a list of such email clients please"
    " visit\n%1" );
  const char * url = "http://www.kolab.org/kolab2-clients.html";
  TQString firstPartTextUntranslated = TQString::fromLatin1( firstPartTextToTranslate ).arg( url );
  TQString firstPartText = i18n( firstPartTextToTranslate ).arg( url );
  if ( firstPartText != firstPartTextUntranslated ) {
    firstPartText.append("\n\n-----------------------------------------------------\n\n");
    firstPartText.append( firstPartTextUntranslated );
  }
  return firstPartText;
}

bool ResourceKolabBase::kmailUpdate( const TQString& resource,
                                     TQ_UINT32& sernum,
                                     const TQString& xml,
                                     const TQString& mimetype,
                                     const TQString& subject,
                                     const CustomHeaderMap& _customHeaders,
                                     const TQStringList& _attachmentURLs,
                                     const TQStringList& _attachmentMimetypes,
                                     const TQStringList& _attachmentNames,
                                     const TQStringList& deletedAttachments )
{
  if ( mSilent )
    return true;

  TQString subj = subject;
  if ( subj.isEmpty() )
    subj = i18n("Internal kolab data: Do not delete this mail.");

  if ( mimetype.startsWith( "application/x-vnd.kolab" ) ) {

    // Save the xml file. Will be deleted at the end of this method
    KTempFile file;
    file.setAutoDelete( true );
    TQTextStream* stream = file.textStream();
    stream->setEncoding( TQTextStream::UnicodeUTF8 );
    *stream << xml;
    file.close();

    // Add the xml file as an attachment
    TQStringList attachmentURLs = _attachmentURLs;
    TQStringList attachmentMimeTypes = _attachmentMimetypes;
    TQStringList attachmentNames = _attachmentNames;
    KURL url;
    url.setPath( file.name() );
    url.setFileEncoding( "UTF-8" );
    attachmentURLs.prepend( url.url() );
    attachmentMimeTypes.prepend( mimetype );
    attachmentNames.prepend( "kolab.xml" );

    CustomHeaderMap customHeaders( _customHeaders );
    customHeaders.insert( "X-Kolab-Type", mimetype );

    return mConnection->kmailUpdate( resource, sernum, subj, plainTextBody(), customHeaders,
        attachmentURLs, attachmentMimeTypes, attachmentNames,
        deletedAttachments );
  } else {
    // ical style, simply put the data inline
    return mConnection->kmailUpdate( resource, sernum, subj, xml, _customHeaders,
        _attachmentURLs, _attachmentMimetypes, _attachmentNames, deletedAttachments );
  }
}

TQString ResourceKolabBase::configFile( const TQString& type ) const
{
  return locateLocal( "config",
                      TQString( "tderesources/kolab/%1rc" ).arg( type ) );
}

bool ResourceKolabBase::connectToKMail() const
{
  return mConnection->connectToKMail();
}

bool ResourceKolabBase::kmailAddSubresource( const TQString& resource,
                                             const TQString& parent,
                                             const TQString& contentsType )
{
  return mConnection->kmailAddSubresource( resource, parent, contentsType );
}

bool ResourceKolabBase::kmailRemoveSubresource( const TQString& resource )
{
  return mConnection->kmailRemoveSubresource( resource );
}

TQString ResourceKolabBase::findWritableResource( const ResourceType &type,
                                                 const ResourceMap& resources,
                                                 const TQString& text )
{
  mErrorCode = NoError;

  // I have to use the label (shown in the dialog) as key here. But given how the
  // label is made up, it should be unique. If it's not, well the dialog would suck anyway...
  TQMap<TQString, TQString> possible;
  TQStringList labels;
  ResourceMap::ConstIterator it;
  for ( it = resources.begin(); it != resources.end(); ++it ) {
    if ( it.data().writable() && it.data().active() ) {
      // Writable and active possibility
      possible[ it.data().label() ] = it.key();
    }
  }

  if ( possible.isEmpty() ) { // None found!!
    kdWarning(5650) << "No writable resource found!" << endl;

    TQString errorText;
    switch( type ) {
      case Events:
        errorText = i18n( "You have no writable event folders so saving will not be possible.\n"
                          "Please create or activate at least one writable event folder and try again." );
        break;
      case Tasks:
        errorText = i18n( "You have no writable task folders so saving will not be possible.\n"
                          "Please create or activate at least one writable task folder and try again." );
        break;
      case Incidences:
        errorText = i18n( "You have no writable calendar folder so saving will not be possible.\n"
                          "Please create or activate at least one writable calendar folder and try again." );
        break;
      case Notes:
        errorText = i18n( "You have no writable notes folders so saving will not be possible.\n"
                          "Please create or activate at least one writable notes folder and try again." );
        break;
      case Contacts:
        errorText = i18n( "You have no writable addressbook folder so saving will not be possible.\n"
                          "Please create or activate at least one writable addressbook folder and try again." );
        break;
    }

    KMessageBox::error( 0, errorText );
    mErrorCode = NoWritableFound;
    return TQString();
  }
  if ( possible.count() == 1 )
    // Just one found
    return possible.begin().data(); // yes this is the subresource key, i.e. location

  TQString t = text;
  if ( t.isEmpty() )
    i18n( "You have more than one writable resource folder. "
          "Please select the one you want to write to." );

  // Several found, ask the user
  TQString chosenLabel = KPIM::FolderSelectDialog::getItem( i18n( "Select Resource Folder" ),
                                                           t, possible.keys() );
  if ( chosenLabel.isEmpty() ) {
    // cancelled
    mErrorCode = UserCancel;
    return TQString();
  }
  return possible[chosenLabel];
}

KMailICalIface::StorageFormat ResourceKolabBase::kmailStorageFormat( const TQString &folder ) const
{
  KMailICalIface::StorageFormat format = (KMailICalIface::StorageFormat) 3;
  mConnection->kmailStorageFormat( format, folder );
  return format;
}
