/*
    This file is part of tdepim.

    Copyright (c) 2004 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "sloxwizard.h"
#include "sloxconfig.h"

#include "tderesources/slox/tdeabcsloxprefs.h"
#include "tderesources/slox/tdeabcresourceslox.h"
#include "tderesources/slox/kcalsloxprefs.h"
#include "tderesources/slox/kcalresourceslox.h"

#include <libkcal/resourcecalendar.h>

#include <klineedit.h>
#include <tdelocale.h>

#include <tqlayout.h>
#include <tqcheckbox.h>
#include <tqlabel.h>


TQString sloxUrl()
{
  TQString url;

  if ( SloxConfig::self()->useHttps() ) url = "https://";
  else url = "http://";

  url += SloxConfig::self()->server();

  return url;
}

class CreateSloxKcalResource : public TDEConfigPropagator::Change
{
  public:
    CreateSloxKcalResource()
      : TDEConfigPropagator::Change( i18n("Create SLOX Calendar Resource") )
    {
    }

    void apply()
    {
      KCal::CalendarResourceManager m( "calendar" );
      m.readConfig();

      KURL url( sloxUrl() );

      KCalResourceSlox *r = new KCalResourceSlox( url );
      r->setResourceName( i18n("Openexchange Server") );
      r->prefs()->setUser( SloxConfig::self()->user() );
      r->prefs()->setPassword( SloxConfig::self()->password() );
      r->setSavePolicy( KCal::ResourceCached::SaveDelayed );
      r->setReloadPolicy( KCal::ResourceCached::ReloadInterval );
      r->setReloadInterval( 20 );
      m.add( r );
      m.writeConfig();

      SloxConfig::self()->setKcalResource( r->identifier() );
    }
};

class UpdateSloxKcalResource : public TDEConfigPropagator::Change
{
  public:
    UpdateSloxKcalResource()
      : TDEConfigPropagator::Change( i18n("Update SLOX Calendar Resource") )
    {
    }

    void apply()
    {
      KCal::CalendarResourceManager m( "calendar" );
      m.readConfig();

      KURL url( sloxUrl() );

      KCal::CalendarResourceManager::Iterator it;
      for ( it = m.begin(); it != m.end(); ++it ) {
        if ( (*it)->identifier() == SloxConfig::kcalResource() ) {
          KCalResourceSlox *r = static_cast<KCalResourceSlox *>( *it );
          r->prefs()->setUrl( url.url() );
          r->prefs()->setUser( SloxConfig::self()->user() );
          r->prefs()->setPassword( SloxConfig::self()->password() );
          r->setSavePolicy( KCal::ResourceCached::SaveDelayed );
          r->setReloadPolicy( KCal::ResourceCached::ReloadInterval );
          r->setReloadInterval( 20 );
        }
      }
      m.writeConfig();
    }
};

class CreateSloxTdeabcResource : public TDEConfigPropagator::Change
{
  public:
    CreateSloxTdeabcResource()
      : TDEConfigPropagator::Change( i18n("Create SLOX Addressbook Resource") )
    {
    }

    void apply()
    {
      KRES::Manager<TDEABC::Resource> m( "contact" );
      m.readConfig();

      KURL url( sloxUrl() );
      TQString user( SloxConfig::self()->user() );
      TQString password( SloxConfig::self()->password() );

      TDEABC::ResourceSlox *r = new TDEABC::ResourceSlox( url, user, password );
      r->setResourceName( i18n("Openexchange Server") );
      m.add( r );
      m.writeConfig();

      SloxConfig::self()->setTdeabcResource( r->identifier() );
    }
};

class UpdateSloxTdeabcResource : public TDEConfigPropagator::Change
{
  public:
    UpdateSloxTdeabcResource()
      : TDEConfigPropagator::Change( i18n("Update SLOX Addressbook Resource") )
    {
    }

    void apply()
    {
      KRES::Manager<TDEABC::Resource> m( "contact" );
      m.readConfig();

      KURL url( sloxUrl() );

      KRES::Manager<TDEABC::Resource>::Iterator it;
      for ( it = m.begin(); it != m.end(); ++it ) {
        if ( (*it)->identifier() == SloxConfig::tdeabcResource() ) {
          TDEABC::ResourceSlox *r = static_cast<TDEABC::ResourceSlox *>( *it );
          r->prefs()->setUrl( url.url() );
          r->prefs()->setUser( SloxConfig::self()->user() );
          r->prefs()->setPassword( SloxConfig::self()->password() );
        }
      }
      m.writeConfig();
    }
};


class SloxPropagator : public TDEConfigPropagator
{
  public:
    SloxPropagator()
      : TDEConfigPropagator( SloxConfig::self(), "slox.kcfg" )
    {
    }

    ~SloxPropagator()
    {
      SloxConfig::self()->writeConfig();
    }

  protected:
    void addCustomChanges( Change::List &changes )
    {
      KCal::CalendarResourceManager m1( "calendar" );
      m1.readConfig();
      KCal::CalendarResourceManager::Iterator it;
      for ( it = m1.begin(); it != m1.end(); ++it ) {
        if ( (*it)->type() == "slox" ) break;
      }
      if ( it == m1.end() ) {
        changes.append( new CreateSloxKcalResource );
      } else {
        if ( (*it)->identifier() == SloxConfig::kcalResource() ) {
          KCal::SloxPrefs *prefs = static_cast<KCalResourceSlox *>( *it )->prefs();
          if ( prefs->url() != sloxUrl() ||
               prefs->user() != SloxConfig::user() ||
               prefs->password() != SloxConfig::password() ) {
            changes.append( new UpdateSloxKcalResource );
          }
        }
      }

      KRES::Manager<TDEABC::Resource> m2( "contact" );
      m2.readConfig();
      KRES::Manager<TDEABC::Resource>::Iterator it2;
      for ( it2 = m2.begin(); it2 != m2.end(); ++it2 ) {
        if ( (*it2)->type() == "slox" ) break;
      }
      if ( it2 == m2.end() ) {
        changes.append( new CreateSloxTdeabcResource );
      } else {
        if ( (*it2)->identifier() == SloxConfig::tdeabcResource() ) {
          TDEABC::SloxPrefs *prefs = static_cast<TDEABC::ResourceSlox *>( *it2 )->prefs();
          if ( prefs->url() != sloxUrl() ||
               prefs->user() != SloxConfig::user() ||
               prefs->password() != SloxConfig::password() ) {
            changes.append( new UpdateSloxTdeabcResource );
          }
        }
      }
    }
};

SloxWizard::SloxWizard() : TDEConfigWizard( new SloxPropagator )
{
  TQFrame *page = createWizardPage( i18n("SUSE LINUX OpenExchange Server") );

  TQGridLayout *topLayout = new TQGridLayout( page );
  topLayout->setSpacing( spacingHint() );

  TQLabel *label = new TQLabel( i18n("Server name:"), page );
  topLayout->addWidget( label, 0, 0 );
  mServerEdit = new KLineEdit( page );
  topLayout->addWidget( mServerEdit, 0, 1 );

  label = new TQLabel( i18n("User name:"), page );
  topLayout->addWidget( label, 1, 0 );
  mUserEdit = new KLineEdit( page );
  topLayout->addWidget( mUserEdit, 1, 1 );

  label = new TQLabel( i18n("Password:"), page );
  topLayout->addWidget( label, 2, 0 );
  mPasswordEdit = new KLineEdit( page );
  mPasswordEdit->setEchoMode( KLineEdit::Password );
  topLayout->addWidget( mPasswordEdit, 2, 1 );

  mSavePasswordCheck = new TQCheckBox( i18n("Save password"), page );
  topLayout->addMultiCellWidget( mSavePasswordCheck, 3, 3, 0, 1 );

  mSecureCheck = new TQCheckBox( i18n("Encrypt communication with server"),
                                page );
  topLayout->addMultiCellWidget( mSecureCheck, 4, 4, 0, 1 );

  topLayout->setRowStretch( 5, 1 );

  setupRulesPage();
  setupChangesPage();

  resize( 400, 300 );
}

SloxWizard::~SloxWizard()
{
}

TQString SloxWizard::validate()
{
  KURL server( mServerEdit->text() );
  if ( !server.protocol().isEmpty() ||
      mServerEdit->text().isEmpty() ||
      mUserEdit->text().isEmpty() ||
      mPasswordEdit->text().isEmpty() )
    return i18n( "Please fill in all fields." );
  return TQString();
}

void SloxWizard::usrReadConfig()
{
  mServerEdit->setText( SloxConfig::self()->server() );
  mUserEdit->setText( SloxConfig::self()->user() );
  mPasswordEdit->setText( SloxConfig::self()->password() );
  mSavePasswordCheck->setChecked( SloxConfig::self()->savePassword() );
  mSecureCheck->setChecked( SloxConfig::self()->useHttps() );
}

void SloxWizard::usrWriteConfig()
{
  SloxConfig::self()->setServer( mServerEdit->text() );
  SloxConfig::self()->setUser( mUserEdit->text() );
  SloxConfig::self()->setPassword( mPasswordEdit->text() );
  SloxConfig::self()->setSavePassword( mSavePasswordCheck->isChecked() );
  SloxConfig::self()->setUseHttps( mSecureCheck->isChecked() );
}
