/***************************************************************************
                          webpricequote.h
                             -------------------
    begin                : Thu Dec 30 2004
    copyright            : (C) 2004 by Ace Jones
    email                : Ace Jones <acejones@users.sourceforge.net>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef WEBPRICEQUOTE_H
#define WEBPRICEQUOTE_H

// ----------------------------------------------------------------------------
// TQt Headers

#include <tqobject.h>
#include <tqdatetime.h>
#include <tqstringlist.h>
#include <tqmap.h>

// ----------------------------------------------------------------------------
// TDE Headers

#include <tdeprocess.h>
namespace TDEIO {
  class Job;
};

// ----------------------------------------------------------------------------
// Project Headers

#include "../mymoney/mymoneymoney.h"

/**
Helper class to attend the process which is running the script, in the case
of a local script being used to fetch the quote.

@author Thomas Baumgart <thb@net-bembel.de> & Ace Jones <acejones@users.sourceforge.net>
*/
class WebPriceQuoteProcess: public TDEProcess
{
  TQ_OBJECT
  
public:
  WebPriceQuoteProcess(void);
  void setSymbol(const TQString& _symbol) { m_symbol = _symbol; m_string.truncate(0); }

public slots:
  void slotReceivedDataFromFilter(TDEProcess*, char*, int);
  void slotProcessExited(TDEProcess*);

signals:
  void processExited(const TQString&);

private:
  TQString m_symbol;
  TQString m_string;
};

/**
Helper class to run the Finance::Quote process. This is used only for the purpose of obtaining
a list of valid sources. The actual price quotes are obtained thru WebPriceQuoteProcess.
The class also contains functions to convert between the rather cryptic source names used
by the Finance::Quote package, and more user-friendly names.

@author Thomas Baumgart <thb@net-bembel.de> & Ace Jones <acejones@users.sourceforge.net>, Tony B<tonybloom@users.sourceforge.net>
 */
class FinanceQuoteProcess: public TDEProcess
{
  TQ_OBJECT
  
  public:
    FinanceQuoteProcess(void);
    void launch (const TQString& scriptPath);
    bool isFinished() { return(m_isDone);};
    TQStringList getSourceList();
    const TQString crypticName(const TQString& niceName);
    const TQString niceName(const TQString& crypticName);

  public slots:
    void slotReceivedDataFromFilter(TDEProcess*, char*, int);
    void slotProcessExited(TDEProcess*);

  private:
    bool m_isDone;
    TQString m_string;
    typedef TQMap<TQString, TQString> fqNameMap;
    fqNameMap m_fqNames;
};

/**
  * @author Thomas Baumgart & Ace Jones
  *
  * This is a helper class to store information about an online source
  * for stock prices or currency exchange rates.
  */
struct WebPriceQuoteSource
{
  WebPriceQuoteSource() {}
  WebPriceQuoteSource(const TQString& name);
  WebPriceQuoteSource(const TQString& name, const TQString& url, const TQString& sym, const TQString& price, const TQString& date, const TQString& dateformat);
  ~WebPriceQuoteSource() {}

  void write(void) const;
  void rename(const TQString& name);
  void remove(void) const;

  TQString    m_name;
  TQString    m_url;
  TQString    m_sym;
  TQString    m_price;
  TQString    m_date;
  TQString    m_dateformat;
  bool       m_skipStripping;
};

/**
Retrieves a price quote from a web-based quote source

@author Ace Jones <acejones@users.sourceforge.net>
*/
class WebPriceQuote: public TQObject
{
  TQ_OBJECT
  
public:
  WebPriceQuote( TQObject* = 0, const char* = 0 );
  ~WebPriceQuote();

  typedef enum _quoteSystemE {
    Native=0,
    FinanceQuote
  } quoteSystemE;

  /**
    * This launches a web-based quote update for the given @p _symbol.
    * When the quote is received back from the web source, it will be
    * emitted on the 'quote' signal.
    *
    * @param _symbol the trading symbol of the stock to fetch a price for
    * @param _id an arbitrary identifier, which will be emitted in the quote
    *                signal when a price is sent back.
    * @param _source the source of the quote (must be a valid value returned
    *                by quoteSources().  Send TQString() to use the default
    *                source.
    * @return bool Whether the quote fetch process was launched successfully
    */

  bool launch(const TQString& _symbol, const TQString& _id, const TQString& _source=TQString());

  /**
    * This returns a list of the names of the quote sources
    * currently defined.
    *
   * @param _system whether to return Native or Finance::Quote source list
   * @return TQStringList of quote source names
    */
  static TQStringList quoteSources(const _quoteSystemE _system=Native);

signals:
  void quote(const TQString&, const TQString&, const TQDate&, const double&);
  void failed(const TQString&, const TQString&);
  void status(const TQString&);
  void error(const TQString&);

protected slots:
  void slotParseQuote(const TQString&);

protected:
  static TQMap<TQString,WebPriceQuoteSource> defaultQuoteSources(void);

private:
  bool download(const KURL& u, TQString & target, TQWidget* window);
  void removeTempFile(const TQString& tmpFile);

private slots:
  void slotResult( TDEIO::Job * job );


private:
  bool launchNative(const TQString& _symbol, const TQString& _id, const TQString& _source=TQString());
  bool launchFinanceQuote(const TQString& _symbol, const TQString& _id, const TQString& _source=TQString());
  void enter_loop(void);

  static TQStringList quoteSourcesNative();
  static TQStringList quoteSourcesFinanceQuote();

  WebPriceQuoteProcess m_filter;
  TQString m_symbol;
  TQString m_id;
  TQDate m_date;
  double m_price;
  WebPriceQuoteSource m_source;
  static TQString m_financeQuoteScriptPath;
  static TQStringList m_financeQuoteSources;


  /**
   * Whether the download succeeded or not. Taken from TDEIO::NetAccess
   */
  bool bJobOK;
  static TQString* lastErrorMsg;
  static int lastErrorCode;
  TQString m_tmpFile;
};

class MyMoneyDateFormat
{
public:
  MyMoneyDateFormat(const TQString& _format): m_format(_format) {}
  TQString convertDate(const TQDate& _in) const;
  TQDate convertString(const TQString& _in, bool _strict=true, unsigned _centurymidpoint = TQDate::currentDate().year() ) const;
  const TQString& format(void) const { return m_format; }
private:
  TQString m_format;
};

namespace convertertest {

/**
Simple class to handle signals/slots for unit tests

@author Ace Jones <acejones@users.sourceforge.net>
*/
class QuoteReceiver : public TQObject
{
TQ_OBJECT
  
public:
    QuoteReceiver(WebPriceQuote* q, TQObject *parent = 0, const char *name = 0);
    ~QuoteReceiver();
public slots:
  void slotGetQuote(const TQString&,const TQDate&, const double&);
  void slotStatus(const TQString&);
  void slotError(const TQString&);
public:
  TQStringList m_statuses;
  TQStringList m_errors;
  MyMoneyMoney m_price;
  TQDate m_date;
};

} // end namespace convertertest


#endif // WEBPRICEQUOTE_H
