/***************************************************************************
    smb4knetworkbrowser_part  -  This Part encapsulates the network
    browser of Smb4K.
                             -------------------
    begin                : Fr Jan 5 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KNETWORKBROWSERPART_H
#define SMB4KNETWORKBROWSERPART_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// KDE includes
#include <tdeactionclasses.h>
#include <tdeparts/part.h>
#include <tdeparts/factory.h>

// TQt includes
#include <tqptrlist.h>
#include <tqlistview.h>

// forward declarations
class Smb4KNetworkBrowser;
class Smb4KNetworkBrowserItem;
class Smb4KWorkgroupItem;
class Smb4KHostItem;
class Smb4KShareItem;

/**
 * This is one of the parts of Smb4K. It contains the network browser.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class Smb4KNetworkBrowserPart : public KParts::Part
{
  TQ_OBJECT
  

  public:
    /**
     * Mode enumeration
     */
    enum Mode { Normal, KonqPlugin };

    /**
     * The constructor.
     *
     * @param parentWidget        The parent widget
     *
     * @param widgetName          The name the widget should have
     *
     * @param parent              The parent object
     *
     * @param name                The name this object should have
     *
     * @param mode                Determines in which mode the KPart should be started
     */
    Smb4KNetworkBrowserPart( TQWidget *parentWidget = 0,
                             const char *widgetName = 0,
                             TQObject *parent = 0,
                             const char *name = 0,
                             Mode mode = Normal );

    /**
     * The destructor.
     */
    virtual ~Smb4KNetworkBrowserPart();

  protected:
    /**
     * Reimplemented from KParts::Part.
     */
    void customEvent( TQCustomEvent *e );

  protected slots:
    /**
     * This slot is called if the user requests the context menu. It shows
     * the menu with the actions defined for the widget.
     *
     * @param item                The item for which the popup menu has been
     *                            requested or NULL.
     *
     * @param pos                 The position where user clicked.
     *
     * @param col                 The column where the user clicked.
     */
    void slotContextMenuRequested( TQListViewItem *item, const TQPoint &pos, int col );

    /**
     * Is called when the selection changed. This slot takes care of the
     * actions being enabled or disabled accordingly. All widget specific
     * stuff has to be done in the browser widget itself.
     *
     * @param item                The selection list view item.
     */
    void slotSelectionChanged( TQListViewItem *item );

    /**
     * Is called when the user pressed a mouse button somewhere in the widget.
     * In addition to Smb4KNetworkBrowserPart::slotSelectionChanged() this slot
     * takes care of the actions being enabled or disabled accordingly. All
     * widget specific stuff has to be done in the browser widget itself.
     *
     * @param item                The selection list view item.
     */
    void slotPressed( TQListViewItem *item );

    /**
     * This slot is called when an item of the browser's list view has been opened.
     * It is used to invoke the functions of the scanner that need to be run.
     *
     * @param item                The list view item that has been opened.
     */
    void slotItemExpanded( TQListViewItem *item );

    /**
     * This slot is called when an item of the browser's list view has been closed.
     *
     * @param item                The list view item that has been closed.
     */
    void slotItemCollapsed( TQListViewItem *item );

    /**
     * This slot is invoked when the user executed an item. It is used to mount
     * shares.
     *
     * @param item                The list view item that has been executed.
     */
    void slotItemExecuted( TQListViewItem *item );

    /**
     * This slot is called when the user moved the mouse over an @p item in the
     * network browser's list view and a tool tip is about to be shown. It will
     * initiate any actions that have to be taken by the scanner to complete
     * the information on the network items shown in the browser.
     *
     * @param item                The item for which the additional info should
     *                            be retrieved
     */
    void slotAboutToShowToolTip( Smb4KNetworkBrowserItem *item );

    /**
     * This slot receives the workgroups/domains found by the scanner. It takes
     * a list of workgroup items @p list and inserts the respective workgroups
     * into the browser window. Obsolete items will be deleted from the network
     * browser.
     *
     * @param list                A list of Smb4KWorkgroupItem objects
     */
    void slotWorkgroups( const TQValueList<Smb4KWorkgroupItem *> &list );

    /**
     * This slot receives the list of workgroup/domain members that were found
     * by the scanner. It takes this @p list and inserts it into the list view.
     * The parent of the items is the workgroup/domain item with the name
     * @p workgroup. Obsolete items will be deleted from the network browser.
     *
     * @param workgroup           The workgroup where the hosts belong
     *
     * @param list                A list of Smb4KHostItem objects
     */
    void slotWorkgroupMembers( const TQString &workgroup, const TQValueList<Smb4KHostItem *> &list );

    /**
     * This slot receives the list of shared resources a host provides. It takes
     * this @p list and inserts its items as children of @p host into the list
     * view. Obsolete items will be deleted from the network browser.
     *
     * @param host                The host where the shares belong
     *
     * @param list                The list of shares that belong to @p host
     */
    void slotShares( const TQString &host, const TQValueList<Smb4KShareItem *> &list );

    /**
     * This slot takes a Smb4KHostItem object @p item, reads the IP address entry
     * from it and updates the list view item representing the host with it.
     *
     * @param item                A Smb4KHostItem with an updated IP address.
     */
    void slotAddIPAddress( Smb4KHostItem *item );

    /**
     * This slot adds additional information to a browser item. It takes an
     * Smb4KHostItem @p item, searches the assossiated browser item and updates its
     * contents.
     *
     * @param item                A Smb4KHostItem with updated contents.
     */
    void slotAddInformation( Smb4KHostItem *item );

    /**
     * This slots is connected to the Smb4KScanner::hostAdded() signal and inserts
     * a single host in the list view. If the host is already present, nothing is
     * done.
     *
     * @param item                A Smb4KHostItem that is to be added to the
     *                            list view.
     */
    void slotInsertHost( Smb4KHostItem *item );

    /**
     * Rescan the network. This slot is connected to the 'Rescan' action.
     */
    void slotRescan();

    /**
     * Abort a network scan. This slot is connected to the 'Abort' action.
     */
    void slotAbort();

    /**
     * Manually mount a share. This slot is connected to the 'Mount Manually'
     * action and opens a mount dialog.
     */
    void slotMountManually();

    /**
     * Provide authentication for the current network object. This slot is
     * connected to the 'Authentication' action.
     */
    void slotAuthentication();

    /**
     * Provide custom options for a server or share. This slot is connected
     * to the 'Custom Options' action.
     */
    void slotCustomOptions();

    /**
     * Bookmark a remote share. This slot is connected to the 'Add Bookmark'
     * action.
     */
    void slotBookmark();

    /**
     * Preview a share. This slot is connected to the 'Preview' action.
     */
    void slotPreview();

    /**
     * Print a document on a remote printer. This slot is connected to the
     * 'Print File' action.
     */
    void slotPrint();

    /**
     * Mount a remote share. This slot is connected to the 'Mount' action.
     */
    void slotMount();

    /**
     * Unmount a share. This slot is only relevant in the Konqueror plugin
     * mode.
     */
    void slotUnmount();

    /**
     * This slot is invoked by the Smb4KMounter::updated() signal and changes
     * the text and icon of all shares that are already mounted.
     */
    void slotMarkMountedShares();

    /**
     * This slot is connected to the Smb4KCore::runStateChanged() signal and
     * is used to enable/disable actions.
     */
    void slotRunStateChanged();

  private:
    /**
     * Set up the actions
     */
    void setupActions();

    /**
     * Load settings for the widget or the actions.
     */
    void loadSettings();

    /**
     * The action menu
     */
    TDEActionMenu *m_menu;

    /**
     * The network browser widget
     */
    Smb4KNetworkBrowser *m_widget;

    /**
     * Determines with which mode the KPart will be started
     */
    Mode m_mode;
};


class TDEInstance;
class TDEAboutData;

class Smb4KNetworkBrowserPartFactory : public KParts::Factory
{
  TQ_OBJECT
  

  public:
    /**
     * The constructor
     */
    Smb4KNetworkBrowserPartFactory();

    /**
     * The destructor
     */
    virtual ~Smb4KNetworkBrowserPartFactory();

    /**
     * Reimplemented from KParts::Factory. This function is used to
     * create a part object.
     *
     * @param parentWidget            The parent of the part's widget.
     *
     * @param widgetName              The name of the part's widget.
     *
     * @param parent                  The parent of the part.
     *
     * @param name                    The name of the part.
     *
     * @param classname               The class name of the part. This should be "KParts::Part".
     *
     * @param args                    A list of additional arguments. They have to be provided as
     *                                name="value" pairs. At the moment this function understands
     *                                the following pairs:
     *                                konqplugin="true"|"false".
     */
    virtual KParts::Part *createPartObject( TQWidget *parentWidget, const char *widgetName,
                                            TQObject *parent, const char *name,
                                            const char *classname, const TQStringList &args = TQStringList() );

    /**
     * The instance
     */
    static TDEInstance *instance();

  private:
    /**
     * The factory's instance.
     */
    static TDEInstance *m_instance;

    /**
     * The factory's TDEAboutData object
     */
    static TDEAboutData *m_about;
};

#endif
