/*
    This file is part of libtdepim.
    Copyright (c) 2004-2005 David Faure <faure@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef DISTRIBUTIONLIST_H
#define DISTRIBUTIONLIST_H

#include <tdeabc/addressee.h>

namespace TDEABC {
class AddressBook;
}

namespace KPIM {

/**
 * @short Distribution list of email addresses
 *
 * This class represents a list of email addresses. Each email address is
 * associated with an address book entry. If the address book entry changes, the
 * entry in the distribution list is automatically updated.
 *
 * This should go into tdelibs in KDE4.
 *
 * @author David Faure <faure@kde.org>
 */
class DistributionList : public TDEABC::Addressee
{
  public:
    /**
     * @short Distribution List Entry
     *
     * This class represents an entry of a distribution list. It consists of an
     * addressee and an email address. If the email address is null, the
     * preferred email address of the addressee is used.
     */
    struct Entry
    {
      typedef TQValueList<Entry> List;

      Entry() {}
      Entry( const Addressee &_addressee, const TQString &_email ) :
          addressee( _addressee ), email( _email ) {}

      Addressee addressee;
      TQString email;
    };

    typedef TQValueList<DistributionList> List;

    /**
     * Create a distribution list.
     */
    DistributionList();
    /**
     * Create a distribution list from an addressee object
     * (this is a kind of down-cast)
     */
    DistributionList( const TDEABC::Addressee& addr );

    /**
     * Destructor.
     */
    ~DistributionList() {}

    /// HACK: reimplemented from Addressee, but it's NOT virtual there
    void setName( const TQString &name );

    /// HACK: reimplemented from Addressee, but it's NOT virtual there
    TQString name() const { return formattedName(); }

    /**
      Insert an entry into this distribution list. If the entry already exists
      nothing happens.
    */
    void insertEntry( const Addressee &, const TQString &email=TQString() );

    /**
      Remove an entry from this distribution list. If the entry doesn't exist
      nothing happens.
    */
    void removeEntry( const Addressee &, const TQString &email=TQString() );

    /// Overload, used by resources to avoid looking up the addressee
    void insertEntry( const TQString& uid, const TQString& email=TQString() );
    /// Overload, used by resources to avoid looking up the addressee
    void removeEntry( const TQString& uid, const TQString& email=TQString() );


    /**
      Return list of email addresses, which belong to this distributon list.
      These addresses can be directly used by e.g. a mail client.
      @param book necessary to look up entries
    */
    TQStringList emails( TDEABC::AddressBook* book ) const;

    /**
      Return list of entries belonging to this distribution list. This function
      is mainly useful for a distribution list editor.
      @param book necessary to look up entries
    */
    Entry::List entries( TDEABC::AddressBook* book ) const;

    // KDE4: should be a method of Addressee
    static bool isDistributionList( const TDEABC::Addressee& addr );

    // KDE4: should be a method of AddressBook
    static DistributionList findByName( TDEABC::AddressBook* book,
                                        const TQString& name,
                                        bool caseSensitive = true );
    // KDE4: should be a method of AddressBook
    // A bit slow (but no more than findByName).
    // From KAddressbook, use Core::distributionLists() instead.
    static TQValueList<DistributionList> allDistributionLists( TDEABC::AddressBook* book );


  private:
    // can't have any data here, use Addressee's methods instead
};

}

#endif /* DISTRIBUTIONLIST_H */

